function dmri_parcel_to_label_file(parcels_file, fs_info_file, output_dir)
% Make .label files for using as ROIs of anatomical connectivity matrix
%
% [Usage]
%    dmri_parcel_to_label_file(parcels_file, fs_info_file);
%
% [Input]
%    parcels_file : parcels information file.
%    fs_info_file : freesurfer infomation file.
%      output_dir : label file output directory.
%
% [Output]
%    none
%
% [Note]
%    Output label files are written for FSL.
%    They contain Freesurfer cortical indicies.
%
%    * left and right parcels
%      parcel1.label
%      parcel2.label
%      ...
%      parcelN.label
%
%    * Subcortical label
%      lh/rh.subcortex.label
%
%    originally created by M.Fukushima
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 3
    error('Please check input arguments.');
end
if exist(parcels_file, 'file') ~= 2
    error(sprintf('Specified parcels file not found : %s', parcels_file));
end
if exist(fs_info_file, 'file') ~= 2
    error(sprintf('Specified freesurfer information file not found: %s', fs_info_file));
end
if exist(fs_info_file, 'dir') ~= 7
    mkdir(output_dir);
end

%
% --- Main Procedure
%
format long

fs = load(fs_info_file);

% File names
lhascfile = fs.FS_wm.lh_smoothwm_asc_file;
rhascfile = fs.FS_wm.rh_smoothwm_asc_file;

% Read .asc file
[l_S, l_v, l_f] = read_asc(lhascfile);
[r_S, r_v, r_f] = read_asc(rhascfile);

load(parcels_file);
size_left = length(lh_cortex_id);
Nlabel = length(lh_cortex_id) + length(rh_cortex_id);

cortex_id = [lh_cortex_id; rh_cortex_id];

% Create label files
fprintf('Now putting label files into %s\n', output_dir);
for nv = 1:Nlabel
  original_ind = cortex_id{nv};

  if nv <= size_left
    lxyz  = l_v(original_ind, 1:3);
    lvals = l_v(original_ind, 4);
  else
    lxyz  = r_v(original_ind, 1:3);
    lvals = r_v(original_ind, 4);
  end

  % Output file name
  labelfile = fullfile(output_dir, filesep, ['parcel' num2str(nv) '.label']);
  lindex = original_ind - 1; % 0-based, M.Fukushima 2012/06/11
  
  % Write .label file 
  ok = write_label(lindex, lxyz, lvals, labelfile, parcels_file);
  if ok ~=1
    error('Writing .label file is failed!!');
  end
end

fprintf('done.\n');


%
% --- Create subcortex label
%
original_ind = fs.FS_wm.lh_subcortex_index;
lxyz  = l_v(original_ind, 1:3);
lvals = l_v(original_ind, 4);
labelfile = fullfile(output_dir, filesep, 'lh.subcortex.label');

lindex = original_ind - 1; % 0-based, M.Fukushima 2012/06/11
ok = write_label(lindex, lxyz, lvals, labelfile, parcels_file);

original_ind = fs.FS_wm.rh_subcortex_index;
lxyz  = r_v(original_ind, 1:3);
lvals = r_v(original_ind, 4);
labelfile = fullfile(output_dir, filesep, 'rh.subcortex.label');

lindex = original_ind - 1; % 0-based, M.Fukushima 2012/06/11
ok = write_label(lindex, lxyz, lvals, labelfile, parcels_file);
