function dmri_fodf_calc(data_niftigz_file, bval_file, bvec_file, ...
                        area_niftigz_file, ...
                        mask_niftigz_file, ...
                        output_mif_file)
% Calculate fiber orientation density function.
%
% [Usage]
%    dmri_fodf_calc(data_niftigz_file, bval_file, bvec_file,...
%                   area_niftigz_file, ...
%                   mask_niftigz_file, ...
%                   output_mif_file);
%
% [Input]
%    data_niftigz_file : gzipped Nifti file(.nii.gz)
%                        4D series of data volumes.
%            bval_file : A text file containing a list of b values applied 
%                      during each volume acquisition. The order of entries 
%                      in this file must match the order of volumes in 
%                      the input data and entries in the gradient 
%                       directions text file. 
%           bvec_file  : A text file containing a list of gradient directions 
%                      applied during diffusion weighted volumes. The order 
%                      of entries in this file must match the order of volumes 
%                      in the input data series. 
%    area_niftigz_file : brain area file(.nii.gz) for response function
%                        estimation made by dti_response_estimation_prepare.m
%    mask_niftigz_file : struct brain mask image file(.nii.gz) on FA coordinate system.
%      output_mif_file : Fiber orientation density function file(.mif)
%
% [Output]
%    none
%
% [Output files]
%     output_mif_file
%     response.txt
%     gradients-x.txt 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 6
    error('Please check input arguments.');
end
if exist(data_niftigz_file, 'file') ~= 2
    error('Specified diffusion weighted data file not found.');
end
if exist(area_niftigz_file, 'file') ~= 2
    error('Specified brain area file not found.');
end
if exist(mask_niftigz_file, 'file') ~= 2
    error('Specified brain mask file not found.');
end
if exist(bval_file, 'file') ~= 2
    error('Specified bval_file not found.');
end
if exist(bvec_file, 'file') ~= 2
    error('Specified bvec_file not found.');
end
if isempty(output_mif_file)
    error('Output filename is empty.');
end


%
% --- Main Procedure
%
if exist(output_mif_file, 'file') == 2
    copyfile(output_mif_file, [output_mif_file, '.bak'], 'f');
    delete(output_mif_file);
end

output_dir = fileparts(output_mif_file);
work_dir = tempname;
mkdir(work_dir);

% data copy and unzip
copyfile(data_niftigz_file, work_dir);
copyfile(area_niftigz_file, work_dir);
copyfile(mask_niftigz_file, work_dir);
system(['gunzip ', fullfile(work_dir, filesep, '*.nii.gz')]);

% create unzipped filename
[p, f, e] = fileparts(data_niftigz_file);
temp_data_nifti_file       = fullfile(work_dir, filesep, strrep([f, e], 'nii.gz', 'nii'));
[p, f, e] = fileparts(area_niftigz_file);
temp_area_nifti_file       = fullfile(work_dir, filesep, strrep([f, e], 'nii.gz', 'nii'));
[p, f, e] = fileparts(mask_niftigz_file);
temp_mask_nifti_file       = fullfile(work_dir, filesep, strrep([f, e], 'nii.gz', 'nii'));

temp_response_file         = fullfile(work_dir, filesep, 'response.txt');
temp_bvecbval_mrtrix_file  = fullfile(work_dir, filesep, 'gradients-x.txt');

% Convert format from bvecs, bval => gradients-x.txt
convert_bval_format_to_mrtrix(bval_file, bvec_file, temp_bvecbval_mrtrix_file);

% Estimate response
cmd = ['estimate_response', ...
       ' '      , temp_data_nifti_file, ...
       ' -grad ', temp_bvecbval_mrtrix_file, ...
       ' '      , temp_area_nifti_file, ...
       ' -lmax 6', ...
       ' ', temp_response_file];
dmri_system(cmd, '-echo');

% csdeconv
cmd = ['csdeconv', ...
       ' '      , temp_data_nifti_file, ...
       ' -grad ', temp_bvecbval_mrtrix_file, ...
       ' '      , temp_response_file, ...
       ' -lmax 6', ...
       ' -mask ', temp_mask_nifti_file, ...
       ' ', output_mif_file];
[status, result] = dmri_system(cmd, '-echo');
if status ~= 0
    error(result);
end

% clean up
copyfile(temp_response_file,        output_dir);
copyfile(temp_bvecbval_mrtrix_file, output_dir);
rmdir(work_dir, 's');


function [isok] = convert_bval_format_to_mrtrix(bval_file, bvecs_file, bvecbval_mrtrix_file)

rotbvecs = load(bvecs_file, '-ascii');
rotbvecs = rotbvecs';

% flip x-axis
rotbvecs(:, 1) = -1 * rotbvecs(:, 1);

% read bvals
bvals = load(bval_file, '-ascii');

bb = [rotbvecs, bvals'];
saveascii(bb, bvecbval_mrtrix_file, 8);
