function [Ind, Delta] = dmri_connect_parm_calc(...
  parcels_file, parcel_dir, cmatfile, output_dmri_file, ...
  thre, v, tau, uni_flag, save_flag)
% Compute binarised anatomical connectivity matrix and fiber length matrix
%
% [Units]
%       v: mm/ms (=m/s)
%     tau: ms
%   Delta: ms
% (f_leng: mm)
%
% 2012/12/03 M.Fukushima
% 2012/12/05 M.Fukushima
% 2013/02/12 M.Fukushima
% 2014/06/02 rhayashi      Nlabel, ofile -> parcel_dir
% 2014/12/25 rhayashi      added parcels_file for saving parcellation info
%                                (brain_file, vbmeg_area_ix)
% 2017/10/25 Y. Takeda     added c_streng for saving connectivity strength info
% 2018/10/23 rhayashi      save 'Area_cortex'  
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 4
    error('Please check input arguments.');
end

if ~exist('thre', 'var')
    thre = 1e-4;
end
if ~exist('v', 'var')
    v = 6; % mm/ms = (m/s)
end
if ~exist('tau', 'var')
    tau = 27; % mm
end
if ~exist('uni_flag', 'var')
    uni_flag = 0;
end
if ~exist('save_flag', 'var')
    save_flag = 1;
end
output_dir = fileparts(output_dmri_file);
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
end

%
% --- Main Procedure
%

% Load parcellation information
warning('off', 'MATLAB:load:variableNotFound');
load(parcels_file, 'brain_file', 'vbmeg_area_ix', 'membershipmat', 'Area_cortex');
warning('on',  'MATLAB:load:variableNotFound');

% Load matrix
load(cmatfile, 'c_count', 'f_leng', 'Ntrack', 't_count');

% Load mask
parcel_file = fullfile(parcel_dir, filesep, 'parcel_mm_coord.mat');

load(parcel_file, 'label', 'Nlabel');
c_label = zeros(Nlabel,1);
for ii = 1:Nlabel
  c_label(ii) = sum(label==ii);
end

fprintf('calculating connectivity matrix...');

%%% Metrics calculation
% Connectivity strength
c_streng = c_count.*repmat(c_label,[1 Nlabel])' ...
  ./(repmat(c_label,[1 Nlabel]).*repmat(t_count,[1 Nlabel])');
c_streng(isnan(c_streng)) = 0; % for missing label voxels
c_streng(isinf(c_streng)) = 0;

% Symmetrize
[c_streng, f_leng] = make_symmetry_matrix(c_streng, f_leng);

% Binarised anatomical connectivity matrix
Ind = sparse(c_streng>thre);
Ind = Ind - diag(diag(Ind));

% Fiber length matrix
Delta = (f_leng.*Ind)/v + tau*Ind;

% For diagonal entries
Ind = Ind + speye(size(Ind,1));
Delta = Delta + tau*speye(size(Ind,1));

dmri_parm = [];
dmri_parm.Nlabel = Nlabel;
dmri_parm.ofile = parcel_file;
dmri_parm.cmatfile = cmatfile;
dmri_parm.thre = thre;
dmri_parm.v = v;
dmri_parm.tau = tau;
dmri_parm.uni_flag = uni_flag;

if exist('uni_flag','var')
  if uni_flag
    % Uniform Delta
    Delta = tau*(Delta>0);
  end
end

if exist('save_flag','var')
  if save_flag
    connectivity_matrix = Ind;
    delay_ms_matrix    = Delta;

    fiber_length_m_matrix = zeros(size(delay_ms_matrix));
    ix = find(delay_ms_matrix);
    fiber_length_m_matrix(ix) = (delay_ms_matrix(ix) - dmri_parm.tau)/1000 * dmri_parm.v;
    fiber_length_m_matrix = sparse(fiber_length_m_matrix); 

    % Save
    save(output_dmri_file,'connectivity_matrix', ...
                          'delay_ms_matrix', ...
                          'fiber_length_m_matrix', ...
                          'brain_file', 'vbmeg_area_ix',...
                          'membershipmat', 'dmri_parm','c_streng');
                      
    if exist('Area_cortex', 'var')
        vb_save(output_dmri_file, 'Area_cortex');
    end
  end
end
fprintf('done\n');
