function dmri_4D_image_correct(niftigz_file)
% This functions applies FSL Eddy Current Correction to the
% Diffusion weighted data(4D image)
% [Usage]
%    dmri_4D_image_correct(niftigz_file);
%
% [Input]
%    niftigz_file    : Diffusion weighted data(4D image). 
%                      (.nii.gz)
% [Output]
%    none
%
% [Output files]
%    Diffusion weighted data(4D image) after applying Eddy Current Correction.
%    The output filename is original + _ecc
%
% [See also]
%    https://users.fmrib.ox.ac.uk/~behrens/fdt_docs/fdt_eddy.html
%    Eddy currents in the gradient coils induce (approximate) stretches 
%    and shears in the diffusion weighted images. These distortions are 
%    different for different gradient directions. Eddy Current Correction 
%    corrects for these distortions, and for simple head motion, using 
%    affine registration to a reference volume. 
%
% [Example]
%    dmri_4D_image_correct('/home/cbi/org/DTI_data.nii.gz');
%       creates /home/cbi/org/DTI_data_m.nii.gz
%               /home/cbi/org/DTI_data_m.bvec
%               /home/cbi/org/DTI_data_m.bval
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('niftigz_file', 'var')
    error('Please check arguments.');
end
if exist(niftigz_file, 'file') ~= 2
    error('Input file not found.');
end

%
% --- Main Procedure
%
disp('Applying Eddy current correction...');
start = tic;
[p, f, e] = fileparts(niftigz_file(1:end-7)); % -7 => .nii.gz
bvec_file = [p, '/', f, '.bvec'];
bval_file = [p, '/', f, '.bval'];

c_niftigz_file = [p, '/', f, '_m.nii.gz'];
c_bvec_file    = [p, '/', f, '_m.bvec'];
c_bval_file    = [p, '/', f, '_m.bval'];

command = ['eddy_correct ' niftigz_file, ' ' c_niftigz_file, ' 0'];
[status, cmdout] = dmri_system(command, '-echo');
if status ~= 0
    error('Please check input arguments');
end

% bvec correction and new bvec file is saved as FSL readable format.
disp('Applying bvec correction..');
ecclog = [p, '/', f, '_m.ecclog'];
corrected_bvec_file = ecc_bvecs(ecclog, bvec_file);
mv_cmd = ['mv ' corrected_bvec_file, ' ', c_bvec_file];
system(mv_cmd);

% just copy and rename bval file
cp_cmd = ['cp ' bval_file, ' ', c_bval_file];
system(cp_cmd);
disp('Finished.');
toc(start);
