function basis_parm = vb_util_check_head_shell_info(proj_root, basis_parm)
% check head parameters of basis_parm and set default value
% [usage]
%   basis_parm = vb_util_check_head_shell_info(proj_root, basis_parm)
% [input]
%    proj_root : <required> project root directory
%   basis_parm : <required> target basis_parm
% [output]
%   basis_parm : checked basis_parm
% [note]
%   checked parameters are as follows:
%     1. basis_parm.sigma
%     2. basis_parm.radius
%     3. basis_parm.radius_method
%     4. basis_parm.head_file (if necessary)
% [history]
%   2006-12-19 (Sako) initial version
%   2007-01-29 (Sako) added proj_root argument
%   2007-06-15 (Sako) reformed radius_method ('MEAN','MAX','MIN','USER')
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('proj_root', 'var'), proj_root = []; end
if ~exist('basis_parm', 'var'), basis_parm = []; end
[proj_root, basis_parm] = inner_check_arguments(proj_root, basis_parm);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = 'vb_util_check_head_shell_info';

if ~isfield(basis_parm, 'sigma')
  % Conductivity  from innermost to outermost
  basis_parm.sigma = [ 0.62 0.03 0.62 ];
end

if ~isfield(basis_parm, 'radius')
  % set basis_parm.radius - Relative radius of sphere - by radius_method
  basis_parm.radius = [ 0.87  0.92   1.0 ];
end

if ~isfield(basis_parm, 'radius_method')
  % default
  basis_parm.radius_method = 'MEAN';
else
  if ~strcmp(basis_parm.radius_method, 'USER') ...
      && ~strcmp(basis_parm.radius_method, 'MEAN') ...
      && ~strcmp(basis_parm.radius_method, 'MAX') ...
      && ~strcmp(basis_parm.radius_method, 'MIN')
    warning('(%s)undefined basis_parm.radius_method : %s', ...
      func_, basis_parm.radius_method);  
  end
end

if strcmp(basis_parm.radius_method, 'USER')
  % 'USER' means that user set radius_method on basis_parm.radius
  % So basically, 'radius' should be set.
  if ~isfield(basis_parm, 'radius'),
  
    if isfield(basis_parm, 'head_file')
      calc_mode = 1;  % mean
      rr = vb_util_calc_head_radius(proj_root, basis_parm.head_file, calc_mode);
      basis_parm.radius = rr;
    else  % ~isfield(basis_parm, 'head_file')
      warning('basis_parm does not have head_file.');
    end
  else
    % if basis_parm.radius has been already set, there is nothing to do
  end
  
elseif strcmp(basis_parm.radius_method, 'MEAN')
  rr = vb_util_calc_head_radius(proj_root, basis_parm.head_file,1);
  basis_parm.radius = rr;
  
elseif strcmp(basis_parm.radius_method, 'MIN')
  rr = vb_util_calc_head_radius(proj_root, basis_parm.head_file,2);
  basis_parm.radius = rr;
  
elseif strcmp(basis_parm.radius_method, 'MAX')
  rr = vb_util_calc_head_radius(proj_root, basis_parm.head_file,3);
  basis_parm.radius = rr;
else
  error('invalid basis_parm.radius_method : %s', basis_parm.radius_method);
end

%
% --- recalc
%
if ~isfield(basis_parm, 'Recalc') || isempty(basis_parm.Recalc)
  % default
  basis_parm.Recalc = false;
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
function [proj_root, basis_parm] = inner_check_arguments(proj_root, basis_parm)
if isempty(basis_parm), error('basis_parm is a required parameter'); end
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
