function rr = vb_util_calc_head_radius(proj_root, head_file, calc_mode)
% calculate radius of head from head file
% [usage]
%   rr = vb_util_calc_head_radius(proj_root, head_file)
% [input]
%   proj_root : <required> project root directory
%   head_file : <required> head file (cell array (plural shell model) or not cell)
%   calc_mode : <optional> calculation mode (1 or 2 or 3 or 4) [1]
%             :  =1 : <default> calculate average of distance
%             :  =2 : apply minimum value of all distances
%             :  =3 : apply maximum value of all distances
%             :  =4 : apply user-specified radius
%             :  =other : <warning> calculate average
% [output]
%          rr : rate of radius
% [note]
%   
% [history]
%   2006-12-04 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('proj_root', 'var') 
  error('proj_root is a required parameter');
end

if ~exist('head_file', 'var') || isempty(head_file)
  error('head_file is a required parameter');
end

if ~exist('calc_mode', 'var') || isempty(calc_mode)
  calc_mode = 1;
elseif isempty(find(calc_mode == [1 2 3 4]))
  error('undefined calc_mode : %d', calc_mode);
end

switch calc_mode
  case  1, this_calc_method = 'MEAN';
  case  2, this_calc_method = 'MIN';
  case  3, this_calc_method = 'MAX';
  case  4, this_calc_method = 'USER';
end

rr = [];

if iscell(head_file)
  for i = 1:length(head_file)
    cur_head = fullfile(proj_root, head_file{i});
    if exist(cur_head, 'file') ~= 2
      error('cannot find head file : %s', cur_head);
    end
    load(cur_head, 'Vhead');
    rr = [rr vb_util_distance_of_points(Vhead, calc_mode)];
  end
  rr = rr ./ max(rr);
  
else
  head_file_path = fullfile(proj_root, head_file);
  if exist(head_file_path, 'file') ~= 2
    error('invalid head_file : %s', head_file_path);
  end
  
  head = load(head_file_path);
  
  if ~isfield(head, 'Vhead')
    error('invalid head data - cannot find Vhead field');
  end
  
  % consistent check of calc_mode with Rmethod
  is_consistent_r_method = false;
  if isfield(head, 'Rmethod') && ~isempty(head.Rmethod)
    if strcmp(head.Rmethod, this_calc_method)
      is_consistent_r_method = true;
    end
  end

  if isfield(head, 'R') && ~isempty(head.R) && is_consistent_r_method
    rr = head.R;
    rr = rr ./ max(rr);
    return;
    
  elseif ~isfield(head, 'Nvertex')
    % maybe single shell model
    rr = vb_util_distance_of_points(head.Vhead, calc_mode);
    rr = rr ./ max(rr);
    
  else
    n_shell = size(head.Nvertex,1);
    
    for sh = 1:n_shell
      cur_vhead = head.Vhead(head.Nvertex(sh,1):head.Nvertex(sh,2),:);
      rr = [rr vb_util_distance_of_points(cur_vhead, calc_mode)];
    end
    rr = rr ./ max(rr);
  end
end
%%% END OF FILE %%%
