function [HeadInfo] = vb_head_get_specified_headinfo(headfile, layer)
% return HeadInfo (its fields are Vhead, Fhead, XXhead) of specified layer
% [usage]
%   [HeadInfo] = vb_head_get_specified_headinfo(headfile, layer)
% [input]
%   headfile : <required> headfile (.head.mat)
%      layer : <required> number of layer or layer tag that you'd like to get
% [output]
%   HeadInfo : <struct> its fields are ...
%            :  Vhead
%            :  Fhead
%            :  XXhead
%            :  sigma (if it exists)
% [note]
%   layer is acceptable 'number' and 'string'.
%   if number is given, this function recognizes as number of layer, and
%   if string is given, recognizes as layer tag (e.g. 'Skull')
%   for layer tag, not case-sensitive
%
%   if specified headfile has valid 'sigma' data, you can get relevant sigma
%
% [history]
%   2007-02-14 (Sako) initial version
%   2007-04-10 (Sako) modified Fhead value and added returning sigma
%   2010-10-25 (Sako) supported the case that a head data is empty
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('headfile', 'var'), headfile = []; end;
if ~exist('layer', 'var'), layer = []; end;

[layer_number, head_info] = inner_check_arguments(headfile, layer);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% --- single layer model
%
% if layer_number == 0 || layer_number == 1
if layer_number == 0
  HeadInfo = head_info;
  return;
end

% --- plural layers model
%
IDX_BEGIN = 1;
IDX_END   = 2;
Vhead_begin = head_info.Nvertex(layer_number, IDX_BEGIN);
Vhead_end   = head_info.Nvertex(layer_number, IDX_END);

Fhead_begin = head_info.Npatch(layer_number, IDX_BEGIN);
Fhead_end   = head_info.Npatch(layer_number, IDX_END);

% HeadInfo.Vhead  = head_info.Vhead ([Vhead_begin:Vhead_end], :);
% HeadInfo.XXhead = head_info.XXhead([Vhead_begin:Vhead_end], :);
vhead  = vb_headinfo_get_vhead(head_info);
xxhead = vb_headinfo_get_xxhead(head_info);
fhead  = vb_headinfo_get_fhead(head_info);

HeadInfo = [];

if ~isempty(vhead)
  cur_vhead = vhead(Vhead_begin:Vhead_end, :);
else
  cur_vhead = [];
end
HeadInfo = vb_headinfo_set_vhead(HeadInfo, cur_vhead);

if ~isempty(xxhead)
  cur_xxhead = xxhead(Vhead_begin:Vhead_end, :);
else
  cur_xxhead = [];
end
HeadInfo = vb_headinfo_set_xxhead(HeadInfo, cur_xxhead);

if ~isempty(fhead)
  tmpFhead  = fhead (Fhead_begin:Fhead_end, :);
  % HeadInfo.Fhead  = tmpFhead - (Vhead_begin-1);
  cur_fhead = tmpFhead-(Vhead_begin-1);
else
  cur_fhead = [];
end
HeadInfo = vb_headinfo_set_fhead(HeadInfo, cur_fhead);

% sigma
sigma = vb_headinfo_get_sigma(head_info);
if ~isempty(sigma)
  HeadInfo = vb_headinfo_set_sigma(HeadInfo, sigma(layer_number));
else
  HeadInfo = vb_headinfo_set_sigma(HeadInfo, []);
end

% layer_tag
layer_tag = vb_headinfo_get_layertag(head_info);
if ~isempty(layer_tag)
  HeadInfo = vb_headinfo_set_layertag(HeadInfo, layer_tag{layer_number});
else
  HeadInfo = vb_headinfo_set_layertag(HeadInfo, []);
end

% Radius
thisRadius = vb_headinfo_get_radius(head_info);
if ~isempty(thisRadius)
  HeadInfo = vb_headinfo_set_radius(HeadInfo, thisRadius(layer_number));
else
  HeadInfo = vb_headinfo_set_radius(HeadInfo, []);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
function [layer_number, head_info] = inner_check_arguments(headfile, layer)
layer_number = 0;

if isempty(headfile), error('headfile is a required parameter'); end;

if exist(headfile, 'file') ~= 2
  error('cannot read headfile : %s', headfile); end;

if isempty(layer)
  error('neither layer_number nor layer_tag are NOT specified'); end;


% --- load
head_info = load(headfile);

% --- check whether be plural layer model or not
if ~isfield(head_info, 'Nvertex')
  if isfield(head_info, 'Vhead') ...
      && isfield(head_info, 'Fhead') ...
      && isfield(head_info, 'XXhead')
      % single layer model
      layer_number = 0;
    return;
  else
    error('incomplete headfile : %s', headfile);
  end
elseif isempty(head_info.Nvertex)
  layer_number = 0;
  return;
end

% plural layers
% Nlayer = size(head_info.Nvertex, 1);
Nlayer = vb_headinfo_get_layer_num(head_info);

if isnumeric(layer)
  layer_number = layer;
  if layer_number > Nlayer
    error('specified layer is over : %d', layer_number);
  end
  return;
else
  if isfield(head_info, 'LayerTag')
    specified_tag = upper(layer);
    
    % matching
    for nl = 1:Nlayer
%       my_tag = upper(deblank(head_info.LayerTag(nl,:)));
      my_tag = upper(deblank(head_info.LayerTag(:,nl)));
      if strcmp(specified_tag, my_tag)
        layer_number = nl;
        break;
      end
    end
    if layer_number == 0
        head_info = [];
    end
  else
    error('There is not LayerTag field in this head model : %s', headfile);
  end
end
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %


%%% END OF FILE %%%
