function vb_job_vb_dynamics(varargin)
% Jointly estimate source activities and interactions
%
% [syntax]
% vb_job_vb_dynamics(dbayes_parm)
% vb_job_vb_dynamics(proj_root,dbayes_parm)    [old style]
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% dbayes_parm: <<struct>> Parameters for variational Bayes method.
% --- fields of dbayes_parm
%  (filename parameters) - relative path from proj_root (inside project files.)
%  bayesfile  : <<string>> Current variance file used as initial values (.bayes.mat). 
%  dbayesfile : <<string>> File for saving source activities and interactions (.dbayes.mat).
%
%                        - fullpath is required (outside project files.)
%  dmri_file  : <<string>> Diffusion MRI file (.dmri.mat).
%
%  (prior parameters: see Fukushima et al., 2015, Theory setion)
%  ieta0 : <<double>> Prior mean of the inverse variance parameters of
%          the MAR coefficients (\bar{\eta}_0^{-1} in TeX representation)
%  g0    : <<double>> Shape parameter of gamma probability distribution
%          for the inverse variance parameters of the MAR coefficients
%
%  (forward model parameters)
%  area_key       : <<string>> ID of cortical area at which current
%                   dipoles are assumed.
%  area_key_global: <<string>> ID of cortical area at whith current
%                   dipoles are assumed in estimation of baseline
%
%  (algorithm parameters)
%  Ntrain    : <<int>> Iteration number of algorithm.
%  Npre_train: <<int>> Iteration number of algorithm (=Ntrain).
%
% [output]
% Estimation result will be saved to the file specified by
% dbayes_parm.dbayesfile with the following variables:
%
%  Model.a    : current noise variance.
%  Model.sx   : observation noise variance.
%  Model.v    : set to 0.
%  Model.Z    : current sources
%  Model.Vz   : source covarinace matrices.
%  Model.vz_sample : # of time samples for each cell entry of Model.Vz
%  Model.mar  : MAR coefficients
%  Model.MAR  : MAR matrix
%  Model.MAR0 : MAR matrix at the 1st iterative update
%  Model.Vmar : Covariace matrices of MAR coefficients
%  Model.ieta : Variance parameters of MAR coefficients
%  Model.a0   : prior mean of current noise variance.
%  Model.sx0  : observation noise variance (baseline).
%  Model.v0   : set to 0.
%  Model.ix   : Dipole indices of Model.a. 
%  Model.Cov  : Noise covariance matrix.
%  dbayes_parm: Settings specific to dynamic estimation. 
%  vb_parm    : Settings for VB iterative updates. 
%  Info       : Free energy and its constituents.
%
% [reference]
% Fukushima et al. (2015) NeuroImage 105, 408-427. 
%
% [history]
%
% 2015/03/18 M.Fukushima
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if length(varargin) == 1
  proj_root = [];
  dbayes_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  dbayes_parm = varargin{2};
end

%
% --- Previous check
%
if ~isfield(dbayes_parm, 'bayes_file')
    error('dbayes_parm.bayes_file is a required field.');
end
init_value_file = fullfile(proj_root, dbayes_parm.bayes_file);
    
if exist(init_value_file, 'file') ~= 2
    error('dbayes_parm.bayes_file not found.');
end

if ~isfield(dbayes_parm, 'dmri_file') || ...
   exist(dbayes_parm.dmri_file, 'file') ~= 2
    error('dbayes_parm.dmri_file not found.');
end
if ~isfield(dbayes_parm, 'dbayes_file') || isempty(dbayes_parm.dbayes_file)
    error('dbayes_parm.dbayes_file: Output filename not specified.');
end
if ~isfield('dbayes_parm', 'area_key')
    dbayes_parm.area_key = [];
end
if ~isfield('dbayes_parm', 'area_key_global');
    dbayes_parm.area_key_global = [];
end

if length(dbayes_parm.dbayes_file) < 11
    error('Please check output filename.(.dbayes.mat)');
end
save_file = fullfile(proj_root, dbayes_parm.dbayes_file);
if ~strcmp(save_file(end-10:end), '.dbayes.mat')
    save_file = [save_file, '.dbayes.mat'];
    warning('save filename doesn''t have an extension(.dbayes.mat)\n filename was changed to %s', save_file);
end

%
% --- Main Procedure
%
proj_root = vb_rm_trailing_slash(proj_root);

const = vb_define_verbose;
VERBOSE_LEVEL_INFO = const.VERBOSE_LEVEL_INFO;

load(init_value_file, 'bayes_parm');
if ~isfield(bayes_parm, 'spatial_smoother')
    dp = vb_set_bayes_default_parameters;
    bayes_parm.spatial_smoother = dp.spatial_smoother;
end

if ~strcmp(bayes_parm.forward_model,'focal')
    warning(['foward_model was changed from ' ...
             '''' bayes_parm.forware_model '''' ...
             ' to ''focal''.']);
    bayes_parm.forward_model = 'focal';
end
if bayes_parm.patch_norm == ON
    error(['The specified initial value file was calculated using ', ...
           'bayes_parm.patch_norm=ON.', sprintf('\n'), ...
           'Please calculate it using bayes_parm.patch_norm=OFF.']);
end

%
% --- Overwrite settings
%

% bayes_parm.megfile
% bayes_parm.megfile_baseline
if isfield(dbayes_parm, 'meg_file') && ~isempty(dbayes_parm.meg_file)
   if ~ischar(dbayes_parm.meg_file)
       error('dbayes_parm.meg_file : accept only char string.');
   end
   bayes_parm.megfile          = {dbayes_parm.meg_file};
   bayes_parm.megfile_baseline = {dbayes_parm.meg_file};
end

%
% Change path to absolute path
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Dynamics estimation -----',VERBOSE_LEVEL_INFO)
bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);

% create temp area file and add diffusion parcellation area
% to the copied area file. then use it instead of original area file.
area_key = 'dmri_parcellation_area';
[bayes_parm_abs, dbayes_parm] = ...
    inner_replace_area_file_with_diffusion_area_file(...
                                    bayes_parm_abs, dbayes_parm, area_key);
%
% Observation noise estimation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Observation noise model specification', ...
        VERBOSE_LEVEL_INFO); 

% Cov = Normalized noise covariance matrix
[Cov,sx00] = vb_observation_noise_specification(bayes_parm_abs);

%
% Preparation of lead fields
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Leadfield matrix preparation',VERBOSE_LEVEL_INFO); 

lf_parm.brainfile  = bayes_parm_abs.brainfile;
lf_parm.areafile   = bayes_parm_abs.areafile;
lf_parm.patch_norm = bayes_parm_abs.patch_norm;
lf_parm.expand_spatial_filter = bayes_parm_abs.expand_spatial_filter;
lf_parm.spatial_smoother = bayes_parm_abs.spatial_smoother;

% Global window
vb_disp('--- Lead field matrix of global window ',VERBOSE_LEVEL_INFO);
lf_parm.basisfile = bayes_parm_abs.basisfile_global;
lf_parm.area_key  = dbayes_parm.area_key_global;
lf_parm.reduce    = bayes_parm_abs.reduce_global;
lf_parm.Rfilt     = bayes_parm_abs.Rfilt_global;

[Gall, ix_all]    = vb_leadfield_preparation(lf_parm);

% Focal window
vb_disp('--- Lead field matrix of focal window ',VERBOSE_LEVEL_INFO);
lf_parm.basisfile = bayes_parm_abs.basisfile;
lf_parm.area_key  = dbayes_parm.area_key;
lf_parm.reduce    = bayes_parm_abs.reduce;
lf_parm.Rfilt     = bayes_parm_abs.Rfilt;
lf_parm.remove_area_key = [];

[Gact, ix_act] = vb_leadfield_preparation(lf_parm);

% Note: Gall and Gact are not normalized here!

%
% Normalization of lead fields
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Nsession = length(Gact);  % Number of sessions

% Normalization constant of leadfield
Bnorm = zeros(Nsession,1);
for n = 1: Nsession
  GGall = sum(Gall{n}.^2, 2);
  Bnorm(n)  = sum(GGall)/length(GGall);
end
bsnorm = sqrt(mean(Bnorm));

% Normalization of leadfield
for n=1:Nsession
  nGall{n} = Gall{n}/bsnorm;
  nGact{n} = Gact{n}/bsnorm;
end

clear Gall Gact;

%
% MEG data preparation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[B, Ntrial, tmp, Tsample] = vb_megdata_preparation(bayes_parm_abs);

%
% Check variable consistency
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_check_variable(B, Cov, nGact, nGall);

%
% Estimation of baseline current variance
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Baseline current variance estimation',VERBOSE_LEVEL_INFO);
      
% current baseline estimation at pretrigger period
% together with spherical sensor noise estimation
[v0, sx0] = vb_baseline_variance_acquisition(bayes_parm_abs, nGall);

if strcmp(bayes_parm_abs.forward_model,'focal');
    sx0 = sx00;
end

%
% Set VB parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Njact = size(nGact{1},2);
Njall = size(nGall{1},2);
vb_parm = vb_set_vb_parm(bayes_parm_abs, B, Njact, Njall, v0, sx0);
vb_parm.Ntrain = dbayes_parm.Ntrain;

%
% Set prior parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
prior_parm.ix = ix_act;
prior_parm.v0 = v0;
prior_parm.a0 = bayes_parm.a0;
prior_parm.a0_act = bayes_parm.a0_act;

w = bayes_parm.prior_weight;
prior_parm.Ta0 = Ntrial * Tsample * w / (2*(1-w));
prior_parm.Ta0_act = prior_parm.Ta0;

prior_parm.act = vb_get_act(bayes_parm_abs.actfile, ...
                         bayes_parm_abs.act_key);
[a0,Ta0] = vb_set_vb_prior(prior_parm);
vb_parm.a0 = a0;
vb_parm.Ta0 = Ta0;

%
% Wiener filter (removed from the previous version)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Model = [];

%
% Estimation with focal (and global) window
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% Structural Connectivity
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

connectivity_matrix = [];
delay_ms_matrix     = [];
load(dbayes_parm.dmri_file, 'connectivity_matrix', 'delay_ms_matrix');

vb_parm.Ind = connectivity_matrix;
meginfo = vb_load_meg_info(bayes_parm_abs.megfile{1});
vb_parm.Delta = round(delay_ms_matrix*meginfo.SampleFreq*1e-3); % ms -> samples

%
% Hyperparameters for the variance parameter of the MAR coefficients
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

vb_parm.ieta0 = dbayes_parm.ieta0;
vb_parm.g0 = dbayes_parm.g0;

%  'focal', 
vb_parm.v0  = 0;
vb_parm.Tv0 = 0; 

  vb_parm.a = vb_calc_initialvalue_a(init_value_file, ...
    bayes_parm_abs.brainfile, dbayes_parm.dmri_file, bsnorm, vb_parm.a_min);
[Model,Info] = vbmeg_ard_estimate25_3(B, [], nGact, ...
                                   Cov, vb_parm, Model);

Model.sx0 = sx0;

%
% Denormalization
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %  Alpha is scaled back by bsnorm
  fprintf('Alpha is scaled back by bsnorm\n')
  Model.a = (Model.a)./bsnorm^2;
  Model.v = (Model.v)./bsnorm^2;
  Model.a0 = (vb_parm.a0)./bsnorm^2;
  Model.v0 = (vb_parm.v0)./bsnorm^2;
  Model.Cov = Cov;
  Model.ix = ix_act;
  for ns = 1:Nsession
    Model.Z{ns} = (Model.Z{ns})./bsnorm;
    for nd = 1:length(Model.Vz{ns})
      Model.Vz{ns}{nd} = (Model.Vz{ns}{nd})./bsnorm^2;
    end
  end

%
% Save result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf('----- Save estimation result in %s\n',save_file)

vb_fsave(save_file, ...
        'Model','dbayes_parm','vb_parm',...
         'Info');

% % project_file save
% proj_file = get_project_filename;
% if isempty(proj_file)
%     return;
% end

% project_file_mgr('load', proj_file);
% project_file_mgr('add', 'bayes_parm', bayes_parm);

function [bayes_parm_abs, dbayes_parm] = ...
    inner_replace_area_file_with_diffusion_area_file(...
                                                bayes_parm_abs, ...
                                                dbayes_parm, area_key)
% copy specified area file as a temporary file and dMRI are into the file,
% then return the filename.
% [Input]
%    bayes_parm_abs  : initial value is calculated by this bayes_parm.
%    dnayes_parm     : dynamics bayes estimation parameter.
%    area_key        : save dmri area into a temporary file by this name.
% [Output]
%    bayes_parm_abs.areafile : temporary area filename including dMRI area.

if ~isfield(dbayes_parm, 'area_key') || isempty(dbayes_parm.area_key)
    dbayes_parm.area_key = area_key;
end
if ~isfield(dbayes_parm, 'area_key_global') || isempty(dbayes_parm.area_key_global)
    dbayes_parm.area_key_global = area_key;
end

% find area key
found = false;
load(bayes_parm_abs.areafile, 'Area');
for k=1:length(Area)
    if strcmp(Area{k}.key, area_key)
        found = true;
    end
end
if found, return; end

% bayes_parm_abs.areafile 
load(dbayes_parm.dmri_file, 'vbmeg_area_ix');

% copy area file to temp area file and put dMRI area into the file.
temp_area_file = [tempname(tempdir), '.area.mat'];
copyfile(bayes_parm_abs.areafile, temp_area_file);
area_new.Iextract = vbmeg_area_ix;
area_new.key      = area_key;
vb_add_area(temp_area_file, area_new);

% use temp_area_file instead of original area file.
bayes_parm_abs.areafile = temp_area_file;
