function [Zact, Jinfo, Jext] = vb_job_current_dynamics(varargin)
% Estimate current time course by using inverse filter. 
% (VBMEG public function)
%
% [syntax]
% [Zact,Jinfo,Jext] = vb_job_current_dynamics(curr_parm)
% [Zact,Jinfo,Jext] = vb_job_current_dynamics(proj_root,curr_parm) [old style]
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% curr_parm: <<struct>> Current estimation parameters. 
% --- fields of curr_parm
%  dbayesfile   : <<string>> Dynamics estimation filename (.dbayes.mat). This
%                 file is required for current estimation and obtained by
%                 vb_job_vb_dynamics. 
%  currfile     : <<string>> Cortical current filename (.curr.mat),
%                 specifying the file to which current estimation result
%                 is saved. 
%  area_key     : !! currently unconfirmed.
%                 <optional> <<string>> Area ID on which cortical current
%                 is estimated. This parameter will be useful if you have
%                 some knowledge about region of interested. 
%  areafile     : !! currently unconfirmed.
%                 <optional> <<string>> Required if cortical area is
%                 specified by field 'area_key'. 
%  trial_average: !! currently unconfirmed.
%                 <optional> <<bool>> If true (default), cortical current
%                 averaged across all trials. If false, cortical current
%                 is estimated for all trials. 
%  jactdir      : !! currently unconfirmed.
%                 <optional> <<string>> Directory for saving trial
%                 current files. This is the relative path from the
%                 cortical current file. If trial_average == false &
%                 jactdir is specified, current for each trial is saved
%                 as trial current file. 
%  dsampf       : !! currently unconfirmed.
%                 <optional> <<int>> Specify frequency of
%                 downsampling. This value must be smaller than the
%                 original sampling frequency of M/EEG data. 
%  overlap_mode : !! currently unconfirmed.
%                 <optional> <<bool>> If false (default), cortical
%                 current is averaged over overlapped time
%                 windows. Otherwise, current is not averaged over
%                 overlapped time windows; current time series of each
%                 time windows are concatenated sequentially for spectral
%                 analysis. 
%  !! currently unconfirmed.
%     If following fields are given, these values are used instead of
%     bayes_parm field in result file (see vb_job_vb): 
%      basisfile
%      megfile  
%      twin_meg
%      Tperiod  
%      Tnext
%      extra.basisfile (for extra dipole)
% ---
%
% [output]
% Zact : <<float matrix>> Z-current.
%  Zact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'. 
%  Zact(Nact,Nsample)          for trial_average = true. 
%  Zact(Nact,Nsample,Ntrials)  for trial_average = false. 
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrials  : # of trials in all session]
% Jinfo: <<struct>> Information of cortical current.
% Jext : <<double matrix>> Extra dipole current. 
%
% [example]
% >> parm.dbayesfile = './NG/result/result3_col.dbayes.mat';
% >> parm.currfile = './result/result3_col_fMTl_1.curr.mat';
% >> vb_job_current_dynamics(parm);
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% check input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if length(varargin) == 1
  proj_root = [];
  curr_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  curr_parm = varargin{2};
else
  error('Error: invalid usage of job_current_dynamics.');
end;

proj_root = vb_rm_trailing_slash(proj_root);

if ~isfield(curr_parm,'trial_average'), 
  curr_parm.trial_average = true; 
end

if ~isfield(curr_parm,'areafile'), 
  curr_parm.areafile = [];
end

%
% set absolute path
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(proj_root)
  currfile  = fullfile(proj_root, filesep, curr_parm.currfile);  % current file
  dbayesfile = fullfile(proj_root, filesep, curr_parm.dbayesfile);
  if isfield(curr_parm, 'areafile')
    areafile   = fullfile(proj_root, filesep, curr_parm.areafile);   % vb_get_area
  end
else
  currfile   = curr_parm.currfile;
  dbayesfile = curr_parm.dbayesfile;

  if isfield(curr_parm, 'areafile')
    areafile   = curr_parm.areafile;
  end
end

%
% Current estimation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('Start current estimation')

if ~isempty(areafile) ...
  & isfield(curr_parm, 'area_key') & ~isempty(curr_parm.area_key),
    Area = vb_get_area(areafile, curr_parm.area_key);
    ix_area = Area.Iextract;
else
    ix_area = [];
end

parm = curr_parm;
parm.ix_area = ix_area;
if curr_parm.trial_average == false ...
    error('Currently we assumes with trial average = ON');
%      && isfield(curr_parm,'jactdir') && ~isempty(curr_parm.jactdir)
%  [Zact ,Jinfo, bayes_parm, vb_parm, MEGinfo, Jext, Pointlist] ...
%      = vb_current_reconstruct_z_tr(proj_root, parm);
else
  [Zact, Jinfo, bayes_parm, vb_parm, MEGinfo, Jext, Pointlist] ...
      = vb_current_reconstruct_z_dynamics(proj_root, parm);
end

% load brain vertex number
brain_file = fullfile(proj_root, bayes_parm.brainfile);
Nvertex = vb_load_cortex_info(brain_file);

%
% Save result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp(sprintf('--- Save estimated current: %s\n',currfile));
load(dbayesfile, 'dbayes_parm');
vb_fsave(currfile,'Zact','Jinfo','MEGinfo', 'Nvertex', 'curr_parm','vb_parm', 'bayes_parm', 'dbayes_parm');

%
% project_file save
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
proj_file = get_project_filename;
if isempty(proj_file)
  return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'current_parm', curr_parm);

return;
