function vb_job_preprocess_meg3(proj_root, parm)
%---
% function vb_job_preprocess_meg3(proj_root, parm)
%
% This is exclusive use of SBI system.
% --
% Reject noisy sensors and trials, and divide trials according to
% experimental conditions in division file. If division file
% (divfile) was not specified, MEG data will not be divided. 
% If parm was not specified, recommended values will be used. 
%
% Input parameters (recommended value)
% - parm.rm_trial  : It can be used to directly specify trials to
%                    be rejected. 
% - parm.verify    : If it is ON (constant), the script shows
%                    trials that should be rejected and the user
%                    must verify each trial by clicking 'OK' button
%                    on a dialog (ON). 
% - parm.jobmode   : Select jobs and order of that in this
%                    preprocessing. 
%                    NOTICE that the result of preprocessing would
%                    be changed if you specify the order of jobs.
%                    The following (default) order is recomended.
%                    1 -- Reject dead sensors 
%                    2 -- Remove drift and correct baseline 
%                    3 -- Remove trials in which EOG signal exceeds
%                         a threshold.
%                    4 -- Reject sensors on which MEG time courses
%                         of some trials are noisy.  
%                    5 -- Reject noisy trials
% - Ohter parameters for each job
% -- 1 -- Reject dead sensors 
%
% -- 2 -- Remove drift and correct baseline 
% - parm.freq      : MEG data frequency.
% - parm.driftremovalmode
%                  : 'linear' -- remove drift with linear fitting.
%                  : 'quadra' -- remove drift with quadra fitting.
%                  : 'sin' -- remove drift with sin-cos fitting.
% - parm.basiocycle: when parm.driftrimovalmode = 'sin', you need
%                    specify this parameter as vector of
%                    basio-cycles, for example parm.basiocycle =
%                    [4,8,16,32] (second). It is recomended that
%                    basiocycle > trial time length.
% - parm.baseline_pretrig
%                  : If this parameter is ON, baseline correction
%                    by using median of MEG data at pretrigger
%                    period for each sensor.
%
% -- 3 -- Remove trials in which EOG signal exceeds a threshold.
% - parm.num_eog_channels : the number of eog channels
% - parm.eog_channel_id : channel ID in which EOG data were stored
%                         (for YOKOGAWA system)
% - parm.th_eog    : Threshold for EOG signal. This parameter must
%                    be vector at the same length as the number of
%                    'parm.num_eog_channels'. Each element of the
%                    vector corresponds to the element of
%                    'parm.eog_channel_id'.
%                    ([0.75 0.75])
%
% -- 4,5 -- Reject sensors on which MEG time courses of some trials are noisy.
%           Reject noisy trials.
% - parm.th_meg    : Threshold for noise in MEG data (8)
% - parm.meg_coef  : Threshold for power of MEG data considered as
%                    a candidate of rejected sensor (5.0)
% - parm.Ntrial_th : The number of trials over which MEG time
%                    courses exhibit noise on the same sensor (5)
%
% 2004-12-28 Taku Yoshioka
% 2005-08-16 Taku Yoshioka
% 2006-06-14 Dai Kawawaki
% 2006-12-19 Dai Kawawaki (bug fix)
% 2007-04-17 Dai Kawawaki (bug fix)
%---
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

proj_root = vb_rm_trailing_slash(proj_root);

megfile = [proj_root filesep parm.megfile];
divfile = parm.divfile;

[MEGinfo, MEGdata, divNo] = init_preprocessing_meg(megfile,divfile);

for job = 1:length(parm.jobmode)
  switch parm.jobmode(job)
   case 1 
    % Reject dead sensors
    [MEGinfo,MEGdata] = reject_dead_sensors(parm,MEGinfo,MEGdata);
    
   case 2
    % Remove drift and correct baseline
    [MEGdata] = remove_drift_correct_baseline(parm,MEGinfo,MEGdata);
    
   case 3
    % Remove trials in which EOG signal exceeds a threshold. 
    [MEGinfo,MEGdata] = remove_trials_when_eye_moved(parm,MEGinfo,MEGdata);
   case 4
    % Reject sensors on which MEG time courses of some trials are noisy. 
    [MEGinfo,MEGdata] = reject_noisy_sensors(parm,MEGinfo,MEGdata);
   case 5
    % Reject noisy trials
    [MEGinfo,MEGdata] = reject_noisy_trials(parm,MEGinfo,MEGdata);
   otherwise
    error('You cannot specify jobmode any other than [1:5].');
  end %switch(parm.jobmode)
end %for job = 1:length(parm.jobmode)

%save division
save_division(parm, megfile, MEGinfo, MEGdata, divNo);
return




%internal functions

%
% Load MEG data
%
function [MEGinfo,MEGdata,divNo] = init_preprocessing_meg(megfile,divfile)
fprintf('--- initializing... \n');

MEGinfo = vb_load_meg_info(megfile);
bexp = vb_load_meg_data(megfile); 
% eog = load_eog_data(megfile);
eog = vb_load_megeeg_data(megfile);
[pick, Qpick, CoilWeight] = vb_load_sensor(megfile);
Ntrial = MEGinfo.Nrepeat_org;

% Load division file
divNo = load_division_file(bexp,divfile,Ntrial);

% set meg data
MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight);

fprintf('--- done.\n');
return

%
% Load division file
%
function divisionNumber = load_division_file(bexp,divfile,Ntrial)
if isempty(divfile),
  divisionNumber = zeros(Ntrial,1);
else
  divisionNumber = load('-ascii',divfile);
  if length(divisionNumber)~=Ntrial;
    error(['The number of trials does not match with the number of ' ...
	   'experimental conditions.']);
  end
end
return

%
% Reject dead sensors
%
function [MEGinfo,MEGdata] = reject_dead_sensors(parm,MEGinfo,MEGdata)
fprintf('--- searching dead sensosrs...\n');

[bexp,eog,pick,Qpick,CoilWeight]= get_meg_data(MEGdata);
chNo = MEGinfo.MEGch_name;%fixed by D.K. at 06-12-19
trNo = MEGinfo.TrialNumber;
sampleTime = MEGinfo.Nsample;
Ntrial = MEGinfo.Nrepeat;
Nchannel = MEGinfo.Nchannel;
rejectChannel = [];

h       = waitbar(0,'Check dead sensor');
prg     = 0;
prg_all = Nchannel;
vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
refresh(h);
if parm.verify==ON, hh = figure; hold on; end;

for ixch = 1:Nchannel
  waitbar(ixch/Nchannel);
  
  for ixtr=1:Ntrial
    max_value = max(bexp(ixch,:,ixtr));
    min_value = min(bexp(ixch,:,ixtr));
    if max_value==min_value
      if parm.verify==ON, 
	figure(hh);clf(hh);hold on; 
	plot(bexp(:,:,ixtr)',':');
	plot(bexp(ixch,:,ixtr)','linewidth',4);
	title(sprintf('trial %d - sensor %d',trNo(ixtr),chNo(ixch)));
	xlim([1 sampleTime]);
	button = ...
	    questdlg(['Reject sensor ' num2str(chNo(ixch)) '?'],...
		     'Sensor rejection','Yes','No','Yes');
      else
	button = 'Yes';
      end
      if strcmp(button,'Yes')
	disp(sprintf('---- sensor %d was rejected at trial %d.',...
		     chNo(ixch), trNo(ixtr)));
	rejectChannel = [rejectChannel; chNo(ixch)];
	break;
      end
    end
  end
  
  for ii=1:15; vb_disp_nonl(sprintf('\b')); end
  vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
  prg = prg+1;
end

close(h); 
vb_disp_nonl(sprintf('\n'));

if parm.verify==ON, close(hh); end;
% modified by DK at 070417
if isempty(vb_setdiff2(MEGinfo.MEGch_id, rejectChannel));
  error('All sensors were rejected. Please check parameters.');
end
[MEGinfo,bexp,pick,Qpick,CoilWeight] = ...
    vb_channel_rejection(MEGinfo,bexp,pick,Qpick,rejectChannel,CoilWeight);
MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight);
fprintf('--- done.\n');
return


%
% remove drift & baseline correction
%
function [MEGdata] = remove_drift_correct_baseline(parm,MEGinfo,MEGdata)
fprintf('--- removing drift component... \n');

[bexp,eog,pick,Qpick,CoilWeight]= get_meg_data(MEGdata);
chNo = MEGinfo.MEGch_name;%fixed by D.K. at 06-12-19
trNo = MEGinfo.TrialNumber;
Ntrial = MEGinfo.Nrepeat;
Nchannel = MEGinfo.Nchannel;
sampleTime = MEGinfo.Nsample;
time = [1:1000/parm.freq:sampleTime];
pre_trigger = MEGinfo.Pretriger;

h       = waitbar(0,'remove drift & correct baseline');
prg     = 0;
prg_all = Ntrial;
vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
refresh(h);

for ixtr=1:Ntrial
  waitbar(ixtr/Ntrial);

  %remove drift
  switch lower(parm.driftremovalmode)
   case 'sin'
    bexp(:,:,ixtr) = vb_polysincosfilt(bexp(:,:,ixtr),...
				    parm.basiocycle,...
				    parm.freq);

   case 'linear'
    for ixch=1:Nchannel
      P = polyfit(time,bexp(ixch,:,ixtr),1);
      dr = P(1)*time + P(2);
      bexp(ixch,:,ixtr) = bexp(ixch,:,ixtr) - dr;
    end
    
   case 'quadra'
    for ixch=1:Nchannel
      P = polyfit(time,bexp(ixch,:,ixtr),2);
      dr = P(1)*time.^2 + P(2)*time + P(3);
      bexp(ixch,:,ixtr) = bexp(ixch,:,ixtr) - dr;
    end

   otherwise %D.K. 06-12-19
    error('invalid usage of parameter for drift rimoval.');
    
  end
  
  %baselien correction with pretrigger
  if parm.baseline_pretrig == ON
    base_line = median(bexp(:,1:pre_trigger,ixtr),2);
    bexp(:,:,ixtr) = bexp(:,:,ixtr) - repmat(base_line, 1, sampleTime);
  end
  
  for ii=1:15; vb_disp_nonl(sprintf('\b')); end
  vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
  prg = prg+1;
end
close(h);
vb_disp_nonl(sprintf('\n'));

MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight);

fprintf('--- done.\n');
return


%
% Remove trials in which EOG signal exceeds a threshold. 
%
function [MEGinfo,MEGdata] = remove_trials_when_eye_moved(parm,MEGinfo,MEGdata);
fprintf('--- rejecting trials in which subject''s eyes were moved...\n');

[bexp,eog,pick,Qpick,CoilWeight]= get_meg_data(MEGdata);
chNo = MEGinfo.MEGch_name;%fixed by D.K. at 06-12-19
trNo = MEGinfo.TrialNumber;
Ntrial = MEGinfo.Nrepeat;
sampleTime = MEGinfo.Nsample;

Neog = parm.num_eog_channels;
eog_id = parm.eog_channel_id;
th_eog = parm.th_eog;
ix_eog = chname2index_eog(MEGinfo,Neog,eog_id);

rejectTrial = [];

h       = waitbar(0,'remove drift & correct baseline');
prg     = 0;
prg_all = Ntrial;
vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
refresh(h);
if parm.verify==ON, hh = figure; hold on; end;

if ~isempty(eog) 
  for ixtr=1:Ntrial
    waitbar(ixtr/Ntrial); 
    
    for n = 1:Neog
      diff_eog = max(eog(ix_eog(n),:,ixtr))-min(eog(ix_eog(n),:,ixtr));
      if diff_eog > th_eog(n) %fixed 06-12-19 D.K.
	if parm.verify==ON, 
	  figure(hh);clf(hh);hold on;
	  subplot(2,1,1); 
	  plot(eog(ix_eog,:,ixtr)');
	  legend('eog channel 1','eog channel 2','...'); 
	  xlim([1 sampleTime]);
	  subplot(2,1,2); 
	  plot(bexp(:,:,ixtr)'); 
	  xlim([1 sampleTime]);
	  title(['Trial ' num2str(ixtr)]);
	  button = questdlg(['Reject trial ' num2str(trNo(ixtr)) '?'],...
			    '   EOG rejection','Yes','No','Yes');
	else
	  button = 'Yes';
	end 
	
	if strcmp(button, 'Yes')
	  rejectTrial = [rejectTrial; trNo(ixtr)];
	  fprintf('---- trial %d was rejected.\n',trNo(ixtr));
	  break;
	end
      end 
    end %for n = 1:Neog
    
    for ii=1:15; vb_disp_nonl(sprintf('\b')); end
    vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
    prg = prg+1;
  end %for ixtr=1:Ntrial
end 

close(h);
vb_disp_nonl(sprintf('\n'));

if parm.verify==ON, close(hh); end; 
% modified by DK at 070417
if isempty(vb_setdiff2(MEGinfo.TrialNumber, rejectTrial)) 
  error('All trials were rejected. Please check parameters.');
end
[MEGinfo,bexp,eog] = vb_trial_rejection(MEGinfo,bexp,eog,rejectTrial);
MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight);
fprintf('--- done.\n');
return

function ix_eog = chname2index_eog(MEGinfo,Neog,eog_id)
switch (lower(MEGinfo.device))
 case 'sbi'
  ix_eog = 1:Neog;
 case 'yokogawa'
  chname = MEGinfo.EEGinfo.all_ch_name;
  [tmp,ix_eog] = intersect(chname,eog_id);
  if isempty(tmp)
    error('There are no such channel name.');
  end
  if length(tmp) ~= length(eog_id)
    warning('Some channel names you specified did not match.');
  end
 otherwise
  error('MEG system is unknown. Please check the MEG system you used.');
end
return


%
% Reject sensors on which MEG time courses of some trials are
% noisy. 
%
function [MEGinfo,MEGdata] = reject_noisy_sensors(parm,MEGinfo,MEGdata)
fprintf('--- rejecting noisy sensors...\n');
[bexp,eog,pick,Qpick,CoilWeight]= get_meg_data(MEGdata);
%fixed by D.K. at 06-12-19 
%fixed by D.K. at 07-04-17 
chNo = MEGinfo.MEGch_id;
trNo = MEGinfo.TrialNumber;
Nchannel = MEGinfo.Nchannel;
Ntrial = MEGinfo.Nrepeat;
sampleTime = MEGinfo.Nsample;

rejectChannel = [];

ixcount = cell(Nchannel,1);
ratio_all = [];

h       = waitbar(0,'Check noisy sensor');
prg     = 0;
prg_all = Ntrial;
vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
refresh(h);

% count the number of noisy trials in each of sensors
for ixtr=1:Ntrial
  waitbar(ixtr/Ntrial); 
  
  tmp = abs(bexp(:,:,ixtr));  
  Z0 = median(tmp(:));
  for ixch=1:Nchannel
    ratio = max(tmp(ixch,:))/median(tmp(ixch,:));
    ratio_all = [ratio_all; ratio];
    if max(tmp(ixch,:)) > parm.meg_coef*Z0 & ratio > parm.th_meg
      ixcount{ixch} = [ixcount{ixch}; ixtr];
    end
  end
  
  for ii=1:15; vb_disp_nonl(sprintf('\b')); end
  vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
  prg = prg+1;
end

clear tmp
close(h); 
vb_disp_nonl(sprintf('\n'));

% confirm sensor rejection
if parm.verify==ON, hh = figure; hold on; end
ntrial_th = parm.Ntrial_th;
for n=1:Nchannel
  if length(ixcount{n})>=ntrial_th % fixed 06-12-19 D.K.
    % Calculate ratio of max to median for all trials
    r = [];
    for c=1:length(ixcount{n})
      tmp = abs(bexp(n,:,ixcount{n}(c)));
      r = [r; max(tmp)/median(tmp)];
    end
    [r,ii] = sort(r); r = flipud(r); ii = flipud(ii); %fixed at 06-12-19 by D.K.
    
    if parm.verify==ON, 
      figure(hh);clf(hh);hold on;
      for trcount=1:ntrial_th
	subplot(ntrial_th,1,trcount); hold on;
	plot(bexp(:,:,ixcount{n}(ii(end-ntrial_th+trcount)))',':');
	xlim([1 sampleTime]);
	plot(bexp(n,:,ixcount{n}(ii(end-ntrial_th+trcount)))','LineWidth',4);
	hold off;
	title(['Trial ' num2str(trNo(ixcount{n}(ii(end-ntrial_th+trcount)))) ...
	       ' max/med:' sprintf('%2.1f',r(end-ntrial_th+trcount))]);
      end
      button = questdlg(['Reject sensor ' num2str(chNo(n)) '?'],...
			'Sensor rejection','Yes','No','Yes');
    else
      button = 'Yes'; 
    end
    
    if strcmp(button,'Yes')
      rejectChannel = [rejectChannel; chNo(n)];
      fprintf('---- channel %d was rejected.\n',chNo(n));
    end
  end
end
rejectChannel = sort(rejectChannel);

if parm.verify==ON, close(hh); end; 
clear tmp;

% modified by DK at 070417
if isempty(vb_setdiff2(MEGinfo.MEGch_id, rejectChannel));
  error('All sensors were rejected. Please check parameters.');
end
[MEGinfo,bexp,pick,Qpick,CoilWeight] = ...
    vb_channel_rejection(MEGinfo,bexp,pick,Qpick,rejectChannel,CoilWeight);
MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight);
fprintf('--- done.\n');
return


%
% Reject noisy trials
%
function [MEGinfo,MEGdata] = reject_noisy_trials(parm,MEGinfo,MEGdata)
fprintf('--- rejecting noisy trials...\n');
[bexp,eog,pick,Qpick,CoilWeight]= get_meg_data(MEGdata);
%fixed by D.K. at 06-12-19
%fixed by D.K. at 07-04-17
chNo = MEGinfo.MEGch_id;
trNo = MEGinfo.TrialNumber;
Nchannel = MEGinfo.Nchannel;
Ntrial = MEGinfo.Nrepeat;
sampleTime = MEGinfo.Nsample;

rejectTrial = [];

ratio_max = zeros(Ntrial,1);
%ix = setdiff(1:Nsensor,ix_sensor_rej);
h       = waitbar(0,'Check noisy trial'); 
prg     = 0;
prg_all = Ntrial;
vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));

for ixtr = 1:Ntrial
  waitbar(ixtr/Ntrial); 
  
  tmp = abs(bexp(:,:,ixtr));
  Z0 = median(tmp(:));
  for ixch=1:Nchannel
    ratio = max(tmp(ixch,:))/median(tmp(ixch,:));
    if max(tmp(ixch,:)) > parm.meg_coef*Z0 
      ratio_max(ixtr) = max([ratio_max(ixtr) ratio]);
    end
  end
  
  for ii=1:15; vb_disp_nonl(sprintf('\b')); end
  vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
  prg = prg+1;
end

clear tmp;
close(h); 
vb_disp_nonl(sprintf('\n'));

if parm.verify==ON, hh=figure; end;
for ixtr = 1:Ntrial
  if ratio_max(ixtr)>parm.th_meg | ~isempty(find(parm.rm_trial==trNo(ixtr))), 
    if parm.verify==ON,
      figure(hh);clf(hh);hold on;
      plot(bexp(:,:,ixtr)'); 
      xlim([1 sampleTime]);
      title(['Trial ' num2str(trNo(ixtr)) ...
	     sprintf(' max/med:%2.1f',ratio_max(ixtr))]);
      button = questdlg(['Reject trial ' num2str(trNo(ixtr)) '?'],...
			'Trial rejection','Yes','No','Yes');
    else
      button = 'Yes';
    end
    
    if strcmp(button,'Yes')
      rejectTrial = [rejectTrial; trNo(ixtr)];
      fprintf('---- trial %d was rejected.\n',trNo(ixtr));
    end
  end
end
if parm.verify == ON, close(hh); end;

% modified by DK at 070417
if isempty(vb_setdiff2(MEGinfo.TrialNumber, rejectTrial)) 
  error('All trials were rejected. Please check parameters.');
end
[MEGinfo,bexp,eog] = vb_trial_rejection(MEGinfo,bexp,eog,rejectTrial);
MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight);
fprintf('--- done.\n');
return


%
% Trial division
%
function save_division(parm, megfile, MEGinfo, MEGdata, divNo)
fprintf('--- now saving...\n');
[bexp_org,eog_org,pick_org,Qpick,CoilWeight]=get_meg_data(MEGdata);
chNo = MEGinfo.MEGch_name;%fixed by D.K. at 06-12-19
trNo = MEGinfo.TrialNumber;
Nchannel = MEGinfo.Nchannel;
Ntrial = MEGinfo.Nrepeat;
MEGinfo.TrialNumber_org = trNo;

%reject rejected trials in divNo
divNo = divNo(trNo);

division = unique(divNo); 
Ncond = length(division);
disp(['---- ' num2str(length(division)) ' conditions in division file']);

for div=1:length(division)
  ixtr = find(divNo==division(div));

  %divide trials
  bexp = bexp_org(:,:,ixtr);
  if ~isempty(eog_org),eog = eog_org(:,:,ixtr);else eog=[]; end;
  
  MEGinfo.Nrepeat = length(ixtr);
  MEGinfo.TrialNumber = trNo(ixtr);

  disp(['---- ' num2str(length(ixtr)) ' trials were remained ' ...
	'in condition ' num2str(division(div))]);

  if parm.verify==ON, 
    answer = inputdlg(['Name for condition ' num2str(division(div)) ...
		       '?'],['Condition ' num2str(division(div))]);
    if isempty(answer) % if "cancel" button was pushed.
      error('Process was aborted.');
    end
  else
    answer{1} = ['cond' num2str(division(div))];
  end
  
  filename = strrep(megfile,'.meg.mat',['_' answer{1} '.meg.mat']);
  disp(['    saved as ' filename]);
  
  % Save MEG data 
  switch lower(MEGinfo.device)
   case 'sbi'
    pick = cat( 2, pick_org(1:Nchannel,:), pick_org([Nchannel+1:Nchannel*2],:)  );
    pick = cat( 2, pick,                   Qpick(1:Nchannel,:) );
    vb_fsave(filename,'pick','bexp','MEGinfo','eog'); 
   case 'yokogawa'
    eeg = eog;
    pick = pick_org;
    vb_fsave(filename,'pick','Qpick','bexp','MEGinfo','eeg'); 
   otherwise
    pick = cat( 2, pick_org(1:Nchannel,:), pick_org([Nchannel+1:Nchannel*2],:)  );
    pick = cat( 2, pick,                   Qpick(1:Nchannel,:) );
    vb_fsave(filename,'pick','bexp','MEGinfo','eog'); 
  end
fprintf('--- done.\n');
end

function MEGdata = set_meg_data(bexp,eog,pick,Qpick,CoilWeight)
MEGdata.bexp = bexp;
MEGdata.eog = eog;
MEGdata.pick = pick;
MEGdata.Qpick = Qpick;
MEGdata.CoilWeight = CoilWeight;
return

function [bexp,eog,pick,Qpick,CoilWeight]=get_meg_data(MEGdata)
bexp = MEGdata.bexp;
eog = MEGdata.eog;
pick = MEGdata.pick;
Qpick = MEGdata.Qpick;
CoilWeight = MEGdata.CoilWeight;
return
