function vb_job_brain_add_sphcoord(varargin)
% Import FreeSurfer's sphere coordinate into cortical surface file. 
% (VBMEG public function)
%
% FreeSurfer's cortical surface model has a (coregistered) sphere
% coordinate values at each vertex. This function is used to import the
% sphere coordinate values into an existing cortical surface file
% (.brain.mat). Struct variables 'Vall' will be added to the cortical
% surface model file (see [output]). 
%
% [syntax]
% vb_job_brain_add_sphcoord(brain_parm)
% vb_job_brain_add_sphcoord(proj_root,brain_parm)
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% brain_parm: <<struct>> Parameters for import. 
% --- fields of brain_parm
%  brain_file: <<string>> Cortical surface model file
%              (.brain.mat). Struct variable 'Vall' will be added to this
%              file (see [output]). 
%  brain_sphL: <<string>> Sphere surface file (.sphere.reg(.asc))
%              corresponding to the left hemisphere of the source
%              model. 
%  brain_sphR: <<string>> Sphere surface file (.sphere.reg(.asc))
%              corresponding to the right hemisphere of the source
%              model.
%  key       : <<string>> Arbitrary ID of the coordinate value
%              (e.g., 'Sphere_coreg'). It is used to load coordinate
%              values by vb_load_cortex. 
% ---
%
% [output]
% Struct variable 'Vall' will be added to brain_parm.brain_file. 
% --- fields of Vall
%  V  : <<cell of matrix>> Spherical coordinate values. 
%  key: <<cell of string>> Used to get spherical coordinate values by
%       vb_load_cortex. 
% ---
%
% [example]
% You can add sphere coordinate values to existing cortical surface model
% file by the following code: 
%
% >> brain_parm.brain_file = 'existing_brain_file'; % .brain.mat
% >> brain_parm.brain_sphL = 'lh.sphere.reg.asc';
% >> brain_parm.brain_sphR = 'rh.sphere.reg.asc';
% >> brain_parm.key        = 'Sphere_coreg'; % arbitrary string available
% >> proj_root = 'your_project_root_directory';
% >> vb_job_brain_add_sphcoord(proj_root,brain_parm);
%
% You can get the spherical coordinate values of the cortical surface
% model by the following code:
%
% >> V = vb_load_cortex('your_brain_file','Sphere_coreg');
% 
% [see also]
% @see vb_load_cortex.m
%
% [history]
% 2010-11-18 Taku Yoshioka
% 2010-12-22 taku-y
%  [minor] Bug fix: 'proj_root' attached with directroies specified by
%  'brain_parm.brain_file'. 
% 2015-11-26 Masa-aki Sato
%  [change] original spherical coordinate is saved to Vall
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% check input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if length(varargin) == 1
  proj_root = [];
  brain_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  brain_parm = varargin{2};
else
  error('Error: invalid usage of vb_job_brain_sphcoord.');
end;

if ~isempty(proj_root), 
  proj_root = vb_rm_trailing_slash(proj_root);
end;

%
% Make absolute path
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
brain_parm_abs = brain_parm;

if isempty(proj_root)
	brain_parm_abs.brain_file = [brain_parm.brain_file];
else
	brain_parm_abs.brain_file = [proj_root filesep brain_parm.brain_file];
end
%
% Verbose level
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
global vbmeg_inst
verbose_const = vb_define_verbose; 

if isempty(vbmeg_inst) | ~isfield(vbmeg_inst,'verbose_level'), 
  verbose_level = verbose_const.VERBOSE_LEVEL_NOTICE;
else
  verbose_level = vbmeg_inst.verbose_level;
end;

%
% Load coordinate values
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[tmp1,tmp2,ix_V_org] = vb_load_cortex_info(brain_parm_abs.brain_file);

VL0 = vb_fs_load_surface(brain_parm_abs.brain_sphL);
VR0 = vb_fs_load_surface(brain_parm_abs.brain_sphR);

%  [change] original spherical coordinate is saved to Vall
V0  = [VL0; VR0];

%  spherical coordinate corresponding to cortical vertex
VL  = VL0(ix_V_org.Left,:);
VR  = VR0(ix_V_org.Right-size(VL0,1),:);

V   = [VL; VR];
%
% Add coordinate values
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
S = load(brain_parm_abs.brain_file);
if isfield(S,'Vall'), 
  Vall = S.Vall;
else
  Vall.V = cell(0);
  Vall.key = cell(0);
end;

flag = true;
flag_org = true;

for i=1:length(Vall.key)
  if strcmp(Vall.key{i},brain_parm_abs.key), 
     flag = false;
     Vall.V{i} = V;
%    if verbose_level>=verbose_const.VERBOSE_LEVEL_NOTICE, 
%      msg = ['Brain coordinate ID ''' brain_parm_abs.key ...
%             ''' is already used. Overwrite ?'];
%      str = questdlg(msg,'Brain coordinate registration','Yes', ...
%		     'Cancel','Cancel');
%      if strcmp(str,'Yes')
%	Vall.V{i} = V;
%	h = msgbox(['Brain coordinate ID ''' brain_parm_abs.key ...
%		    ''' was overwritten.']);
%	uiwait(h);
%      end;
%    else
      vb_disp(['Brain coordinate ID ''' brain_parm_abs.key ...
               ''' is already ' ...
               'used, so it was overwritten with new data. '], ...
              verbose_const.VERBOSE_LEVEL_WARNING); 
%      Vall.V{i} = V;
%    end;
  end;
  
  if strcmp(Vall.key{i},'orig_sphere'), 
    flag_org = false;
  end
end;

if flag_org, 
  Vall.V{length(Vall.key)+1} = V0;
  Vall.key{length(Vall.key)+1} = 'orig_sphere';
end;

if flag, 
  Vall.V{length(Vall.key)+1} = V;
  Vall.key{length(Vall.key)+1} = brain_parm_abs.key;
end;

vb_save(brain_parm_abs.brain_file,'Vall');

%
% project_file save
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% proj_file = get_project_filename;
% if isempty(proj_file)
%   return;
% end;
% 
% project_file_mgr('load', proj_file);
% project_file_mgr('add', 'brain_parm', brain_parm);
% 
% return;
