function vb_select_area(parm,plot_parm)
% Create area data on brain model.
%
% [syntax]
% vb_select_area(parm,plot_parm)
% 
% [input]
% parm     : <<struct>>
% --- fields of parm
%  brainfile: <<string>>
%  areafile : <<string>>
%  actfile  : <<string>>
% ---
% plot_parm: <<struct>>
%
% [example]
% >> parm.brainfile = './MS/data/MS.brain.mat';
% >> parm.areafile = './MS/data/MS.area.mat'; 
% >> parm.actfile = './MS/data/MS.act.mat';
% >> parm.mode = 1; % inflate display
% >> plot_parm = vb_set_plot_parm;
% >> vb_select_area(parm,plot_parm);
%
% [inner functions]
% init_global_variables
% create_display_panel
% create_select_panel
% create_peak_panel
% create_frame
% plot_brain_inner
% create_axes
% create_close_button
% update_listbox_area
% init_stack
%
% [callback functions]
% callback_return_points
% callback_select_point3d
% callback_radius
% callback_undo
% callback_return_points
% callback_remove_area
% change_modality 
% select_act 
% select_bayesfile 
% vb_select_area
% cancel_all_area
% change_label
% change_transparent
% select_peak_area
% search_peak
% clear_estimation_result
% change_exclude_radius
% select_disp_area
% change_J_ratio
% print_figure
% change_plot_parm 
% change_timewindow
%
% clear_peak
% close_window
%
% [global variables in select_area_variables]
% brainfile
% areafile
% inflatefile
% actfile
% plot_parm
% Radius
% ix_peak1
% ix_peak2
% act_keys
% area_keys
% disp_area
% V
% V0
% ix_disp_area
% bayesfile
%
% F : Inverse filter of the result file
% ix: Indices of the dipoles associated with the inverse filter
% J : Current distribution on all of the dipoles 
% B : MEG time course
% 
% [GUI handles]
% rb1 (activityfile/bayesfile)
% lb1 (activity map list)
% lb2 (area list)
% lb3 (display area)
% pb1 (select bayesfile)
% pb2 (cancel all area)
% pb3 (peak1 search)
% pb4 (peak1 select area)
% pb5 (peak2 search)
% pb6 (peak2 select area)
% pb7 (clear estimation result)
% pb8 (select time window)
% pb9 (remove area data)
% tx1 (bayesfile)
% tx2 (peak1 index)
% tx3 (peak2 index)
% tx4 (peak distance)
% ax1 (brain model)
% ed1 (radius)
% ed2 (excluded radius)
% cb2 (transparent)
% jmin_edit   (Jmin_ratio)
% jmin_slider (Jmin_ratio)
% jmax_edit   (Jmax_ratio)
% jmax_slider (Jmax_ratio)
%
% [history]
% 2004-11-11 Taku Yoshioka
% 2005-08-19 Taku Yoshioka (for ver 0.3)
% 2010-08-26 Taku Yoshioka
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% If 'parm' is not a struct, it is regarded as a callback function. 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isstruct(parm)
  eval(parm);
  return;
end;

%
% Initialize global variables
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear global select_area_variables
global select_area_variables;
init_global_variables(parm,plot_parm);

%
% Create GUI
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure;
select_area_variables.ax1 = create_axes([0.05 0.45 0.45 0.5]);
create_display_panel([0.55 0.38 0.35 0.52]);
create_select_panel([0.55 0.1 0.35 0.28]);
create_peak_panel([0.1 0.15 0.45 0.23]);
create_close_button([0.1 0.1 0.45 0.05]);
plot_brain_inner;

%
% Display brain model
% 2010-08-26 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function plot_brain_inner(angle)

global select_area_variables

plot_parm = select_area_variables.plot_parm;

% 3D view angle
if nargin==0
  axes(select_area_variables.ax1);
  [az,el] = view; 
  angle = [az el];
  cla;
end

actfile = select_area_variables.actfile;
lb1 = select_area_variables.lb1;
act_keys = select_area_variables.act_keys;

% which type of the data is selected ?
if get(select_area_variables.rb1(1),'Value')
  n = get(lb1,'Value');
  if ~isempty(n)
    act = vb_get_act(actfile,act_keys(n(1)));
    xxP = act.xxP;
    plot_parm.J = xxP;
  end
elseif get(select_area_variables.rb1(2),'Value')
  plot_parm.J = select_area_variables.J_map;
end

% Display area
plot_parm.area.key = select_area_variables.disp_area;
plot_parm.area.clr = {'b','g','r','c','m','y'};

% Transparency
cb2 = select_area_variables.cb2;
if get(cb2,'Value')
  plot_parm.area.transparent = 1;
else
  plot_parm.area.transparent = 0;
end

% Plotting parameters
%plot_parm.brainfile = select_area_variables.brainfile;
plot_parm.areafile = select_area_variables.areafile;
plot_parm.index = select_area_variables.ix_disp_area;
brain_file = select_area_variables.brainfile;

%plot_parm.wmin = plot_parm.wmin * plot_parm.Jmin_ratio;
%plot_parm.wmax = plot_parm.wmax * plot_parm.Jmax_ratio;
wmin = plot_parm.wmin * plot_parm.Jmin_ratio;
wmax = plot_parm.wmax * plot_parm.Jmax_ratio;
J = plot_parm.J;

% Plot
switch select_area_variables.mode
 case 0,
  % vb_plot_brain(plot_parm);
  [V,F] = vb_load_cortex(brain_file);
  h = vb_plot_cortex(plot_parm,V,F,[],J,wmin,wmax);
  view([0 0]); camlight headlight;
  view([0 180]); camlight headlight;
  
 case 1,
  % plot_parm.sulcus = 1;
  % vb_plot_brain_inflate(plot_parm);
  [V,F,xx,inf_C] = vb_load_cortex(brain_file,'Inflate'); % 'xx' not used
  h = vb_plot_cortex(plot_parm,V,F,inf_C,J,wmin,wmax);
end;

view(angle);
axis equal;
axis off;
if ~isempty(h) & ishandle(h) & strcmp(get(h,'Type'),'light'), 
  camlight(h,'headlight');
end;

% Plot peaks
V = select_area_variables.V;
ix_peak1 = select_area_variables.ix_peak1;
ix_peak2 = select_area_variables.ix_peak2;
ix_disp_area = select_area_variables.ix_disp_area;

if ~isempty(ix_peak1)
  if ~isempty(find(ix_disp_area==ix_peak1)) | ...
	isempty(ix_disp_area)
    plot3(V(ix_peak1,1),V(ix_peak1,2),V(ix_peak1,3),'rx',...
	  'MarkerSize',plot_parm.ms,'LineWidth',plot_parm.lw,...
	  'EraseMode','xor');
  end
end

if ~isempty(ix_peak2) 
  if ~isempty(find(ix_disp_area==ix_peak2)) | ...
	isempty(ix_disp_area)
    plot3(V(ix_peak2,1),V(ix_peak2,2),V(ix_peak2,3),'bx',...
	  'MarkerSize',plot_parm.ms,'LineWidth',plot_parm.lw,...
	  'EraseMode','xor');
  end
end

%
% Callback function: select vertices on the plotted model. 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function callback_select_point3d

global select_area_variables;

plot_parm = select_area_variables.plot_parm;
V = select_area_variables.V;
nextDD = select_area_variables.nextDD;
nextIX = select_area_variables.nextIX;
Radius = select_area_variables.Radius;
stack_points = select_area_variables.stack_points;
ix_stack_points = select_area_variables.ix_stack_points;

zoom off;
rotate3d off;

% Disable buttons
h = get(gcf,'Children');
for i = 1:length(h)
  if strcmp(get(h(i),'Type'),'uicontrol')
    set(h(i),'Enable','off');
  end
end

% Vertex selection
P = [];
h_tmp = [];
while 1
  k = waitforbuttonpress;
  tmp = (select3d)';
  if k ~= 0 | length(tmp) == 0
    break;
  end
  P = [P; tmp];
  h_tmp = [h_tmp; plot3(tmp(1),tmp(2),tmp(3),'kx')];
end

% Clear plots
for i = 1:length(h_tmp)
  delete(h_tmp(i));
end

%
% 򤵤줿ǥγʻѴ
% Vϥե졼ȥǥĺɸP򤵤줿ѥå
% ͱƤȤκɸ
% ix     : Extracted point index in the whole brain
% h_plot : ixбץåȤΥեåϥɥ
%
ix = [];
for i = 1:size(P,1)
  tmp = repmat(P(i,:), [length(V) 1]);
  [tmp tmp2] = min(sum((tmp - V).^2, 2));
  ix = [ix; tmp2];
end;
ix

% Extract area with specified radius
ix = unique(ix);
tmp = [];
for i = 1:length(ix)
  ix_tmp = find(nextDD{ix(i)} <= Radius);
  tmp = [tmp; nextIX{ix(i)}(ix_tmp)];
end
ix = unique([ix; tmp]);

% Plot selected vertices
h_plot = plot3(V(ix,1), V(ix,2), V(ix,3), ... 
	       'r.', 'MarkerSize',plot_parm.ms);

% Take indices and graphics handles of the selected vertices to
% the stack
tmp = cell(2,1);
tmp{1} = ix;
tmp{2} = h_plot;
ix_stack_points = ix_stack_points + 1;
stack_points{ix_stack_points} = tmp;

% Enable buttons
for i = 1:length(h)
  if strcmp(get(h(i),'Type'),'uicontrol')
    set(h(i),'Enable','on');
  end
end

if get(select_area_variables.rb1(1),'Value')
  set(select_area_variables.lb1,'Enable','on');
  set(select_area_variables.pb1,'Enable','off');
  set(select_area_variables.pb7,'Enable','off');
  set(select_area_variables.tx1,'Enable','off');
else
  set(select_area_variables.lb1,'Enable','off');
  set(select_area_variables.pb1,'Enable','on');
  set(select_area_variables.pb7,'Enable','on');
  set(select_area_variables.tx1,'Enable','inactive');
end

select_area_variables.stack_points = stack_points;
select_area_variables.ix_stack_points = ix_stack_points;

return;

%
% Callback function: select radius
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function callback_select_radius

global select_area_variables;

select_area_variables.Radius = ...
    str2num(get(select_area_variables.ed1,'String'))*1e-3;

return;

%
% Undo
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function callback_undo

global select_area_variables;

ix_stack_points = select_area_variables.ix_stack_points;
stack_points = select_area_variables.stack_points;

if ix_stack_points > 0
  tmp = stack_points{ix_stack_points};
  
  % Delete points on axis
  h = tmp{2};
  for i = 1:length(h)
    if length(h(i)) ~= 0
      delete(h(i));
    end
  end
  
  ix_stack_points = ix_stack_points - 1;
end

select_area_variables.ix_stack_points = ix_stack_points;
select_area_variables.stack_points = stack_points;

tmp1 = [];
tmp2 = [];

return;

%
% Area registration
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function callback_return_points

global select_area_variables;

ix_stack_points = select_area_variables.ix_stack_points;
stack_points = select_area_variables.stack_points;
areafile = select_area_variables.areafile;

Iextract = [];
Iextract_seq = cell(ix_stack_points,1);

for i = 1:ix_stack_points
  tmp = stack_points{i};
  Iextract = [Iextract; tmp{1}];
  Iextract_seq{i} = unique(tmp{1});
end

fprintf('--- %d vertices were selected.\n',...
	length(unique(Iextract)));
area_key = inputdlg('Input ID for the selected region: ');

NewArea.Iextract = unique(Iextract);
NewArea.key = area_key{1};
NewArea.Iextract_seq = Iextract_seq;

vb_add_area(areafile,NewArea);

init_stack;
plot_brain_inner;
update_listbox_area;

return;

%
% Subfunction: initialize global variables
% 2004-03-14 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function init_global_variables(parm,plot_parm)

global select_area_variables;

plot_parm.Jmax_ratio = 1.0;
plot_parm.Jmin_ratio = 1.0;

select_area_variables.plot_parm = plot_parm;
select_area_variables.brainfile = parm.brainfile;
select_area_variables.areafile = parm.areafile;
select_area_variables.actfile = parm.actfile;
select_area_variables.mode = parm.mode;
select_area_variables.bayesfile = '';

brainfile = parm.brainfile;
actfile = parm.actfile;
areafile = parm.areafile;

% brain vertex position
V = vb_load_cortex(brainfile);
select_area_variables.V0 = V;

if parm.mode == 1, % inflate model
  V = vb_load_cortex(brainfile,'Inflate');
end

select_area_variables.V = V; 

[nextDD,nextIX] = vb_load_cortex_neighbour(brainfile); 
select_area_variables.nextDD = nextDD;
select_area_variables.nextIX = nextIX;

% Stack of selected area
init_stack;

% Radius of selected area
select_area_variables.Radius = 0;

% Activity map/Area keyset
select_area_variables.act_keys = vb_get_keyset_act(actfile);
select_area_variables.area_keys = vb_get_keyset_area(areafile);

% Selected area for display
select_area_variables.disp_area = {};

% Selected peak index
select_area_variables.ix_peak1 = [];
select_area_variables.ix_peak2 = [];

% Display region
select_area_variables.ix_disp_area = [];

% Current distribution
select_area_variables.J_map = [];

return;

%
% Subfunction: create frame
% 2004-03-14 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function create_frame(pos)

uicontrol('Style','frame','Units','normalized',...
	  'Position',pos);

return;

%
% Subfunction: create display panel
% 2004-03-14 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function create_display_panel(pos0)

global select_area_variables

create_frame(pos0);

m = 0.03*pos0(3);% margin
x = pos0(1)+m;   % origin (x)
y = pos0(2)+m;   % origin (y)
w = pos0(3)-2*m; % width
h = pos0(4)-2*m; % height

% Zoom in
pos = [x y+0.9*h 0.5*w 0.1*h];
uicontrol('Style','pushbutton','String','Zoom in/out',...
	  'Units','normalized','Position',pos,...
	  'Callback','zoom on');

% Rotation
pos = [x y+0.8*h 0.5*w 0.1*h];
uicontrol('Style','pushbutton','String','Rotation',...
	  'Units','normalized','Position',pos,...
	  'Callback','rotate3d');
%
% rb1: Radio button (1):Activity map (2):Estimation result
%

% Activity map radio button
pos = [x+0.5*w y+0.9*h 0.5*w 0.1*h]; 
rb1(1) = ...
    uicontrol('Style','radiobutton','String','Activity map',...
	      'Units','normalized','Position',pos,...
	      'Value',0,'Callback',...
	      'vb_select_area(''change_modality(1)'')');
set(rb1(1),'Value',1); % Initial value

% Activity map list
pos = [x+0.5*w y+0.5*h 0.5*w 0.4*h];
str = [];
keys = select_area_variables.act_keys;
for i = 1:length(keys)
  str = [str keys{i}];
  if i ~= length(keys)
    str = [str '|'];
  end
end
lb1 = ...
    uicontrol('Style','listbox','String',str,...
	      'Units','normalized','Position',pos,...
	      'Callback','vb_select_area(''select_act'')',...
	      'Min',1,'Max',30);
set(lb1,'Value',[1]); % Initial value

% VB estimation result
pos = [x+0.5*w y+0.4*h 0.5*w 0.1*h];
rb1(2) = ...
    uicontrol('Style','radiobutton','String','VB result',...
	      'Units','normalized','Position',pos,...
	      'Value',0,'Callback',...
	      'vb_select_area(''change_modality(2)'')');
set(rb1(2),'Value',0); % Initial value

% VB result filename
pos = [x+0.5*w y+0.3*h 0.5*w 0.1*h];
tx1 = uicontrol('Style','edit','String','','Enable','off',...
		'Units','normalized','Position',pos);

% Load button
pos = [x+0.5*w y+0.2*h 0.5*w 0.1*h];
pb1 = uicontrol('Style','pushbutton','String','Load result',...
		'Units','normalized','Position',pos,...
		'Enable','off','Callback',...
		'vb_select_area(''select_bayesfile'')');

% Time window button
pos = [x+0.5*w y+0.1*h 0.5*w 0.1*h];
pb8 = uicontrol('Style','pushbutton','String','Time window',...
		'Units','normalized','Position',pos,...
		'Enable','off','Callback',...
		'vb_select_area(''change_timewindow'')');

% Clear estimation result
pos = [x+0.5*w y 0.5*w 0.1*h];
pb7 = uicontrol('Style','pushbutton','String','Clear result',...
		'Units','normalized','Position',pos,...
		'Enable','off','Callback',...
		'vb_select_area(''clear_estimation_result'')');

% Display area
pos = [x y+0.7*h 0.5*w 0.1*h];
uicontrol('Style','text','String','Display',...
	  'Units','normalized','Position',pos);

str = [];
keys = select_area_variables.area_keys;
for i = 1:length(keys)
  str = [str keys{i}];
  if i ~= length(keys)
    str = [str '|'];
  end
end

pos = [x y+0.3*h 0.5*w 0.4*h];
lb3 = uicontrol('Style','listbox','String',str,...
		'Units','normalized','Position',pos,...
		'Callback','vb_select_area(''select_disp_area'')',...
		'Min',1,'Max',1);
set(lb3,'Value',length(keys)); % Initial value

% J Max / Min Ratio
Jmax_ratio = select_area_variables.plot_parm.Jmax_ratio;
Jmin_ratio = select_area_variables.plot_parm.Jmin_ratio;
pos = [x y+0.25*h 0.20*w 0.05*h];
uicontrol('Style', 'text', 'String', 'JMax ratio',...
		'Units', 'normalized', 'Position', pos, 'HorizontalAlignment', 'left');
pos = [x+0.15*w y+0.25*h 0.1*w 0.05*h];
jmax_edit = uicontrol('Style', 'edit', 'String', num2str(Jmax_ratio), ...
          'Units', 'normalized', 'Position', pos,...
          'Callback', 'vb_select_area(''change_J_ratio'')', 'Tag', 'jmax_edit');
    
pos = [x+0.25*w y+0.25*h 0.25*w 0.05*h];
jmax_slider = uicontrol('Style', 'slider', 'SliderStep', [0.001, 0.01],...
          'Units', 'normalized', 'Position', pos, ...
          'Callback', 'vb_select_area(''change_J_ratio'')', 'Tag', 'jmax_slider',...
          'Max', 1, 'Min', 0, 'Value', Jmax_ratio);

pos = [x y+0.2*h 0.20*w 0.05*h];
uicontrol('Style', 'text', 'String', 'JMin ratio',...
		'Units', 'normalized', 'Position', pos,...
        'HorizontalAlignment', 'left');
pos = [x+0.15*w y+0.2*h 0.1*w 0.05*h];
jmin_edit = uicontrol('Style', 'edit', 'String', num2str(Jmin_ratio), ...
          'Units', 'normalized', 'Position', pos,...
          'Callback', 'vb_select_area(''change_J_ratio'')', 'Tag', 'jmin_edit');

pos = [x+0.25*w y+0.2*h 0.25*w 0.05*h];
jmin_slider = uicontrol('Style', 'slider', 'SliderStep', [0.001, 0.01],...
          'Units', 'normalized', 'Position', pos,...
          'Callback', 'vb_select_area(''change_J_ratio'')', 'Tag', 'jmin_slider',...
          'Max', 1, 'Min', 0, 'Value', Jmin_ratio);

% plot parameter
pos = [x y+0.1*h 0.5*w 0.1*h];
uicontrol('Style','pushbutton','String','Plot parm.',...
	  'Units','normalized','Position',pos,...
	  'Callback','vb_select_area(''change_plot_parm'')');

% print figure
pos = [x y 0.5*w 0.1*h];
uicontrol('Style','pushbutton','String','Print',...
	  'Units','normalized','Position',pos,...
	  'Callback','vb_select_area(''print_figure'')');

%
% GUI handles
%
select_area_variables.rb1 = rb1;
select_area_variables.lb1 = lb1;
select_area_variables.tx1 = tx1;
select_area_variables.pb1 = pb1;
select_area_variables.pb7 = pb7;
select_area_variables.lb3 = lb3;
select_area_variables.pb8 = pb8;
select_area_variables.jmin_edit   = jmin_edit;
select_area_variables.jmin_slider = jmin_slider;
select_area_variables.jmax_edit   = jmax_edit;
select_area_variables.jmax_slider = jmax_slider;

return;

%
% Change display activity map and estimation result
% 2004-03-14 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function change_modality(c)

global select_area_variables;

switch c
 case 1, % display activity map
  set(select_area_variables.lb1,'Enable','on');
  set(select_area_variables.rb1(1),'Value',1);
  set(select_area_variables.pb1,'Enable','off');
  set(select_area_variables.pb7,'Enable','off');
  set(select_area_variables.tx1,'Enable','off');
  set(select_area_variables.rb1(2),'Value',0);
  set(select_area_variables.pb8,'Enable','off');
  
 case 2, % display estimation result
  set(select_area_variables.lb1,'Enable','off');
  set(select_area_variables.rb1(1),'Value',0);
  set(select_area_variables.pb1,'Enable','on');
  set(select_area_variables.pb7,'Enable','on');
  set(select_area_variables.tx1,'Enable','inactive');
  set(select_area_variables.rb1(2),'Value',1);
  if isempty(select_area_variables.bayesfile)
    set(select_area_variables.pb8,'Enable','off');
  else
    set(select_area_variables.pb8,'Enable','on');
  end
end

plot_brain_inner;

return;

%
% Create axes
% 2004-03-12 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function h = create_axes(pos)

axes;
set(gca,'Position',pos);
h = gca;

return;

%
% Select bayes file
% 2004-03-14 Taku Yoshioka
% change by M. Sato 
% load current file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function select_bayesfile

global select_area_variables;

[pname,fname] = vb_file_select({'.curr.mat'});
if isempty(fname), return; end

fname = fname{1};
resultfile = [pname filesep fname];
load(resultfile,'Jact','ix_act','Tstart','Tend','Lact')
select_area_variables.bayesfile = resultfile; 
I = size(select_area_variables.V,1);

[Nj, T] = size(Jact);
JJ = sqrt(sum(reshape(Jact.^2 , [Nj/Lact Lact*T]), 2)/(Lact*T));

% Set global variables
select_area_variables.ix = ix_act;
select_area_variables.J_map = zeros(I,1);
select_area_variables.J_map(ix_act) = JJ;

% Update GUI
set(select_area_variables.tx1,'String',fname);
set(select_area_variables.pb8,'Enable','on');

% Plot brain
plot_brain_inner;

return;

%
% Select activity map
% 2004-03-14 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function select_act

plot_brain_inner;

return;

%
% Create select panel
% 2004-03-14 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function create_select_panel(pos0)

global select_area_variables

create_frame(pos0);

m = 0.04*pos0(4);% margin
x = pos0(1)+m;   % origin (x)
y = pos0(2)+m;   % origin (y)
w = pos0(3)-2*m; % width
h = pos0(4)-2*m; % height

% Select area
pos = [x+0.5*w y+0.8*h 0.5*w 0.2*h];
uicontrol('Style','pushbutton','String','Select area',...
	  'Units','normalized','Position',pos,...
	  'Callback','vb_select_area(''callback_select_point3d'')');

% Radius
pos = [x y+0.8*h 0.3*w 0.2*h];
uicontrol('Style','text','String','Radius [mm]',...
	  'Units','normalized','Position',pos);

pos = [x+0.3*w y+0.8*h 0.2*w 0.2*h];
ed1 = uicontrol('Style','edit','String','0',...
		'Units','normalized','Position',pos,...
		'Callback', ...
		'vb_select_area(''callback_select_radius'')');

% Undo
pos = [x+0.5*w y+0.6*h 0.5*w 0.2*h];
uicontrol('Style','pushbutton','String','Undo ',...
	  'Units','normalized','Position',pos,...
	  'Callback','vb_select_area(''callback_undo'')');

% Registration
pos = [x+0.5*w y+0.4*h 0.5*w 0.2*h];
uicontrol('Style','pushbutton','String','Registration',...
	  'Units','normalized','Position',pos,...
	  'Callback',...
	  'vb_select_area(''callback_return_points;'')');

% Area list
pos = [x y+0.2*h 0.5*w 0.6*h];
str = [];
keys = select_area_variables.area_keys;
for i = 1:length(keys)
  str = [str keys{i}];
  if i ~= length(keys)
    str = [str '|'];
  end
end
lb2 = uicontrol('Style','listbox','String',str,...
		'Units','normalized','Position',pos,...
		'Callback','vb_select_area(''select_area_inner'')',...
		'Min',1,'Max',30);
set(lb2,'Value',[]); % Initial value

% Cancel all area
pos = [x y 0.5*w 0.2*h];
pb2 = uicontrol('Style','pushbutton','String','Cancel all',...
		'Units','normalized','Position',pos,...
		'Enable','on','Callback',...
		'vb_select_area(''cancel_all_area'')');

% Transparent
pos = [x+0.5*w y+0.2*h 0.5*w 0.2*h];
cb2 = uicontrol('Style','checkbox','String','Transparent',...
		'Units','normalized','Position',pos,...
		'Value',0,'Callback',...
		'vb_select_area(''plot_brain_inner'')');
set(cb2,'Value',0); % Initial value

% Remove area data
pos = [x+0.5*w y 0.5*w 0.2*h];
pb9 = uicontrol('Style','pushbutton','String','Remove',...
		'Units','normalized','Position',pos,...
		'Callback',...
		'vb_select_area(''callback_remove_area'')');

select_area_variables.lb2 = lb2;
select_area_variables.pb2 = pb2;
select_area_variables.cb2 = cb2;
select_area_variables.ed1 = ed1;
select_area_variables.pb9 = pb9;

return;

%
% Select area
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function select_area_inner

global select_area_variables;

% indices of selected region
na = get(select_area_variables.lb2,'Value');

select_area_variables.disp_area = cell(length(na),1);
for i = 1:length(na)
  select_area_variables.disp_area{i} = ...
      select_area_variables.area_keys{na(i)};
end

plot_brain_inner;

return;

%
% Cancel all area
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function cancel_all_area

global select_area_variables

set(select_area_variables.lb2,'Value',[]);
select_area_variables.disp_area = {};

plot_brain_inner;

return;

%
% Create peak panel
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function create_peak_panel(pos0)

global select_area_variables;

create_frame(pos0);

m = 0.04*pos0(4);% margin
x = pos0(1)+m;   % origin (x)
y = pos0(2)+m;   % origin (y)
w = pos0(3)-2*m; % width
h = pos0(4)-2*m; % height

%
% peak1
%
pos = [x y+0.75*h 0.2*w 0.25*h];
uicontrol('Style','text','String','Peak 1',...
	  'Units','normalized','Position',pos,...
	  'ForegroundColor','r');

pos = [x+0.2*w y+0.75*h 0.2*w 0.25*h];
tx2 = uicontrol('Style','edit','String','','Enable','inactive',...
		'Units','normalized','Position',pos);

pos = [x+0.4*w y+0.75*h 0.3*w 0.25*h];
pb3 = uicontrol('Style','pushbutton','String','Search peak',...
		'Units','normalized','Position',pos,...
		'Enable','on','Callback',...
		'vb_select_area(''search_peak(1)'')');

pos = [x+0.7*w y+0.75*h 0.3*w 0.25*h];
pb4 = uicontrol('Style','pushbutton','String','Registration',...
		'Units','normalized','Position',pos,...
		'Enable','off','Callback',...
		'vb_select_area(''select_peak_area(1);'')');

%
% peak2
%
pos = [x y+0.5*h 0.2*w 0.25*h];
uicontrol('Style','text','String','Peak 2',...
	  'Units','normalized','Position',pos,...
	  'ForegroundColor','b');

pos = [x+0.2*w y+0.5*h 0.2*w 0.25*h];
tx3 = uicontrol('Style','edit','String','','Enable','inactive',...
		'Units','normalized','Position',pos);

pos = [x+0.4*w y+0.5*h 0.3*w 0.25*h];
pb5 = uicontrol('Style','pushbutton','String','Search peak',...
		'Units','normalized','Position',pos,...
		'Enable','on','Callback',...
		'vb_select_area(''search_peak(2)'')');

pos = [x+0.7*w y+0.5*h 0.3*w 0.25*h];
pb6 = uicontrol('Style','pushbutton','String','Registration',...
		'Units','normalized','Position',pos,...
		'Enable','off','Callback',...
		'vb_select_area(''select_peak_area(2);'')');

%
% Excluded radius
%
pos = [x y+0.25*h 0.6*w 0.25*h];
uicontrol('Style','text','String','Excluded radius [mm]',...
	  'Units','normalized','Position',pos);

pos = [x+0.6*w y+0.25*h 0.4*w 0.25*h];
ed2 = uicontrol('Style','edit','String','0',...
		'Units','normalized','Position',pos,...
		'Callback', ...
		'vb_select_area(''change_exclude_radius'')');

%
% Distance
%
pos = [x y 0.6*w 0.25*h];
uicontrol('Style','text','String','Distance (euclid/cortex)',...
	  'Units','normalized','Position',pos);

pos = [x+0.6*w y 0.4*w 0.25*h];
tx4 = uicontrol('Style','edit','String','','Enable','inactive',...
		'Units','normalized','Position',pos);

select_area_variables.tx2 = tx2;
select_area_variables.pb3 = pb3;
select_area_variables.pb4 = pb4;
select_area_variables.tx3 = tx3;
select_area_variables.pb5 = pb5;
select_area_variables.pb6 = pb6;
select_area_variables.tx4 = tx4;
select_area_variables.ed2 = ed2;

return;

%
% Create close button
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function create_close_button(pos)

uicontrol('Style','pushbutton','String','Close',...
	  'Units','normalized','Position',pos,...
	  'Enable','on','Callback',...
	  'vb_select_area(''close_window'')');

return;

%
% Select area around peak position
% 2004-03-15 Taku Yoshioka
%
function select_peak_area(i)

global select_area_variables;

ix_peak1 = select_area_variables.ix_peak1;
ix_peak2 = select_area_variables.ix_peak2;
Radius = select_area_variables.Radius;
nextDD = select_area_variables.nextDD;
nextIX = select_area_variables.nextIX;

switch i
 case 1, 
  ix = find(nextDD{ix_peak1}<=Radius);
  Iextract = nextIX{ix_peak1}(ix);
 case 2,
  ix = find(nextDD{ix_peak2}<=Radius);
  Iextract = nextIX{ix_peak2}(ix);
end
  
areafile = select_area_variables.areafile;

fprintf('--- %d vertices were selected.\n',...
	length(Iextract));
area_key = inputdlg('Input ID for the selected region: ');

NewArea.Iextract = unique(Iextract);
NewArea.key = area_key{1};
NewArea.Iextract_seq = '';

vb_add_area(areafile,NewArea);

init_stack;
plot_brain_inner;
update_listbox_area;

return;

%
% Search peak
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function search_peak(ipeak)

global select_area_variables;

areafile = select_area_variables.areafile;
area_keys = select_area_variables.area_keys;
nextDD = select_area_variables.nextDD;
nextIX = select_area_variables.nextIX;
ix_peak1 = select_area_variables.ix_peak1;

% Search region
na = get(select_area_variables.lb2,'Value');
if length(na)==0 % No area selected
  ix_search = 1:length(nextDD);
else
  ix_search = [];
  for i = 1:length(na)
    Area = vb_get_area(areafile,area_keys{na(i)});
    ix_search = [ix_search; Area.Iextract];
  end
end

ix_search = unique(ix_search);

% Exclude region around peak 1
if ipeak==2 & ~isempty(ix_peak1)
  Radius = get(select_area_variables.ed2,'String');
  Radius = str2num(Radius)*1e-3;
  ix = find(nextDD{ix_peak1}<=Radius);
  ix = nextIX{ix_peak1}(ix);
  ix_search = setdiff(ix_search,ix);
end

ix_mask = zeros(length(nextDD),1);
ix_mask(ix_search) = 1;

% Find peak from selected modality
if get(select_area_variables.rb1(1),'Value')
  na = get(select_area_variables.lb1,'Value');
  act = vb_get_act(select_area_variables.actfile,...
		select_area_variables.act_keys{na(1)});
  act.xxP = act.xxP.*ix_mask;
  [tmp,ix_peak] = max(abs(act.xxP));
elseif ~isempty(select_area_variables.J_map)
  [tmp,ix_peak] = max(abs(select_area_variables.J_map.*ix_mask));
else
  return;
end

switch ipeak
 case 1,
  select_area_variables.ix_peak1 = ix_peak;
  set(select_area_variables.pb4,'Enable','on');
  set(select_area_variables.tx2,'String',num2str(ix_peak));
 case 2,
  select_area_variables.ix_peak2 = ix_peak;
  set(select_area_variables.pb6,'Enable','on');
  set(select_area_variables.tx3,'String',num2str(ix_peak));
end

if ~isempty(select_area_variables.ix_peak1) & ...
      ~isempty(select_area_variables.ix_peak2)
  ix1 = select_area_variables.ix_peak1;
  ix2 = select_area_variables.ix_peak2;
  V0 = select_area_variables.V0;
  d1 = sqrt(sum((V0(ix1,:)-V0(ix2,:)).^2));
  ix = find(nextIX{ix1}==ix2);
  if isempty(ix)
    str = sprintf('%2.1f/>Rmax',1000*d1);
  else
    str = sprintf('%2.1f/%2.1f',1000*d1,1000*nextDD{ix1}(ix));
  end
  set(select_area_variables.tx4,'String',str);
end

plot_brain_inner;

return;

%
% Clear estimation result
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function clear_estimation_result

global select_area_variables;

% Clear filename on GUI
set(select_area_variables.tx1,'String','');
set(select_area_variables.pb8,'Enable','off');

% Clear global variables
select_area_variables.J_map = [];
select_area_variables.F = [];
select_area_variables.ix = [];
select_area_variables.B = [];

% Plot brain model
plot_brain_inner;

return;

%
% Change exluded radius
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function change_exclude_radius

return;

%
% Select display area
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function select_disp_area

global select_area_variables;

area_keys = select_area_variables.area_keys;

na = get(select_area_variables.lb3,'Value');
Area = vb_get_area(select_area_variables.areafile,area_keys{na(1)});
select_area_variables.ix_disp_area = Area.Iextract;
plot_brain_inner;

return;

%
% Update listbox of areas
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function update_listbox_area

global select_area_variables;

area_old = select_area_variables.area_keys;
area_new = vb_get_keyset_area(select_area_variables.areafile);

if length(area_old)~=length(area_new)
  select_area_variables.area_keys = area_new;

  % Listbox string table
  str = [];
  for i = 1:length(area_new)
    str = [str area_new{i}];
    if i~=length(area_new)
      str = [str '|'];
    end
  end
  set(select_area_variables.lb2,'String',str);
  set(select_area_variables.lb3,'String',str);

  % Re-select area for lb2
  ix_old = get(select_area_variables.lb2,'Value');
  ix_new = [];
  for i=1:length(ix_old)
    key_old = area_old{ix_old(i)};
    j = strmatch(key_old,area_new,'exact');
    if ~isempty(j)
      ix_new = [ix_new; j(1)];
    end
  end
  set(select_area_variables.lb2,'Value',ix_new);
  
  % Re-select area for lb3
  ix_old = get(select_area_variables.lb3,'Value');
  ix_new = [];
  for i=1:length(ix_old)
    key_old = area_old{ix_old(i)};
    j = strmatch(key_old,area_new,'exact');
    if ~isempty(j)
      ix_new = [ix_new; j(1)];
    end
  end
  set(select_area_variables.lb3,'Value',ix_new);
end

return;

%
% Change J Max/Min ratio
% 2006-09-27 rhayashi
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function change_J_ratio

global select_area_variables
plot_parm = select_area_variables.plot_parm;

err = false;
hObj = gcbo; % current action object

switch(hObj)
    case select_area_variables.jmin_edit
        Jmin_ratio = str2double(get(hObj, 'String'));
        if ~(0<= Jmin_ratio && Jmin_ratio <= 1.0)
            errordlg('Jmin ratio should be from 0 to 1.0', 'error');
            % set previous value
            set(hObj, 'String', num2str(plot_parm.Jmin_ratio));
            err = true;
            return;
        end
        set(select_area_variables.jmin_slider, 'Value', Jmin_ratio);
        plot_parm.Jmin_ratio = Jmin_ratio;
    case select_area_variables.jmax_edit
        Jmax_ratio = str2double(get(hObj, 'String'));
        if ~(0<= Jmax_ratio && Jmax_ratio <= 1.0)
            errordlg('Jmax ratio should be from 0 to 1.0', 'error');
            % set previous value
            set(hObj, 'String', num2str(plot_parm.Jmax_ratio));
            err = true;
            return;
        end
        set(select_area_variables.jmax_slider, 'Value', Jmax_ratio);
        plot_parm.Jmax_ratio = Jmax_ratio;
    case select_area_variables.jmin_slider
        Jmin_ratio = get(hObj, 'Value');
        set(select_area_variables.jmin_edit, 'String', num2str(Jmin_ratio));
        plot_parm.Jmin_ratio = Jmin_ratio;
    case select_area_variables.jmax_slider
        Jmax_ratio = get(hObj, 'Value');
        set(select_area_variables.jmax_edit, 'String', num2str(Jmax_ratio));
        plot_parm.Jmax_ratio = Jmax_ratio;
    otherwise
        % Reset value
        Jmax_ratio = plot_parm.Jmax_ratio;
        Jmin_ratio = plot_parm.Jmin_ratio;
        set(select_area_variables.jmax_slider, 'Value', Jmax_ratio);
        set(select_area_variables.jmin_slider, 'Value', Jmin_ratio);
        set(select_area_variables.jmax_edit, 'String', num2str(Jmax_ratio));
        set(select_area_variables.jmin_edit, 'String', num2str(Jmin_ratio));
end

select_area_variables.plot_parm = plot_parm;
plot_brain_inner;

return;

%
% Initialize stack of selected points
% 2004-03-15 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function init_stack

global select_area_variables;

select_area_variables.stack_points = cell(256,1); % buffer size: 256
select_area_variables.ix_stack_points = 0;

return;

%
% Print figure
% 2004-03-16 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function print_figure

global select_area_variables;

plot_parm = select_area_variables.plot_parm;

axes(select_area_variables.ax1);
[az,el] = view;
angle = [az el];

h = figure; 
set(gcf,'Renderer','zbuffer');
plot_brain_inner(angle);

material dull;
camlight headlight;

vb_epsfig(h,plot_parm.print_ratio,plot_parm.print_size);
axis tight;
if plot_parm.light==2, 
  camlight headlight;
end
[fname,pname] = uiputfile({'*.eps';'EPS-file(*.eps)'}, 'Save as', 'Untitled.eps');
if (isnumeric(fname) && fname == 0 )
    % cancel
    return;
else
    print(h, '-depsc2',[pname fname]);
    disp(['Output epsfile as ' [pname fname]]);
    msgbox('File saved.', 'msg');
end
    
close(h);

return;

%
% Change plot parameters
% 2004-03-25 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function change_plot_parm

global select_area_variables;

axes(select_area_variables.ax1);
[az,el] = view; 
select_area_variables.plot_parm.angle = [az el];

plot_parm = vb_plotparmdlg(select_area_variables.plot_parm);
select_area_variables.plot_parm = plot_parm;
change_J_ratio;
axes(select_area_variables.ax1);
view(plot_parm.angle);
plot_brain_inner;

return;

%
% Change time window
% 2004-04-01 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function change_timewindow

global select_area_variables;

ix = select_area_variables.ix;
J_map = select_area_variables.J_map;
resultfile = select_area_variables.bayesfile;
plot_parm  = select_area_variables.plot_parm;

if ~isempty(resultfile)  
  load(resultfile, 'Jact','ix_act','Tstart','Tend','Lact', 'MEGinfo');
  t = [Tstart:Tend];
  [Nj, T] = size(Jact);
  
  if isfield(select_area_variables, 'Twin_from')
      Twin_from = select_area_variables.Twin_from;
  else
      Twin_from = Tstart;
  end
  
  if isfield(select_area_variables, 'Twin_to')
      Twin_to = select_area_variables.Twin_to;
  else
      Twin_to = Tend;
  end
  
  [Twin_from, Twin_to, cancelled] = ...
    select_time_window(MEGinfo.Pretriger, Tstart, Tend, ...
                    MEGinfo.SampleFreq, ...
                    Twin_from, Twin_to); % initial value
  if cancelled, return; end
  
  Twin = [Twin_from Twin_to];
  
  tix  = find( t >= Twin(1) & t <= Twin(2) );

  Jact = Jact(:,tix);
  T  = length(tix);
  
  JJ = sqrt(sum(reshape(Jact, [Nj/Lact Lact*T]).^2, 2)/(Lact*T));
  
  % Update current distribution
  J_map = zeros(length(J_map),1);
  J_map(ix) = JJ;
  select_area_variables.J_map = J_map;
  select_area_variables.Twin_from = Twin_from;
  select_area_variables.Twin_to   = Twin_to;

  % Plot brain model
  plot_brain_inner;
end

return;

%
% Close this window
% 2004-04-02 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function close_window

close;

return;

%
% Remove selected area data
% 2004-06-18 Taku Yoshioka
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function callback_remove_area

global select_area_variables;

lb2 = select_area_variables.lb2; 
areafile = select_area_variables.areafile;

ix_key = get(lb2,'Value');
area_key = vb_get_keyset_area(areafile);

for i=length(ix_key):-1:1
  vb_rm_area(areafile,area_key{ix_key(i)});
end

update_listbox_area;

return;
