function job_plot_currentmap(brain_file,curr_file,ix_trial,area_file,act_file,filedlg_mode)
% Plot spatio-temporal pattern of estimated current.
%
% [syntax]
% job_plot_currentmap(brain_file,curr_file,ix_trial,area_file,act_file,filedlg_mode)
%
% [input]
% brain_file  : <optional> <<string>> Cortical surface model file
%               (.brain.mat).
% curr_file   : <optional> <<string>> Cortical current file (.curr.mat) or
%               time-frequency file (.tf.mat; not supported yet). 
% ix_trial    : <optional> <<int>> Trial index of the current data. If this
%               parameter is empty, average current data will be loaded. 
% area_file   : <optional> <<string>> Cortical area file (.area.mat).
% act_file    : <optional> <<string>> Cortical activity file (.act.mat).
% filedlg_mode: <optional> <<int>> See job_plot_cmap_setting_dlg (default=1).
%
% [example]
% job_plot_currentmap('./sbj_dir/brain/3D.brain.mat', ...
%                     './sbj_dir/result/result1.curr.mat',12, ...
%                     './sbj_dir/brain/3D.area.mat', ...
%                     './sbj_dir/brain/3D.act.mat')
%
% [note for developers]
% Plotting mode (current/time-freq) is determined as follows: 
%  If Jact of plotcmap{n} is not empty, plotting mode is 'current'. If
%  Jact is empty and data of plotcmap{n} is not empty, plotting mode is
%  'time-freq'. Otherwise, plotting mode is not defined. 
%
% Fields of vbmeg_inst.plotcmap{n} are as follows:
% ---
%  (cortical surface model variables)
%  brain_file: Cortical surface model file.
%  model     : Indicating original or inflated display. 
%  V         : Vertex coordinate values for cortical surface model. 
%  Vinf      : Vertex coordinate values for inflated surface model. 
%  F         : Face indices. 
%  inf_C     : Face color for inflated model. 
%  xxA       : Cortical area size (assigned for vertices, not faces). 
%  Vtal      : Talairach brain coordinate values. 
%  nextDD    : Distance to the neighbouring vertices. 
%  nextIX    : Indices of the neighbouring vertices.
%
%  (cortical current variables)
%  curr_file: Cortical current file. 
%  Jact     : J-current matrix. Cortical current data is always loaded as
%             J-current (@see job_plot_cmap_load_current). 
%  Jmax     : Maximum amplitude in Jact. This is set when loadging
%             cortical current data (@see job_plot_cmap_load_current). 
%  Jmax_plot: Maximum threshold of current for display. 
%  Jmin_plot: Minimum threshold of current for display. 
%  ix_trial : Trial index
%  ave_mode : current average mode
%  toi      : Time of interest (selected time window). 
%  foi      : Frequency of interest (ignored).
%  Jinfo    : Current information accessed by vb_load_current_info.
%
%  (time-frequency variables)
%  tffile    : Time-frequency file. 
%  data      : Time-frequency data. 
%  TFmax     : Maximum amplitude in data. This is set when loading
%             time-frequency data (@see job_plot_cmap_load_current). 
%  TFmin     : Minimum value in data. This variable is used to judge tf
%              value is always positive or not. 
%  TFmax_plot: Maximum threshold of TFdata for display.
%  TFmax_plot: Minimum threshold of TFdata for display.
%  ix_trial  : Trial index (ignored). 
%  toi       : Time of interest (selected time window).
%  foi       : Frequency of interest (selected frequency window).
%  TFinfo    : Time-frequency data information. 
%
%  (activity map variables)
%  act_file  : Activity map file currently loaded.
%  xxP       : Activity map to be shown.
%
%  (cortical area variables)
%  area_file : Cortical area file currently loaded.
%  area_key  : ID of cortical area to be shown. 
% 
%  (plotting parameters)
%  plot_parm: Structure of plot parameters.
%  ix_peak  : Index of selected vertex.
%  J_hold   : Switch of hold mode for displaying timecourses. 
%  W        : Smoothing matrix used for timecourse plot.
%
%  (obsolete variables in 0.9-0.a.0 or later)
%  Jbck      : Background current, for compatibility with older versions. 
%  ix_act    : Vertex indices for J-current. -> ix_act in Jinfo.
%  ix_bck    : Indices of background vertices.
%  Tstart    : 
%  T         : Actual time. -> Tmsec in Jinfo or TFinfo.
%  t         : Selected time window -> toi. 
%  patch_norm: Flag of patch normalization -> patch_norm in Jinfo.
% ---
%
% [history]
% 2007-01-30 Taku Yoshioka
% 2007-03-13 Taku Yoshioka
% 2008-09-29 Taku Yoshioka
% 2008-12-16 Taku Yoshioka
%  Current map threshold is replaced by actual amplitude. 
% 2010-12-01 taku-y
%  [internal change] 'Jinfo' replaces 'MEGinfo'. 
% 2010-12-11 taku-y
%  [enhancement] Time-frequency data supported. 
% 2011-01-28 taku-y
%  [enhancement] Activity map plot supported. 
% 2011-02-16 taku-y
%  [enhancement] Cortical area plot supported. 
%  [enhancement] Show area list including selected vertex. 
% 2011-03-14 taku-y
%  [debug] Automatic update of camera position when pressing viewpoint
%  buttons on GUI ('Left', 'Right', etc.). 
% 2011-04-01 taku-y
%  [debug] Make '>/>>' buttons available. 
% 2012-01-26 sako
%  [minor change] n_trial -> "n_trial" was replaced to "ix_trial"
% 2012-02-17 taku-y
%  [major] File menu was replaced with a new dialog. 
% 2012-02-28 taku-y
%  [minor] File dialog was modified to recognize input arguments as
%  default value of the file names and trial index. 
% 2012-08-31 taku-y
%  [minor] Added file dialog mode (filedlg_mode) to make cortical current
%  file be optional in the dialog. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- Check input parameters
if nargin<6                    , filedlg_mode = 0; end
if nargin<5                    , act_file   = '' ; end
if nargin<4                    , area_file  = '' ; end
if nargin<3 | isempty(ix_trial), ix_trial   = 1  ; end
if nargin<2                    , curr_file  = '' ; end
if nargin<1                    , brain_file = '' ; end

inst_id = create_instance;
if isempty(inst_id), return; end

% --- check model
if ~exist('model', 'var'), model =0; end % Inflated model
if model ~= 0, model = 1; end            % original model

% --- Create GUI and update plots
create_GUI(inst_id,brain_file,curr_file,ix_trial,area_file,act_file, ...
           model,filedlg_mode);
job_plot_cmap_update_filterradius(inst_id);
job_plot_cmap_update_timecourse(inst_id); 
job_plot_cmap_update_spatialpattern(inst_id);

return;

% --- Inner function: create instance of GUI
function inst_id = create_instance

global vbmeg_inst

if isfield(vbmeg_inst,'plotcmap'),
  for inst_id=1:vbmeg_inst.const.PLOTCMAP_MAX
    if isempty(vbmeg_inst.plotcmap{inst_id}), return; end;
  end
  vb_disp(['The number of instances of ''job_plot_currentmap''' ...
	   ' GUIs reaches the maximum number.']);
  inst_id = [];
elseif isfield(vbmeg_inst,'const'), 
  vbmeg_inst.plotcmap = cell(vbmeg_inst.const.PLOTCMAP_MAX,1);
  inst_id = 1;
else
  vb_disp(['''vbmeg_inst'' was not correct. You have to invoke' ...
	   '''vbmeg'' to create an instance of VBMEG.']); 
  inst_id = []; 
end

return;

% --- Inner function: create GUI
function create_GUI(inst_id,brain_file,curr_file,ix_trial,area_file, ...
                    act_file,model,filedlg_mode)

global vbmeg_inst;

% --- Initialize parameters
% Cortical surface model variables
plotcmap.brain_file = '';
plotcmap.V      = [];
plotcmap.Vinf   = [];
plotcmap.F      = [];
plotcmap.inf_C  = [];
plotcmap.xxA    = [];
plotcmap.Vtal   = [];
plotcmap.nextDD = [];
plotcmap.nextIX = [];

% Cortical current variables
Jmin_plot = 0;
Jmax_plot = 1;
plotcmap.curr_file = '';
plotcmap.Jact      = [];
plotcmap.Jmax      = [];
plotcmap.Jmin      = [];
plotcmap.Jmax_plot = Jmax_plot;
plotcmap.Jmin_plot = Jmin_plot;
plotcmap.ix_trial  = 1;
plotcmap.ave_mode  = true;
plotcmap.toi       = [];
plotcmap.foi       = [];
plotcmap.Jinfo     = [];

% Time-frequency variables
plotcmap.tf_file    = '';
plotcmap.data       = [];
plotcmap.TFinfo     = [];
plotcmap.TFmax      = [];
plotcmap.TFmin      = [];
plotcmap.TFmin_plot = 0;
plotcmap.TFmax_plot = 1;

% Activity map variables
plotcmap.act_file = '';
plotcmap.xxP      = [];

% Cortical area variables
plotcmap.area_file = '';
plotcmap.area_key  = [];

% Plotting parameters
plotcmap.plot_parm = vb_set_plot_parm;
plotcmap.plot_parm.index = [];
plotcmap.plot_parm.LRflag = 'LR';
plotcmap.ix_peak = 1;
plotcmap.J_hold = [];
% plotcmap.ix_selected_area = [];

% file dialog parameters
plotcmap.filedlg_mode = filedlg_mode;

% Obsolete variables in 0.9-0.a.0 or later
plotcmap.Jbck       = [];
plotcmap.ix_act     = [];
plotcmap.ix_bck     = [];
plotcmap.Tstart     = [];
plotcmap.T          = [];
plotcmap.t          = []; 
plotcmap.patch_norm = [];

% --- Open GUI
h_fig = open('job_plot_cmap.fig');
set(h_fig,'HandleVisibility','on');
set(h_fig,'Pointer','watch');
drawnow; 

% input parameter
plotcmap.model_type = model;

% --- handles of GUI components
plotcmap.h_fig     = h_fig;
plotcmap.h_time    = get_child_handle(h_fig,'plot_time');
plotcmap.h_space   = get_child_handle(h_fig,'plot_brain');
plotcmap.h_cbar    = get_child_handle(h_fig,'colorbar');
plotcmap.h_cbar_tf = get_child_handle(h_fig,'colorbar_tf');
plotcmap.ed1 = get_child_handle(h_fig,'trial_number');
plotcmap.ed2 = get_child_handle(h_fig,'time_window_start');
plotcmap.ed3 = get_child_handle(h_fig,'time_window_end');
plotcmap.ed4 = get_child_handle(h_fig,'ix_peak');
plotcmap.ed5 = get_child_handle(h_fig,'spatial_peak_size');
plotcmap.ed6 = get_child_handle(h_fig,'tal_coord');
plotcmap.ed7 = get_child_handle(h_fig,'temporal_peak_size');
plotcmap.ed8 = get_child_handle(h_fig,'freq_window_min');
plotcmap.ed9 = get_child_handle(h_fig,'freq_window_max');
plotcmap.ed10 = get_child_handle(h_fig,'Jmax_tf_plot_edit');
plotcmap.pb1 = get_child_handle(get_child_handle(h_fig,'save_menu'), ...
                                'spatial_pattern_menu');
plotcmap.pb2 = get_child_handle(get_child_handle(h_fig,'save_menu'), ...
                                'temporal_pattern_menu');
plotcmap.pb3 = get_child_handle(h_fig,'rotate');
plotcmap.pb4 = get_child_handle(h_fig,'move_forward');
plotcmap.pb5 = get_child_handle(h_fig,'move_forward_fast');
plotcmap.pb6 = get_child_handle(h_fig,'move_back');
plotcmap.pb7 = get_child_handle(h_fig,'move_back_fast');
plotcmap.pb8 = get_child_handle(h_fig,'search_spatial_peak');
plotcmap.pb9 = get_child_handle(h_fig,'search_temporal_peak');
plotcmap.pb10 = get_child_handle(get_child_handle(h_fig, 'save_menu'), ...
                                 'cortical_activity_map_menu');
plotcmap.pb12 = get_child_handle(get_child_handle(h_fig, 'save_menu'), ...
                                 'cortical_area_menu');
plotcmap.pb13 = get_child_handle(h_fig,'left_button');
plotcmap.pb14 = get_child_handle(h_fig,'right_button');
plotcmap.pb15 = get_child_handle(h_fig,'top_button');
plotcmap.pb16 = get_child_handle(h_fig,'bottom_button');
plotcmap.pb17 = get_child_handle(h_fig,'front_button');
plotcmap.pb18 = get_child_handle(h_fig,'back_button');
plotcmap.rb1 = get_child_handle(h_fig,'talairach');
plotcmap.rb2 = get_child_handle(h_fig,'mni');
plotcmap.cb1 = get_child_handle(h_fig,'fix_colorbar');
plotcmap.cb2 = get_child_handle(h_fig,'left_brain');
plotcmap.cb3 = get_child_handle(h_fig,'right_brain');
plotcmap.cb4 = get_child_handle(h_fig,'hold_on');
plotcmap.cb5 = get_child_handle(h_fig,'show_roi');
plotcmap.cb6 = get_child_handle(h_fig,'fix_colorbar_tf');
plotcmap.cb7 = get_child_handle(h_fig,'show_act_map');
plotcmap.cb8 = get_child_handle(h_fig,'show_cortical_area');
plotcmap.lb1 = get_child_handle(h_fig,'act_key_list');
plotcmap.lb2 = get_child_handle(h_fig,'area_key_list');
plotcmap.lb3 = get_child_handle(h_fig,'cortical_area');
plotcmap.Jmin_plot_edit = get_child_handle(h_fig, 'Jmin_plot_edit');
plotcmap.Jmax_plot_edit = get_child_handle(h_fig, 'Jmax_plot_edit');
plotcmap.model_type_original_rb ...
    = get_child_handle(h_fig,'model_type_original_rb');
plotcmap.model_type_inflate_rb ...
    = get_child_handle(h_fig,'model_type_inflate_rb');
plotcmap.load_data_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'load_data_menu');
plotcmap.loaded_files_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'loaded_files_menu');
plotcmap.close_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'close_menu');

plotcmap.h_rotate  = rotate3d(h_fig); % to control camlight

% Set GUI values
set(plotcmap.ed1,'String','1');
set(plotcmap.ed1,'Enable','off')

% --- callback functions
inst_str = num2str(inst_id); 
plotcmap_str = ['vbmeg_inst.plotcmap{' inst_str '}'];

% Remove instance after window close
command = ['global vbmeg_inst; '...
	   'vbmeg_inst.plotcmap{' inst_str '}=[];'];
set(h_fig,'DeleteFcn',command);

% Zoom button
command = ['job_plot_cmap_push_zoom_button(' inst_str ');']; 
set(plotcmap.pb3,'Callback',command);
if vb_matlab_version('>=', '7.6')
  set(plotcmap.pb3, 'Visible', 'off');
end

% Rotation of cortex model
if vb_matlab_version('>=', '7.3');
    set(plotcmap.h_rotate, 'ActionPostCallback', @after_rotate_callback);
end

% Trial number
command = ['job_plot_cmap_change_trial(' inst_str ');'];
set(plotcmap.ed1,'Callback',command);

% Time window
command = ['global vbmeg_inst; ' ...
	   plotcmap_str '.toi(1)=' ...
	   'str2num(get(' plotcmap_str '.ed2,''String''));' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.ed2,'Callback',command);
command = ['global vbmeg_inst; ' ...
	   plotcmap_str '.toi(2)=' ...
	   'str2num(get(' plotcmap_str '.ed3,''String''));' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.ed3,'Callback',command);

% Frequency window
command = ['global vbmeg_inst; ' ...
	   plotcmap_str '.foi(1)=' ...
	   'str2num(get(' plotcmap_str '.ed8,''String''));' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.ed8,'Callback',command);
command = ['global vbmeg_inst; ' ...
	   plotcmap_str '.foi(2)=' ...
	   'str2num(get(' plotcmap_str '.ed9,''String''));' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.ed9,'Callback',command);

% Time window shift
command = ['global vbmeg_inst; '...
	   'dt = ' plotcmap_str '.toi(2)-' plotcmap_str '.toi(1);' ...
	   plotcmap_str '.toi=' plotcmap_str '.toi+dt;' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.pb4,'Callback',command);
command = ['global vbmeg_inst; '...
	   'dt = ' plotcmap_str '.toi(2)-' plotcmap_str '.toi(1);' ...
	   plotcmap_str '.toi=' plotcmap_str '.toi+5*dt;' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.pb5,'Callback',command);
command = ['global vbmeg_inst; '...
	   'dt = ' plotcmap_str '.toi(2)-' plotcmap_str '.toi(1);' ...
	   plotcmap_str '.toi=' plotcmap_str '.toi-dt;' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.pb6,'Callback',command);
command = ['global vbmeg_inst; '...
	   'dt = ' plotcmap_str '.toi(2)-' plotcmap_str '.toi(1);' ...
	   plotcmap_str '.toi=' plotcmap_str '.toi-5*dt;' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.pb7,'Callback',command);

% Search peak
command = ['job_plot_cmap_spatial_peak(' inst_str ');'];
set(plotcmap.pb8,'Callback',command);
command = ['job_plot_cmap_temporal_peak(' inst_str ');'];
set(plotcmap.pb9,'Callback',command);

% Jmax_plot_edit and Jmin_plot_edit
command = ['job_plot_cmap_change_Jplot(' inst_str ');'];
set(plotcmap.Jmin_plot_edit,'Callback',command);
set(plotcmap.Jmax_plot_edit,'Callback',command);
set(plotcmap.Jmin_plot_edit,'String','');
set(plotcmap.Jmax_plot_edit,'String','');

% Jmax_tf_plot_edit
command = ['job_plot_cmap_change_Jplot_tf(' inst_str ');'];
set(plotcmap.ed10,'Callback',command);
set(plotcmap.ed10,'String','');

% Print figure
command = ['job_plot_cmap_print_spatialpattern(' inst_str ');'];
set(plotcmap.pb1,'Callback',command);
command = ['job_plot_cmap_print_temporalpattern(' inst_str ');'];
set(plotcmap.pb2,'Callback',command);

% Change vertex 
command = ['global vbmeg_inst;' ...
	   plotcmap_str '.ix_peak = ' ...
	   'str2num(get(' plotcmap_str '.ed4,''String''));' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');' ...
	   'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.ed4,'Callback',command);

% Change standard brain coordinate
command = ['global vbmeg_inst;' ...
	   'job_plot_cmap_change_talcoord(' inst_str ');'];
set(plotcmap.ed6,'Callback',command);

% Change filter radius
command = ['job_plot_cmap_update_filterradius(' inst_str ');' ...
           'job_plot_cmap_update_spatialpattern(' inst_str ');' ...
	   'job_plot_cmap_update_timecourse(' inst_str ');'];
set(plotcmap.ed5,'Callback',command);

% Store activity map
command = ['job_plot_cmap_register_activity(' inst_str ');'];
set(plotcmap.pb10,'Callback',command);

% Store area
command = ['job_plot_cmap_register_area(' inst_str ');'];
set(plotcmap.pb12,'Callback',command);

% Radio button check
command = ['job_plot_cmap_change_rb1(' inst_str ');'];
set(plotcmap.rb1,'Callback',command);
command = ['job_plot_cmap_change_rb2(' inst_str ');'];
set(plotcmap.rb2,'Callback',command);

% Hold on
command = ['job_plot_cmap_holdon(' inst_str ');'];
set(plotcmap.cb4,'Callback',command);

% Show ROI
command = ['job_plot_cmap_show_roi(' inst_str ');'];
set(plotcmap.cb5,'Callback',command);

% Fix colorbar
command = ['job_plot_cmap_update_colorbarscale(' inst_str ');'];
set(plotcmap.cb1,'Callback',command);
set(plotcmap.cb6,'Callback',command);

% Left/Right brain
command = ['job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.cb2,'Callback',command);
set(plotcmap.cb3,'Callback',command);

% Show activity map
command = ['job_plot_cmap_update_colorbarscale(' inst_str ');' ...
           'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.cb7,'Callback',command);

% Show cortical area
%command = ['job_plot_cmap_update_colorbarscale(' inst_str ');' ...
%           'job_plot_cmap_update_spatialpattern(' inst_str ');'];
command = ['job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.cb8,'Callback',command);

% Select activity map
command = ['global vbmeg_inst; '...
           'job_plot_cmap_update_xxP(' inst_str ');' ...
           'job_plot_cmap_update_colorbarscale(' inst_str ');' ...
           'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.lb1,'Callback',command);

% Select cortical area
command = ['global vbmeg_inst; '...
           'job_plot_cmap_update_area(' inst_str ');' ...
           'job_plot_cmap_update_spatialpattern(' inst_str ');'];
set(plotcmap.lb2,'Callback',command);

% Model type
model_type_group = [plotcmap.model_type_inflate_rb,
                    plotcmap.model_type_original_rb];
command = ['job_plot_cmap_change_modeltype(' inst_str ');'];
set(model_type_group, 'Callback', command);
set(model_type_group, 'Value', 0); % both off
if model == 0
    set(plotcmap.model_type_inflate_rb, 'Value', 1);
else
    set(plotcmap.model_type_original_rb, 'Value', 1);
end

% % Cancel
% set(h_fig,'WindowButtonUpFcn','rotate3d off;');

% Load data
command = ['job_plot_cmap_load_data(' inst_str ');'];
set(plotcmap.load_data_menu, 'Callback', command);

% Show loaded files
command = ['job_plot_cmap_show_loaded_files(' inst_str ');']; 
set(plotcmap.loaded_files_menu, 'Callback', command);

% Close window
command = 'close;';
set(plotcmap.close_menu,'Callback',command);

% Angle
command = ['job_plot_cmap_change_angle(' inst_str ',''left'');' ...
           'job_plot_cmap_update_camera(' inst_str ');'];
set(plotcmap.pb13,'Callback',command);
command = ['job_plot_cmap_change_angle(' inst_str ',''right'');' ...
           'job_plot_cmap_update_camera(' inst_str ');'];
set(plotcmap.pb14,'Callback',command);
command = ['job_plot_cmap_change_angle(' inst_str ',''top'');' ...
           'job_plot_cmap_update_camera(' inst_str ');'];
set(plotcmap.pb15,'Callback',command);
command = ['job_plot_cmap_change_angle(' inst_str ',''bottom'');' ...
           'job_plot_cmap_update_camera(' inst_str ');'];
set(plotcmap.pb16,'Callback',command);
command = ['job_plot_cmap_change_angle(' inst_str ',''front'');' ...
           'job_plot_cmap_update_camera(' inst_str ');'];
set(plotcmap.pb17,'Callback',command);
command = ['job_plot_cmap_change_angle(' inst_str ',''back'');' ...
           'job_plot_cmap_update_camera(' inst_str ');'];
set(plotcmap.pb18,'Callback',command);

% --- properties of axis for cortex plot
set(h_fig,'Renderer','zbuffer');
set(h_fig,'CurrentAxes',plotcmap.h_space);
axis off;
%axis tight; % 2006/11/22 hayashi delete
%axis equal; % These commands causes axes LimMode change(auto-->manual)
             % after MATLAB 7.0.1. and brain model is not displayed. 

% --- set instance
vbmeg_inst.plotcmap{inst_id} = plotcmap; 

% Set pointer to arrow
set(h_fig,'Pointer','arrow');

% --- Load data
parms = [];

if ~isempty(brain_file),
  parms.brain_file = brain_file;
end

if ~isempty(curr_file),
  [tmp1,tmp2] = fileparts(curr_file);
  [tmp1,tmp2,str_ext] = fileparts([tmp1 filesep tmp2]);
  
  switch str_ext
   case '.curr',
    parms.curr_file = curr_file;
    parms.tf_file   = [];
    
   case '.tf',
    parms.curr_file = [];
    parms.tf_file   = curr_file;
    
   otherwise,
    vb_disp('WARNING',['Invalid extention for curr_file (job_plot_' ...
                       'currentmap.m)']);
  end
end

if ~isempty(ix_trial),
  parms.ix_trial = ix_trial;
  parms.ave_mode = false;
end

if ~isempty(area_file),
  parms.area_file = area_file;
end

if ~isempty(act_file),
  parms.act_file = act_file;
end

job_plot_cmap_load_data(inst_id,parms);

return;
