function [varargout] = pa_leadfield_util(fig, command, parm)
% This function is subcontracting function of leadfield GUI.
% [USAGE]
%    [varargout] = pa_leadfield_util(<fig>, <command> [,parm]);
% [IN]
%        fig : figure handle of leadfield gui.
%    command : utility type
%       parm : other parameters
% [OUT]
%    varargout{n}: return value
%           @command : 'get_basis_mode'  ==> BASIS_MODE
%           @command : 'get_bem_mode'    ==> BEM_MODE
%           @command : 'get_normal_mode' ==> NORMAL_MODE
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('fig', 'var') || isempty(fig) || ~ishandle(fig)
    error('invalid figure was specified.');
end
if ~exist('command', 'var') || isempty(command)
    error('invalid command was specified.');
end

%
% --- Main Procedure
%

% load data(@see pa_leadfield_create)
data = guidata(fig);

switch(command)
    case 'update_output_fname'
        update_output_fname(data.H);
    case 'update_exec_push_status'
        update_exec_push_status(data.H);
    case 'set_basis_mode'
        set_basis_mode(parm, data.H);
    case 'get_basis_mode'
        % return const basis mode number
        varargout{1} = get_basis_mode(data.H);
    case 'set_bem_mode'
        set_bem_mode(parm, data.H);
    case 'get_bem_mode'
        % return const bem mode number
        varargout{1} = get_bem_mode(data.H);
    case 'set_normal_mode'
        set_normal_mode(parm, data.H);
    case 'get_normal_mode'
        % return const normal mode number
        varargout{1} = get_normal_mode(data.H);
    case 'exec_leadfield'
        exec_leadfield(data);
    case 'reset_parameter'
        reset_parameter(data.H);
    case 'check_model'
        check_model(data.H);
end

% save data
guidata(fig, data);

function update_output_fname(H)
% This function makes output filename and sets filename to GUI.
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    global vbmeg_inst;
    define = vbmeg_inst.const;

    save_dir = get(H.save_dir_edit, 'String');

    megmat_file = get(H.megmat_file_edit, 'String');

    [fpath, fname, fext] = vb_get_file_parts(megmat_file);

    base_name = [fname, fext];
    dot_ix = strfind(base_name, '.');
    if ~isempty(dot_ix)
        base_name = base_name(1:dot_ix(end-1)-1);
    end

    %make basis filename
    basis_file = [save_dir, filesep, base_name, define.BASIS_EXTENSION];
    
    set(H.basis_file_edit, 'String', basis_file);

function update_exec_push_status(H)
% This function changes Exec button status.(clickable or not)
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    global vbmeg_inst;
    define = vbmeg_inst.const;

    if ~isempty(get(H.megmat_file_edit, 'String')) & ...
       ~isempty(get(H.brain_file_edit, 'String')) & ...
       ~isempty(get(H.save_dir_edit, 'String')) & ...
       ~isempty(get(H.basis_file_edit, 'String'))

        % Head file check.
        mode = get_bem_mode(H);
        if (mode == define.BASIS_MEG_BEM) || ...
           (mode == define.BASIS_EEG_BEM) || ...
           (mode == define.BASIS_MEG_HARMONICS)
            if isempty(get(H.head_file_edit, 'String'))
                set(H.exec_push, 'Enable', 'off');
                return;
            end
        end

        % Head center check.
        if (mode == define.BASIS_MEG_SPHERE) || ...
           (mode == define.BASIS_EEG_SPHERE)
           if isempty(get(H.center_head_edit, 'String'))
               set(H.exec_push, 'Enable', 'off');
               return;
           end
        end

        % Sphere radius check.
        if (mode == define.BASIS_EEG_SPHERE)
            if isempty(get(H.sphere_radius_edit, 'String'))
               set(H.exec_push, 'Enable', 'off');
               return;
           end
        end

        % Conductivity check
        if (mode == define.BASIS_EEG_SPHERE) || ...
           (mode == define.BASIS_EEG_BEM)
            if isempty(get(H.conductivity_edit, 'String'))
               set(H.exec_push, 'Enable', 'off');
               return;
           end
        end

        set(H.exec_push, 'Enable', 'on');
    else
        set(H.exec_push, 'Enable', 'off');
    end

function set_basis_mode(mode, H)
% This function sets GUI basis mode.
% (basis_mode ---> GUI popup value)
% [IN]
%    mode : basis mode
%       H : GUI component handles
% [OUT]
%    none
%
    global vbmeg_inst;
    define = vbmeg_inst.const;

    if mode == define.CURRENT_NORMAL
        value = 1;
    elseif mode == define.CURRENT_TANGENT
        value = 2;
    elseif mode == define.CURRENT_3D
        value = 3;
    else
        disp('Illigal usage:set_basis_mode().');
    end

    set(H.BasisMode_popup, 'Value', value);

function mode = get_basis_mode(H)
% This function returns basis mode.
% (GUI popup value ---> basis_mode)
% [IN]
%    H : GUI component handles
% [OUT]
%    mode : basis mode
%
    global vbmeg_inst;
    define = vbmeg_inst.const;

    value = get(H.BasisMode_popup, 'Value');
    if value == 1
        mode = define.CURRENT_NORMAL;
    elseif value == 2
        mode = define.CURRENT_TANGENT;
    elseif value == 3
        mode = define.CURRENT_3D;
    else
        disp('Invalid Basis mode!!');
    end

function set_bem_mode(mode, H)
% This function sets GUI bem mode.
% (bem_mode ---> GUI popup value)
% [IN]
%    mode : bem mode
%       H : GUI component handles
% [OUT]
%    none
%
    global vbmeg_inst;
    define = vbmeg_inst.const;

    gui_set_head_center(H, 'off');
    gui_set_headfile(H, 'off');
    gui_set_conductivity(H, 'off');
    gui_set_sphere_radius(H, 'off');

    if mode == define.BASIS_MEG_SPHERE
        value = 1;
        gui_set_head_center(H, 'on');
    elseif mode == define.BASIS_MEG_BEM
        value = 2;
        gui_set_headfile(H, 'on');
    elseif mode == define.BASIS_MEG_HARMONICS
        value =  3;
        gui_set_headfile(H, 'on');
    elseif mode == define.BASIS_EEG_SPHERE
        value = 4;
        gui_set_head_center(H, 'on');
        gui_set_sphere_radius(H, 'on');
        gui_set_conductivity(H, 'on');
    elseif mode == define.BASIS_EEG_BEM
        value = 5;
        gui_set_headfile(H, 'on');
        gui_set_conductivity(H, 'on');
    else
        disp('Illigal usage : set_bem_mode().');
    end
    set(H.BemMode_popup, 'Value', value);

function gui_set_sphere_radius(H, on_off)
    set(H.sphere_radius_label, 'Enable', on_off);
    set(H.sphere_radius_edit, 'Enable', on_off);

function gui_set_head_center(H, on_off)
    set(H.center_head_label, 'Enable', on_off);
    set(H.center_head_edit, 'Enable', on_off);
    set(H.center_head_push, 'Enable', on_off);

function gui_set_conductivity(H, on_off)
    set(H.conductivity_label, 'Enable', on_off);
    set(H.conductivity_edit, 'Enable', on_off);

function gui_set_headfile(H, on_off)
    set(H.headfile_label, 'Enable', on_off);
    set(H.head_file_edit, 'Enable', on_off);
    set(H.head_file_push, 'Enable', on_off);

function mode = get_bem_mode(H)
% This function returns bem mode.
% (GUI popup value ---> bem_mode)
% [IN]
%    H : GUI component handles
% [OUT]
%    mode : bem mode
%
    global vbmeg_inst;
    define = vbmeg_inst.const;

    value = get(H.BemMode_popup, 'Value');
    
    if value == 1
        mode = define.BASIS_MEG_SPHERE;
    elseif value == 2
        mode = define.BASIS_MEG_BEM;
    elseif value == 3
        mode = define.BASIS_MEG_HARMONICS;
    elseif value == 4
        mode = define.BASIS_EEG_SPHERE;
    elseif value == 5
        mode = define.BASIS_EEG_BEM;
    else
        disp('Invalid bem mode!!');
    end

function set_normal_mode(mode, H)
% This function sets GUI normal mode.
% (normal_mode ---> GUI popup value)
% [IN]
%    mode : normal mode
%       H : GUI component handles
% [OUT]
%    none
%
    global vbmeg_inst;
    define = vbmeg_inst.const;

    if mode == define.NORMAL_ADJACENT_MEAN
        value = 1;
    elseif mode == define.NORMAL_NEIGHBOR_MEAN
        value = 2;
    else
        disp('Illigal usage : set_normal_mode().');
    end

    set(H.NormalMode_popup, 'Value', value);

function mode = get_normal_mode(H)
% This function returns normal_mode by constant value.
% (GUI popup value ---> normal_mode)
% [IN]
%    H : GUI component handles
% [OUT]
%    mode : normal mode
%
    global vbmeg_inst;
    define = vbmeg_inst.const;

    value = get(H.NormalMode_popup, 'Value');

    if value == 1
        mode = define.NORMAL_ADJACENT_MEAN;
    elseif value == 2
        mode = define.NORMAL_NEIGHBOR_MEAN;
    else
        disp('Illigal usage : get_normal_mode().');
    end

function exec_leadfield(data)
% This function execute vb_job_leadfield with set parameter.
% [IN]
%    data : data of leafield gui(@see pa_leadfield_create)
% [OUT]
%    none

    H = data.H; % component handles
    
    %%%%%%%%%%%%%%%%%%%%
    % Parameter setting
    %%%%%%%%%%%%%%%%%%%%
    % project root directory
    proj_root = get(H.prjroot_edit, 'String');

    % advanced parameters are set here.
    basis_parm = data.basis_parm;

    % MEG-MAT file
    basis_parm.meg_file   = get(H.megmat_file_edit, 'String');

    % BRAIN file
    basis_parm.brain_file = get(H.brain_file_edit, 'String');
    
    % AREA file
    basis_parm.area_file  = get(H.area_file_edit, 'String');

    % HEAD file
    basis_parm.head_file  = get(H.head_file_edit, 'String');

    % MEG/EEG device name
    basis_parm.device     = [];

    % Center of the head
    basis_parm.Vcenter    = eval([ '[', get(H.center_head_edit, 'String'), ']' ]);
    
    % bem_mode: LFvZ@
    %  MEG
    %      BASIS_MEG_SPHERE    : MEG f  (Sarvas̎)
    %      BASIS_MEG_BEM       : MEG |eVꎟߎAK[L@Aʓd
    %      BASIS_MEG_HARMONICS : MEG ʊ֐WJ
    %  EEG
    %      BASIS_EEG_SPHERE    : EEG 3wf
    %      BASIS_EEG_BEM       : EEG |eVꎟߎAK[L@
    basis_parm.bem_mode   = get_bem_mode(H);

    % Basis_mode : number of independent curent direction
    %       CURRENT_NORMAL  :  current vextor = xx(n,:)
    %       CURRENT_TANGENT :  current vextor = orthogonal vectors to V(n,:)
    %       CURRENT_3D      :  current vextor = x, y, z -direction
    basis_parm.Basis_mode = get_basis_mode(H);

    % normal_mode: Average mode for normal direction
    %       NORMAL_ADJACENT_MEAN  = 0;  % אړ_̖@
    %       NORMAL_NEIGHBOR_MEAN  = 1;  % BV]fߖT_̖@
    basis_parm.normal_mode = get_normal_mode(H);

    % Conductivity
    basis_parm.sigma = eval([ '[', get(H.conductivity_edit, 'String'), ']' ]);
    
    % Sphere Radius
    basis_parm.radius = eval([ '[', get(H.sphere_radius_edit, 'String'), ']' ]);

    % BASIS file(output)
    basis_parm.basis_file = get(H.basis_file_edit, 'String');

    %
    % --- Advanced parameters
    %
    basis_parm.area_key = get(H.areakey_edit, 'String');
    
    err_code = vb_basisparm_check(basis_parm, proj_root);
    
    if err_code == 0
        %%%%%%%%%%%%%%%%%%%%
        % execute job
        %%%%%%%%%%%%%%%%%%%%
        str = get(H.exec_push, 'String');
        set(H.exec_push, 'String', 'Processing...');
        pause(0.01);%for label string update
        vb_job_leadfield(proj_root, basis_parm);
        set(H.exec_push, 'String', 'Exec');
    else
        % Show error reason.
        err_table =  ...
            {1, 'basis_parm is empty.';
             2, 'Project root directory is not specified.';
             4, 'Cortical model file is invalid.';
             8, 'MEG/EEG file is invalid.';
             16, 'Normal mode is invalid.';
             32, 'Basis mode is invalid.';
             64, 'Cortical area file or AreaID is invalid. ';
             128,sprintf('%s\n%s', 'Coordinate type of sensor is invalid.', ...
                         'Did you specify POS-MAT file when MEG/EEG file was made?')};
        for k=1:size(err_table)
            if bitand(err_code, err_table{k, 1})
                errordlg(err_table{k, 2}, 'error');
                return;
            end
        end
    end

function reset_parameter(H)
% This function resets brain GUI.
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    parm = vb_set_basis_parm;
    pa_leadfield_set_parm(H.figure, parm);
%     set(H.megmat_file_edit ,'String', '');
%     set(H.brain_file_edit ,'String', '');
%     set(H.area_file_edit ,'String', '');
%     set(H.head_file_edit ,'String', '');
%     set(H.areakey_edit, 'String', '');
%     set(H.save_dir_edit ,'String', '');
%     set(H.basis_file_edit ,'String', '');
%     
%     set(H.BemMode_popup, 'Value', 1);
%     set(H.BasisMode_popup, 'Value', 1);
%     set(H.NormalMode_popup, 'Value', 2);
    
function check_model(H)
% This function display forward model.
% [IN]
%    H: 
% [OUT]
%    none

global vbmeg_inst;
define = vbmeg_inst.const;

% move to some default parameter set in future
head_color = [0.9 0.8 0.8; 0.9 0.9 1.0; 1.0 0.8 0.8; ...
              1.0 0.8 0.8; 1.0 0.8 0.8; 1.0 0.8 0.8];
brain_color = [0.8 0.8 0.8];
falpha = [0.4 0.3 0.2 0.2];

% Project root directory
proj_root = get(H.prjroot_edit, 'String');

% MEG-MAT file
meg_file   = get(H.megmat_file_edit, 'String');

% BRAIN file
brain_file = get(H.brain_file_edit, 'String');
    
% HEAD file
head_file  = get(H.head_file_edit, 'String');

% Leadfield calculation method
bem_mode = get_bem_mode(H);

if isempty(proj_root), return; end

flag = 0;

if ~isempty(brain_file)
  h = figure;
  set(h, 'Menubar', 'none');
  set(h, 'Name', 'Head model/Sensor location', 'NumberTitle', 'off');
  set(h, 'Unit', 'normalized', 'Position', [0.3, 0.3, 0.5, 0.5]);
  hold on; flag = 1;
  V = vb_load_cortex([proj_root '/' brain_file], 'subj');
  plot3(V(:,1), V(:,2), V(:,3), 'b.');
  rotate3d(h, 'on');
  set(gca, 'Color', 'k');
  axis equal;

end

% Plot sensor
if ~isempty(meg_file)
  if ~flag, figure; hold on; flag = 1; end
  pick = vb_load_sensor([proj_root '/' meg_file]);
  if ~isempty(pick)
      plot3(pick(:,1),pick(:,2),pick(:,3),'bo');
  end
  axis equal;
end

if bem_mode == define.BASIS_MEG_SPHERE
  % MEG sphere
elseif bem_mode == define.BASIS_EEG_SPHERE
  % EEG sphere
elseif ~isempty(head_file)
  if ~flag, figure; hold on; flag = 1; end
  head_file = [proj_root '/' head_file];
  Nlayer = vb_head_get_layer_num(head_file);
  
  for l=1:Nlayer
    headinfo = vb_head_get_specified_headinfo(head_file,l);
    patch('Faces',headinfo.Fhead,'Vertices',headinfo.Vhead,...
          'FaceColor',head_color(l,:),'EdgeColor','none',...
          'FaceAlpha',falpha(l));
  end
end

if flag==1, 
  axis equal;
  axis tight;
  xlabel('X [m]');
  ylabel('Y [m]');
  zlabel('Z [m]'); 
end
