function varargout = current_gui(varargin)
% This function is for using current gui.
% Usage :
%    key : 'set_parm', 'get_parm'
%
%      current_gui(fig, 'set_parm', <current_parm>)
%        current_parm will be set to launched gui.
%
%      current_gui(fig, 'get_parm')
%        make current_parm structure from launched gui
%        and returns current_parm.
%
% inner functions:
%   set_parm_gui
%   get_parm_gui
%   exec
%   vb_index_to_time
%   vb_time_to_index
%   subsmpl_index_to_index
%   index_to_subsmpl_index
%   data_len_to_time_len
%   time_len_to_data_len
%   callback
%   set_result_file
%   read_time_parameters
%   change_save_mode
%   select_time_window_time_course
%   reload_gui_area_file
% Related functions:
%   vb_job_current
%   job_plot_meg
%
% --- History
% ??-??-?? Ryosuke Hayashi
% 2009-04-02 Taku Yoshioka
%  Parameter name changed within this code for readability
%  (just replacing 'resultfile' to bayesfile)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%if nargin == 0
%    key = 'init';
%else
%    key = varargin{1};
%end
%
%if ~strcmp(key, 'init')
%    h = get(0, 'CurrentFigure');
%    data = figure_load_data(h);
%end
fig = varargin{1};
key = varargin{2};
data = guidata(fig);

switch(key)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % --- public
    % These interfaces are able to use from external function.
    %
    case 'set_parm'
        current_parm = varargin{3};
        data = set_parm_gui(data, current_parm);
        data.current_parm = current_parm;
    case 'get_parm'
        varargout{1} = data.current_parm(data);
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % --- protected
    % These interfaces are for current_gui.
    % Don't use from external function.
    %
    case 'callback'
        hObj = varargin{3};
        data = callback(data.H, hObj, data);
        if ~ishandle(fig), return; end
        current_parm = get_parm_gui(data);
        guidata(fig, data);
        current_gui(fig, 'merge_parm', current_parm);
    case 'merge_parm'
        merge_parm = varargin{3};
        parm_names = fieldnames(merge_parm);
        for k=1:length(parm_names)
            data.current_parm.(parm_names{k}) = merge_parm.(parm_names{k});
        end
end

% Save application data to the figure.
guidata(fig, data);

function [data] = set_parm_gui(data, current_parm)
% Set parameters to GUI function.
% [IN]
%            data : application data
%    current_parm : current parameter structure.
% [OUT]
%    data : some updated application data

    H = data.H;

    % Set default
    if isempty(current_parm) || ~exist('current_parm', 'var')
        current_parm = vb_set_current_parm;
    end

    % result file
    if isfield(current_parm, 'bayesfile') && ~isempty(current_parm.bayesfile)
        proj_root = get(H.proj_root_edit, 'String');
        if isfield(current_parm, 'megfile')
            megfile = current_parm.megfile;
        else
            megfile = [];
        end
        [err, t_parm] = set_result_file(H, proj_root, current_parm.bayesfile, megfile);
        if ~err, data.time_parm = t_parm; end
    end

    % current file
    if isfield(current_parm, 'currfile') && ~isempty(current_parm.currfile)
        [path_name, file_name, ext] = fileparts(current_parm.currfile);
        set(H.save_directory_edit, 'String', path_name);
        set(H.current_file_edit, 'String', current_parm.currfile);
        update_output_filename(H);
    end

    % trial average
    if isfield(current_parm, 'trial_average') && ~isempty(current_parm.trial_average)
        if current_parm.trial_average == ON
            change_trial_average(H, ON);
        else
            change_trial_average(H, OFF);
        end
    end

    % tsubsmpl
    if isfield(current_parm, 'tsubsmpl') && ~isempty(current_parm.tsubsmpl)
        tsubsmpl = current_parm.tsubsmpl;
        twin_msec = []; % [from, to] by msec
        twin_meg  = []; % [from, to] by sample
        step_time = data_len_to_time_len(...
                         tsubsmpl(2) - tsubsmpl(1), ...
                         data.time_parm); % sample
        Nsample = length(tsubsmpl);
        % Convert sample number to msec.
        [twin_meg(1), err, err_msg] = ...
            subsmpl_index_to_index(tsubsmpl(1), data.time_parm);
        [twin_msec(1), err, err_msg] = ...
            vb_index_to_time(twin_meg(1), data.time_parm);
        [twin_meg(2), err, err_msg] = ...
            subsmpl_index_to_index(tsubsmpl(Nsample), data.time_parm);
        [twin_msec(2), err, err_msg] = ...
            vb_index_to_time(twin_meg(2), data.time_parm);
        % set time to gui
        set(H.time_window_from_edit, 'String', sprintf('%.2f', twin_msec(1)));
        set(H.time_window_to_edit, 'String', sprintf('%.2f', twin_msec(2)));
        set(H.time_step_edit, 'String', sprintf('%.2f', step_time));
    end

    % jactdir
    if isfield(current_parm, 'jactdir') && ~isempty(current_parm.jactdir)
        if ~isempty(current_parm.jactdir)
            set(H.jact_directory_edit, 'String', current_parm.jactdir);
            change_save_mode(H, SAVE_MODE_MULTIPLE);
        end
    end

    % area_file, area_key
    if isfield(current_parm, 'areafile') && ~isempty(current_parm.areafile)
        if isfield(current_parm, 'area_key')
            reload_gui_area_file(H, current_parm.areafile, current_parm.area_key);
        else
            reload_gui_area_file(H, current_parm.areafile);
        end
    end

    % flag
    current_parm.flag = 0;

    update_exec_button_state(H);

% end of set_parm_gui()

function current_parm = get_parm_gui(data)
% Get parameters function from GUI.
% [IN]
%    data : application data
% [OUT]
%    current_parm : current parameter structure.

    H = data.H;
    
    current_parm = struct;

    % result file
    current_parm.bayesfile = get(H.result_file_edit, 'String');

    % MEG/EEG file
    meg_file = get(H.meg_file_edit, 'String');
    if ~isempty(meg_file)
        current_parm.megfile = meg_file;
    end
    
    % current file
    current_file = get(H.current_file_edit, 'String');
    current_parm.currfile = current_file;

    % trial average
    if get(H.trial_average_on_button, 'Value') == 1
        current_parm.trial_average = ON;
    else
        current_parm.trial_average = OFF;
    end

    % tsubsmpl
    twin_subsmple = [];
    twin_meg  = [];
    step_size_str = get(H.time_step_edit, 'String');
    twin_from_str = get(H.time_window_from_edit, 'String');
    twin_to_str   = get(H.time_window_to_edit, 'String');

    if ~isempty(data.time_parm)
        % Step size by sampling length
        step_size = time_len_to_data_len(...
                        str2double(step_size_str),...
                        data.time_parm);
        % msec-->index-->subsmpl_index
        [twin_meg(1), err, err_msg] = ...
            vb_time_to_index(str2double(twin_from_str), data.time_parm);
        [twin_meg(2), err, err_msg] = ...
            vb_time_to_index(str2double(twin_to_str), data.time_parm);
        [twin_subsmple(1), err, err_msg] = ...
            index_to_subsmpl_index(twin_meg(1), data.time_parm);
        [twin_subsmple(2), err, err_msg] = ...
            index_to_subsmpl_index(twin_meg(2), data.time_parm);
	
        current_parm.tsubsmpl = [twin_subsmple(1):step_size:twin_subsmple(2)];
    end

    % jactdir
    if get(H.save_file_multiple_button, 'Value') == 1
        current_parm.jactdir = get(H.jact_directory_edit, 'String');
        if isempty(current_parm.jactdir)
            current_parm.jactdir = '.';
        end
    end

    % area_file, area_key
    current_parm.areafile = get(H.area_file_edit, 'String');
    area_keys = get(H.area_key_listbox, 'String');
    if ischar(area_keys)
        current_parm.area_key = area_keys;
    else
        current_parm.area_key = area_keys{get(H.area_key_listbox, 'Value')};
    end

    % flag
    current_parm.flag = 0;


% end of get_parm_gui()

function [err] = exec(H, data)
% This function execute vb_job_current() by using gui parameters.
% [IN]
%    H : all handles of components
% [OUT]
%   err : true or false

    err = false;

    proj_root = get(H.proj_root_edit, 'String');
    current_parm = get_parm_gui(data);

    str = get(H.exec_push, 'String');
    set(H.exec_push, 'String', 'Processing...');
    drawnow;
    % Execute current estimation
    vb_job_current(proj_root, current_parm);
    set(H.exec_push, 'String', 'Exec');

% end of exec()

function [data] = callback(H, hObj, data)
% GUI event processing function.
% This function is called when user pushes, editing, any action.
% [IN]
%       H : gui component handles.
%    hObj : action component
%    data : application data
% [OUT]
%    data : some updated application data

    global vbmeg_inst;
    define = vbmeg_inst.const;

    switch(hObj)
        case H.proj_root_push
            dir_name = dir_dialog(get(H.proj_root_edit, 'String'));
            if dir_name ~= 0
                set(H.proj_root_edit, 'String', dir_name);
            end
        case H.result_file_push
            proj_root = get(H.proj_root_edit, 'String');
            [dirname, fnames, cancelled] = pa_prjfile_select(proj_root, ...
                                                             {define.BAYES_EXTENSION}, ...
                                                             'Select current variance file');
            if ~cancelled
                result_file = [dirname, filesep, fnames{1}]; % relative from proj_root
                [err, time_parm] = set_result_file(H, proj_root, result_file);
                if ~err
                    data.time_parm = time_parm;
                end
                % make output current filename from bayes.mat
                [p_, f_, e_] = vb_get_file_parts(result_file);
                ix = find(f_ == '.');
                if ~isempty(ix)
                    set(H.current_file_edit, 'String', ...
                                             [f_(1:ix(1)-1), define.CURRENT_EXTENSION]);
                end
                update_output_filename(H);
            end
        case H.meg_file_push
            proj_root = get(H.proj_root_edit, 'String');
            [dir fname] = pa_prjfile_select(proj_root, ...
                {define.MEG2_EXTENSION, define.EEG_EXTENSION}, ...
                'Select MEG/EEG file');
            if ~isempty(dir)
                meg_file = [dir, filesep, fname{1}];
                set(H.meg_file_edit, 'String', meg_file);
                result_file = get(H.result_file_edit, 'String');
                [err, time_parm] = set_result_file(H, proj_root, result_file, meg_file);
                if ~err
                    data.time_parm = time_parm;
                end
            end
        case {H.trial_average_on_button, H.trial_average_off_button}
            if hObj == H.trial_average_on_button
                if get(H.save_file_multiple_button, 'Value')
                    change_trial_average(H, OFF);
                    uiwait(...
                        msgbox(['Set Current file format = "single file" '...
                                'before changing trial average ON.'],...
                                'message', 'modal'));
                else
                    change_trial_average(H, ON);
                end
            else
                change_trial_average(H, OFF);
            end
        case {H.time_window_from_edit, H.time_window_to_edit}
            time_value = str2double(get(hObj, 'String'));
            time_index = vb_time_to_index(time_value, data.time_parm);
            time_value = vb_index_to_time(time_index, data.time_parm);
            set(hObj, 'String', sprintf('%.2f', time_value));
        case H.select_time_window_push
            time_parm = data.time_parm;
            if isempty(time_parm)
                errordlg('first, set result file.', 'error');
            else
                select_time_window_time_course(H, time_parm);
            end
        case H.time_step_edit
            [time_len] = str2double(get(hObj, 'String'));
            if time_len <= 0
                errordlg('Time step should be positive value.', 'error');
                return;
            else
                % Convert inputted time length to multiple number of
                % minimum unit of time length.
                [data_len] = time_len_to_data_len(time_len, data.time_parm);
                [time_len] = data_len_to_time_len(data_len, data.time_parm);
                set(hObj, 'String', sprintf('%.2f', time_len));
            end
        case H.area_file_push
            proj_root = get(H.proj_root_edit, 'String');
            [dirname, fnames, cancelled] = pa_prjfile_select(...
                                                    proj_root, ...
                                                    {define.AREA_EXTENSION}, ...
                                                    'Select cortical area file');
            if ~cancelled
                area_file = [dirname '/' fnames{1}];
                reload_gui_area_file(H, area_file);
            end
        case H.area_file_reset_push
            set(H.area_file_edit, 'String', []);
            set(H.area_key_listbox, 'String', []);
            set(H.area_key_listbox, 'Value', 1);
        case {H.save_file_single_button, H.save_file_multiple_button}
            if hObj == H.save_file_single_button
                % single file
                change_save_mode(H, SAVE_MODE_SINGLE);
            else
                % multiple files
                if get(H.trial_average_on_button, 'Value')
                    change_save_mode(H, SAVE_MODE_SINGLE);
                    uiwait(...
                        msgbox(['Set trial average = "OFF" ' ...
                                'before changing Current file format to Multiple files.'] ,...
                                'message', 'modal'));
                else
                    change_save_mode(H, SAVE_MODE_MULTIPLE);
                end
            end
        case H.save_directory_push
            proj_root = get(H.proj_root_edit, 'String');
            [dirname, cancelled] = pa_prjdir_select(proj_root);
            if ~cancelled
                set(H.save_directory_edit, 'String', dirname);
                update_output_filename(H);
            end
        case H.exec_push
            err = exec(H, data);
        case H.advanced_parameter_push
            this = data.H.figure;
            if ishandle(data.adv_fig)
                figure(data.adv_fig);
            else
                data.adv_fig = current_gui_advanced_new(this, data.current_parm);
            end
        %%%%%%%%
        % Menu
        case {H.figure, H.menu_close}
             current_gui_delete(data.H.figure);
             return;
        case H.menu_current_timecourse_estimation
            url = vb_users_manual_url('Current timecourse estimation');
            vb_browser_show(url);
    end

    update_exec_button_state(H);

% end of callback()

function update_exec_button_state(H)
% This function check status of gui and change exec button status.
% [IN]
%    H : handles of GUI
    button_state = 'off';
    if ~isempty(get(H.result_file_edit, 'String')) & ...
       ~isempty(get(H.time_window_from_edit, 'String')) & ...
       ~isempty(get(H.time_window_to_edit, 'String')) & ...
       ~isempty(get(H.time_step_edit, 'String')) & ...
       ~isempty(get(H.save_directory_edit, 'String')) & ...
       ~isempty(get(H.current_file_edit, 'String'))

       button_state = 'on';
%        % Save file single
%        if get(H.save_file_single_button, 'Value')
%            button_state = 'on';
%        else
%        % Save file multiple
%            if ~isempty(get(H.jact_directory_edit, 'String'))
%                button_state = 'on';
%            end
%        end
   end

   set(H.exec_push, 'Enable', button_state);

% end of update_exec_button_state()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% subsmpl_index, index, time Conversion
%
% bayes_parm.twin_meg  is meg_data index( = index)
% current_parm.tsubsmpl is bayes_parm.twin_meg index ( = subsmpl_index)

function [time, err, err_msg] = vb_index_to_time(index, time_parm)
% Convert meg_data index to meg_data time.
% [IN]
%        index : meg_data index number.
%    time_parm : time parameter.
% [OUT]
%        time : time corresponds to sampling data index. 
%         err : true or false 
%     err_msg : error reason
% [exception]
%     inputted index cannot be processed.
%     check function usage.

    time = [];
    err = false;
    err_msg = [];

    if isempty(time_parm)
        err = true;
        err_msg = 'result file is not specified.';
        return;
    end

    if ~( (time_parm.twin_meg(1) <= index) &&  (index <= time_parm.twin_meg(2)) )
        error('vb_index_to_time : inputted index is out of range.');
    end

    time = (index - time_parm.pretrigger - 1) * (1000/time_parm.sample_freq);

% end of vb_index_to_time()

function [index, err, err_msg] = vb_time_to_index(time, time_parm)
% Convert meg_data time to meg_data index.
% [IN]
%         time : meg_data time.
%    time_parm : time parameter.
% [OUT]
%      index : meg_data index number
%        err : true or false
%    err_msg : error reason
%

    index = [];
    err = false;
    err_msg = [];

    if isempty(time_parm)
        err = true;
        err_msg = 'result file is not specified.';
        return;
    end

    index = round(time * (time_parm.sample_freq/1000) + time_parm.pretrigger + 1);

    % out of range is checked and collect(set min or max).
    if index < time_parm.twin_meg(1), index = time_parm.twin_meg(1); end;
    if time_parm.twin_meg(2) < index, index = time_parm.twin_meg(2); end;

% end of vb_time_to_index()

function [data_len, err, err_msg] = time_len_to_data_len(time_len, time_parm)
% Convert time length to sampling length.
% [IN]
%     time_len : time length.
%    time_parm : time parameter.
% [OUT]
%    data_len : data length(sample)
%         err : true or false
%     err_msg : error reason

    data_len = [];
    err = false;
    err_msg = [];

    if time_len <= 0;
        err = true; 
        err_msg = 'Time length should be positive value.';
        return;
    end
    if isempty(time_parm)
        err = true;
        err_msg = 'result file is not specified.';
        return;
    end

    data_len = ceil(time_len * (time_parm.sample_freq/1000));
    twin_meg = time_parm.twin_meg;

% end of time_len_to_data_len()

function [time_len, err, err_msg] = data_len_to_time_len(data_len, time_parm)
% Convert sampling number to time length.
% [IN]
%     data_len : sampling number.
%    time_parm : time parameter.
% [OUT]
%    time_len : time length
%         err : true or false
%     err_msg : error reason
% [exception]
%     inputted data_len cannot be processed.
%     check function usage.

    time_len = [];
    err = false;
    err_msg = [];

    data_len_int = double(int64(data_len));
    if( data_len <=0 || (data_len - data_len_int) > 0 )
        error('data_len_to_time_len : data_len should be positive integer.');
    end
    if isempty(time_parm)
        err = true;
        err_msg = 'result file is not specified.';
        return;
    end

    data_len = ceil(data_len);
    time_len = data_len * (1000/time_parm.sample_freq);

% end of data_len_to_time_len()

function [subsmpl_index, err, err_msg] = index_to_subsmpl_index(index, time_parm)
% Convert meg_data index to tsubsmpl index
% [IN]
%        index : meg_data index number.
%    time_parm : time parameter.
% [OUT]
%    subsmpl_index : current_parm.tsubsmpl index.
%              err : true or false
%          err_msg : error reason
% [exception]
%     inputted index cannot be processed.
%     check function usage.

    subsmpl_index = [];
    err = false;
    err_msg = [];

    index_int = double(int64(index));
    if( index <= 0 || (index_int - index) > 0 )
        error('index_to_subsmpl_index : index should be positive integer.');
    end
    if isempty(time_parm)
        err = true;
        err_msg = 'result file is not specified.';
        return;
    end

    if ~( (time_parm.twin_meg(1) <= index) &&  (index <= time_parm.twin_meg(2)) )
        error('index_to_subsmpl_index : inputted index is out of range.');
    end
    subsmpl_index = index - time_parm.twin_meg(1) + 1;

% end of index_to_subsmpl_index()

function [index, err, err_msg] = subsmpl_index_to_index(subsmpl_index, time_parm);
% Convert curret_parm.tsubsmpl index to meg_data index
% [IN]
%    subsmpl_index : current_parm.tsubsmpl index.
%        time_parm : time parameter.
% [OUT]
%      index : meg_data index number.
%        err : true or false
%    err_msg : error reason
% [exception]
%     inputted subsmpl_index cannot be processed.
%     check function usage.

    index = [];
    err = false;
    err_msg = [];

    subsmpl_int = double(int64(subsmpl_index));
    if( subsmpl_index <= 0 || (subsmpl_index - subsmpl_int) > 0 )
        error('subsmpl_index_to_index : subsmple_index should be positive integer.');
    end
    if isempty(time_parm)
        err = true;
        err_msg = 'result file is not specified.';
        return;
    end

    index = ceil(subsmpl_index) + time_parm.twin_meg(1) - 1;
    if ~( (time_parm.twin_meg(1) <= index) &&  (index <= time_parm.twin_meg(2)) )
        error('subsmpl_index_to_index : inputted subsmpl_index is out of range.');
    end

% end of subsmpl_index_to_index()

function [err, time_parm] = set_result_file(H, proj_root, result_file, meg_file)
% This function read result file and initialize gui of timewindow part.
% [IN]
%              H : gui component handles
%      proj_root : project root directory
%    result_file : result_file(relative path from proj_root)
%       meg_file : external meg file(optional:relative path from prj_root)
%
% [OUT]
%          err : true(file read failed.) or false
%    time_parm : time parameter

    err = false;

    % Read time parameter from result file
    if ~exist('meg_file', 'var')
        meg_file = [];
    end
    [time_parm, err, err_msg] = read_time_parameters(proj_root, result_file, meg_file);
    if err
        errordlg(['result file read error occured -- detail :' err_msg], 'error');
    else
        %
        %--- Initialize gui of timewindow part.
        %

        % result file name
        set(H.result_file_edit, 'String', result_file);

        % time window
        [time] = vb_index_to_time(time_parm.twin_meg(1), time_parm);
        set(H.time_window_from_edit, 'String', sprintf('%.2f', time)); 
        [time] = vb_index_to_time(time_parm.twin_meg(2), time_parm);
        set(H.time_window_to_edit, 'String', sprintf('%.2f', time));

        % time step
        [time_len] = data_len_to_time_len(1, time_parm); % 1 sample step
        set(H.time_step_edit, 'String', num2str(time_len-0.001));
        
        % meg_file
        set(H.meg_file_edit, 'String', time_parm.meg_file);
        set(H.meg_file_edit, 'Enable', 'on');
        set(H.meg_file_push, 'Enable', 'on');
    end

% end of set_result_file()

function [time_parm, err, err_msg] = read_time_parameters(proj_root, result_file, meg_file)
% This function read result file and meg_file and make 'time_parm' structure.
% [IN]
%       proj_root : project root directory
%     result_file : result file name(relative path from proj_root)
%        meg_file : external meg file input(relative path from proj_root)
% [OUT]
%      time_parm : time parameters
%            err : true(failed to read file) or false.
%        err_msg : error reason string.
%    

    time_parm = [];
    err = false;
    err_msg = [];
    if ~exist('meg_file', 'var')
        meg_file = [];
    end

    % load bayes_parm
    result_file = [proj_root, '/', result_file]; % absolute path
    
    if ~exist(result_file, 'file'), 
        err = true;
        err_msg = 'result file does not exist.';
        return;
    end;
    load(result_file, 'bayes_parm');

    % load Meg data time parameter.
    if isempty(meg_file)
        meg_file = bayes_parm.megfile{1};
    end


    meg_file_abs = [proj_root, '/', meg_file];    % absolute path
    if ~exist(meg_file_abs, 'file')
        err = true;
        err_msg = 'megfile does not exist.';
        return;
    end
    [MEGinfo] = vb_load_meg_info(meg_file_abs);

    % make time parameter
    time_parm = struct;
    time_parm.meg_file    = meg_file;
    time_parm.pretrigger  = vb_meginfo_get_pre_trigger(MEGinfo);
    time_parm.sample_freq = vb_meginfo_get_sampling_frequency(MEGinfo);
    time_parm.Nsample     = vb_meginfo_get_sample_number(MEGinfo);
    time_parm.twin_meg    = bayes_parm.twin_meg;

% end of read_time_parameters()

function change_save_mode(H, mode)
% This function changes save file mode on gui.
% [IN]
%       H : component handles of gui
%    mode : SAVE_MODE_SINGLE or SAVE_MODE_MULTIPLE

    switch(mode)
        case SAVE_MODE_SINGLE
            set(H.save_file_single_button, 'Value', 1);
            set(H.save_file_multiple_button, 'Value', 0);
            set(H.jact_directory_text, 'Visible', 'off');
            set(H.jact_directory_text2, 'Visible', 'off');
            set(H.jact_directory_edit, 'Visible', 'off');
        case SAVE_MODE_MULTIPLE
            set(H.save_file_single_button, 'Value', 0);
            set(H.save_file_multiple_button, 'Value', 1);
            % 20091008 rhayashi visible on
            set(H.jact_directory_text, 'Visible', 'on');
            set(H.jact_directory_text2, 'Visible', 'on');
            set(H.jact_directory_edit, 'Visible', 'on');
    end

% end of change_save_mode()

function update_output_filename(H)
% This function update output filenames(current file, jact directory).
% [IN]
%       H : component handles of gui.

    save_dir = get(H.save_directory_edit, 'String');

    current_file_edit = get(H.current_file_edit, 'String');
    jact_dir_edit = get(H.jact_directory_edit, 'String');

%    [path_name, current_file, ext_cur] = fileparts(current_file_edit);
%    [path_name, jact_dir] = fileparts(jact_dir_edit);
    [path_name, current_file, ext_cur] = vb_get_file_parts(current_file_edit);
    [path_name, jact_dir] = vb_get_file_parts(jact_dir_edit);

    set(H.current_file_edit, 'String', [save_dir, filesep, current_file, ext_cur]);
%    set(H.jact_directory_edit, 'String', [save_dir]);

% end of update_output_filename()

function change_trial_average(H, mode)
% This function changes trial average mode on gui.
% [IN]
%       H : component handles of gui.
%    mode : ON or OFF

    switch(mode)
        case ON
            set(H.trial_average_on_button, 'Value', 1);
            set(H.trial_average_off_button, 'Value', 0);
        case OFF
            set(H.trial_average_on_button, 'Value', 0);
            set(H.trial_average_off_button, 'Value', 1);
    end

% end of change_trial_average()

function select_time_window_time_course(H, time_parm)
% This function launch job_plot_meg and set selected time window to gui.
% [IN]
%            H : component handles
%    time_parm : 
    %
    % --- Time window setting in current_gui.
    %
    twin_ini = [];
    [twin_ini(1)] = vb_time_to_index(...
                    str2double(get(H.time_window_from_edit, 'String')), time_parm);
    [twin_ini(2)] = vb_time_to_index(...
                    str2double(get(H.time_window_to_edit, 'String')), time_parm);
    %
    %  --- launch time window selection gui
    %

    % hide current_gui figure handle not to be painted from
    % job_plot_meg.
    gui_cancelled = true;
    set(H.figure, 'HandleVisibility', 'off'); 
    try
        proj_root = get(H.proj_root_edit, 'String');
        meg_file  = [proj_root, '/', time_parm.meg_file];
        [twin_meg, gui_cancelled] = ...
            job_plot_meg(meg_file, ...          % meg file
                        2, ...                  % intaraction mode
                        time_parm.twin_meg, ... % selectable range
                        twin_ini);              % initial selected range
    catch
        [last_msg, last_id] = lasterr;
        errordlg(last_msg, 'error');
    end
    set(H.figure, 'HandleVisibility', 'on');
    % set time window from job_plot_meg selection
    if ~gui_cancelled
        [time] = vb_index_to_time(twin_meg(1), time_parm);
        set(H.time_window_from_edit, 'String', sprintf('%.2f', time));
        [time] = vb_index_to_time(twin_meg(2), time_parm);
        set(H.time_window_to_edit, 'String', sprintf('%.2f', time));
    end

% end of select_time_window_time_course()

function reload_gui_area_file(H, area_file, area_key)
% This function sets area_file data to gui.
% [IN]
%            H : gui component handles.
%    area_file : area file(relative path)
%  --- optional
%     area_key : area_key string. 
%                If area_keys match this string, set focus.

    % set gui to area filename
    set(H.area_file_edit, 'String', area_file);

    % get area keys
    keys = {};
    proj_root = get(H.proj_root_edit, 'String');
    area_file = [proj_root '/' area_file];

    if exist(area_file, 'file') ~= 2
        return;
    end

    load(area_file, 'Area');
    for n = 1 : length(Area)
        keys{n} = Area{n}.key;
    end

    % search focus line when area_key is specified.
    if exist('area_key', 'var')
        focus_line = strmatch(area_key, keys, 'exact');
        if isempty(focus_line), focus_line = 1; end;
    else
        focus_line = 1;
    end

    % set area keys to the popup
    set(H.area_key_listbox, 'String', keys);
    set(H.area_key_listbox, 'Value', focus_line);

% end of reload_gui_area_file()

%
%  --- Define constants
%
function v = DEFAULT_CURRENT_FNAME
    v = 'current.curr.mat';
function v = SAVE_MODE_SINGLE
    v = 1;
function v = SAVE_MODE_MULTIPLE
    v = 2;
function v = OVERLAPPED_MODE
    v = 1;
function v = CONCATENATE_MODE
    v = 2;
function v = APPLICATION_NAME
    v = 'Current estimation';
