function [info_cell] = bm_manager_get_model_info(obj, index)
% get brain model information
% [USAGE]
%    [info] = bm_manager_get_model_info(<obj>, <index_no>);
% [IN]
%      obj : bm_manager object
%    index : bm_list index
% [OUT]
%    info_cell : cell array of info struct.
%    info_cell{k}.model_name  : <<string>>  model name
%                .class_type  : <<string>>  brain_data.class_type
%                .conductivity: <<double>>  conductivity value
%               [.surf_close] : This field exists when class_type is 'surf_data'
%                               <<boolean>> true / false
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('obj', 'var'), error('obj is a required parameter.'); end
if ~strcmp(obj.class_type, 'bm_manager')
    error('class type is invalid.');
end
if ~exist('index', 'var'), error('index is a required parameter.'); end

%
% --- Main Procedure
%
brain_data_cell = bm_list_get_brain_model(obj.bm_list, index);

% allocate values to return
Nmodel = length(brain_data_cell);
info_cell   = cell(Nmodel, 1);

for k=1:Nmodel
    info = struct;
    brain_data = brain_data_cell{k};

    % get model info
    info.model_name = brain_data_get_model_name(brain_data);

    % get conductivity
    info.conductivity = brain_data_get_conductivity(brain_data);

    % class type
    switch(brain_data.class_type)
        case 'surf_data'
            info.class_type = 'surf_data';
        case 'mask_image_data'
            info.class_type = 'mask_image_data';
        otherwise
            error('Unknown class_type is specified.');
    end

    % surface close status
    if strcmp(brain_data.class_type, 'surf_data')
        info.surf_close = surf_data_is_closed(brain_data);
    end

    % set model info to return value
    info_cell{k} = info;
end
