function varargout = bayes_parm_editor(obj, command, varargin)
% bayes_parm_editor controller
% [USAGE]
%    [out] = bayes_parm_editor(obj, command, someparm);
% [IN]
%         obj : bayes_parm_editor object
%     command : command to the object
%    varargin : 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('obj', 'var') || isempty(obj) || ~ishandle(obj)
    error('invalid obj is specified.');
end
if ~exist('command', 'var') || isempty(command)
    error('command is a required paramteter.');
end

%
% --- Main Procedure
%

% load data
data = guidata(obj);

switch(command)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % GUI initialize/exit function
    %
    case 'finish_request'
        bayes_parm_advanced_close(data.adv_fig);
        bayes_parm_basic_close(data.basic_fig);
        return;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % bayes_parm getter/setter functions
    %
    case 'merge_bayes_parm'
        parms = varargin{1};
        parm_names = fieldnames(parms);
        for k=1:length(parm_names)
            data.bayes_parm.(parm_names{k}) = parms.(parm_names{k});
        end
    case 'set_a0_act_auto'
        data.a0_act_auto = varargin{1};
    case 'get_a0_act_auto'
        varargout{1} = data.a0_act_auto;
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % meg_file data getter/settter functions
    %
    case 'set_megfile_info'
        filename = varargin{1};
        if isempty(filename)
            data.meg_info = [];
        else
            data.meg_info = vb_load_meg_info(filename);
            pre_trigger = vb_meginfo_get_pre_trigger(data.meg_info);
            if isempty(pre_trigger)
                data.meg_info = vb_meginfo_set_pre_trigger(data.meg_info, 1);
            end
        end
    case 'get_megfile_info'
        varargout{1} = data.meg_info;
        
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % sensor_noise_file data getter/settter functions
    %
    case 'set_sensor_noise_file_info'
        filename = varargin{1};
        if isempty(filename)
            data.sensor_noise_info = [];
        else
            data.sensor_noise_info = vb_load_meg_info(filename);
            pre_trigger = vb_meginfo_get_pre_trigger(data.sensor_noise_info);
            if isempty(pre_trigger)
                data.sensor_noise_info = ...
                    vb_meginfo_set_pre_trigger(data.sensor_noise_info, 1);
            end
        end
    case 'get_sensor_noise_file_info'
        varargout{1} = data.sensor_noise_info;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % data save/load functions
    %
    case 'save_bayes_parm'
        % Why bayes_parm is not assigned to H.bayes_parm,
        % User inputted value bayes_parm.a0_act will be lost.
        % If user checked a0_act automatic setting ON, and 
        % a0_act value was inputted, User may change
        % a0_act automatic setting OFF.
%        bayes_parm_advanced_close(data.adv_fig, true);
%        % data is updated previous line (by advanced parameter gui close).
%        % reload data.
%        data = guidata(obj);
        bayes_parm = bayes_parm_check(data, data.bayes_parm);
        
        [dirname, filename, selected] = bayes_parm_gui_util(...
                                                'filedlg',...
                                                {'.bayes_param.mat'},...
                                                1, pwd);
        proj_root = bayes_parm_basic_gui(data.basic_fig, 'get_project_root');
        if selected
            filename = [dirname, filesep, filename{1}];
            vb_save(filename, 'bayes_parm', 'proj_root');
            %msgbox('save finished.', 'notice');
        end
    case 'load_bayes_parm'
%        bayes_parm_advanced_close(data.adv_fig, true);
%        % data is updated previous line (by advanced parameter gui close).
%        % reload data.
%        data = guidata(obj);
        [dirname, filename, selected] = bayes_parm_gui_util(...
                                                'filedlg',...
                                                {'.bayes_param.mat'},...
                                                0, pwd);
        if selected
            filename = [dirname, filesep, filename{1}];
            load(filename, 'bayes_parm', 'proj_root');
            data.bayes_parm = bayes_parm;
            if isempty(data.bayes_parm.a0_act)
                data.a0_act_auto = 1;
            else
                data.a0_act_auto = 0;
            end
            bayes_parm_basic_gui(data.basic_fig, 'set_project_root', proj_root);
            bayes_parm_basic_gui(data.basic_fig, ...
                                'gui_update', data.bayes_parm);
            % The data of sensornoise_info and meg_info data are updated
            % previous line (by setting basic parameter).
            % reload here.
            data_tmp = guidata(obj);
            data.sensor_noise_info = data_tmp.sensor_noise_info;
            data.meg_info          = data_tmp.meg_info;
    end
        
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % execute functions
    %        
    case 'exec_estimation'
        % Why bayes_parm is not assigned to H.bayes_parm,
        % User inputted value bayes_parm.a0_act will be lost.
        % If user checked a0_act automatic setting ON, and 
        % a0_act value was inputted, User may change
        % a0_act automatic setting OFF.
 %       bayes_parm_advanced_close(data.adv_fig, true);
 %       % data is updated previous line (by advanced parameter gui close).
 %       % reload data.
 %       data = guidata(obj);
        [bayes_parm, err, errormsg] = bayes_parm_check(data, data.bayes_parm);
        if err, errordlg(errormsg, 'error');, return; end
        proj_root = bayes_parm_basic_gui(data.basic_fig, 'get_project_root');
        vb_job_vb(proj_root, bayes_parm);
 
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % time <--> sampling data conversion functions
    %        
    case 'time_to_index'
        time = varargin{1}; % time point(msec)
        type = varargin{2}; % 'meg' or 'noise' file
        [index, err] = time_to_index_inner(data, time, type);
        varargout{1} = index;
        varargout{2} = err;
    case 'index_to_time'
        index = varargin{1}; % data index
        type = varargin{2}; % 'meg' or 'noise' file
        [time, err] = index_to_time_inner(data, index, type);
        varargout{1} = time;
        varargout{2} = err;
    case 'time_len_to_data_len'
        time_len = varargin{1}; %time length(msec)
        type = varargin{2}; % 'meg' or 'noise' file
        [data_len, err] = time_len_to_data_len(data, time_len, type);
        varargout{1} = data_len;
        varargout{2} = err;
    case 'data_len_to_time_len'
        data_len = varargin{1}; %data length
        type = varargin{2}; % 'meg' or 'noise' file
        [time_len, err] = data_len_to_time_len(data, data_len, type);
        varargout{1} = time_len;
        varargout{2} = err;
end

% save data
guidata(obj, data);

function [index, err] = time_to_index_inner(data, time, file_type)
% This function returns nearest rounded up sampling index from
% specified time.
% [IN]
%         data : application data (@see bayes_parm_editor_init)
%         time : time point(msec)
%    file_type : 'meg'   (use meg_file)
%                'noise' (use sensor_noise_file)
% [OUT]
%      index : nearest rounded up sampling index from specified time.
%      err       : 1 or 2 (1:file does not specified, 2:user input error)

    err = false;
    index = [];
    
    meg_info = [];
    switch(file_type)
        case 'meg'
            meg_info = data.meg_info;
        case 'noise'
            meg_info = data.sensor_noise_info;
    end
    
    if isempty(meg_info)
        err = 1;
        return;
    end

    try
      index = vb_time_to_index(time,meg_info);
        % sampling length(msec).
        %sampling_length = 1000/sample_freq;
        %
        % time list of sample points
        %sampling_point = -(sampling_length * pretrigger) : sampling_length : ...
        %                  (sampling_length * (Nsample-pretrigger-1));
        %
        %for k=1:length(time)
        %    if time(k) < min(sampling_point)
        %        time(k) = min(sampling_point);
        %    elseif time(k) > max(sampling_point)
        %        time(k) = max(sampling_point);
        %    end
        %
        %    % Index of MEG data
        %    index(k) = min(find(sampling_point>=time(k)));
        %end
    catch
        err = 2;
    end

function [time, err] = index_to_time_inner(data, index, file_type)
% This function returns time that corresponds to sampling data index. 
% [IN]
%         data : application data (@see bayes_parm_editor_init)
%        index : sampling data index number.
%    file_type : 'meg'   (use meg_file)
%                  'noise' (use sensor_noise_file)
% [OUT]
%    time      : time that corresponds to sampleing data index. 
%    err       : 1 or 2 (1:file does not specified, 2:user input error)

    err = false;
    time = [];
    
    meg_info = [];
    switch(file_type)
        case 'meg'
            meg_info = data.meg_info;
        case 'noise'
            meg_info = data.sensor_noise_info;
    end

    if isempty(meg_info)
        err = 1;
        return;
    end        

    try
      time = vb_index_to_time(index,meg_info);
        %% sampling length(msec).
        %sampling_length = 1000/sample_freq;
        %% time list of sample points(msec)
        %sampling_point = -(sampling_length * pretrigger) : sampling_length : ...
        %                    (sampling_length * (Nsample-pretrigger-1));
        %
        %for k=1:length(index)
        %    % array over flow check
        %    if index(k) > length(sampling_point)
        %        index(k) = length(sampling_point);
        %    end
        %    % return time
        %    time(k) = sampling_point(index(k));
        %end
    catch
        err = 2;
        return;
    end

function [data_len, err] = time_len_to_data_len(data, time_len, file_type)
% This function converts from time length to sampling length.
% [IN]
%         data : application data (@see bayes_parm_editor_init)
%     time_len : time length(msec)
%    file_type : 'meg'   (use meg_file)
%                'noise' (use sensor_noise_file)
% [OUT]
%    data_len : sampling length
%         err : 1 or 2 (1:file does not specified, 2:user input error)

    err = false;
    data_len = [];
    
    meg_info = [];
    switch(file_type)
        case 'meg'
            meg_info = data.meg_info;
        case 'noise'
            meg_info = data.sensor_noise_info;
    end

    if isempty(meg_info)
        err = 1;
        return;
    end        

    try
      sample_freq     = vb_meginfo_get_sampling_frequency(meg_info);
      data_len = vb_time_to_numsamp(time_len,sample_freq);
        
        % time length adjust(multiple of sampling_length)
        %sampling_length = 1000/sample_freq;
        %data_len = ceil(time_len / sampling_length);
    catch
        err = 2;
    end

function [time_len, err] = data_len_to_time_len(data, data_len, file_type)
% This function converts from sampling length to time length.
% [IN]
%         data : application data (@see bayes_parm_editor_init)
%     data_len : sampling length
%    file_type : 'meg'   (use meg_file)
%                'noise' (use sensor_noise_file)
% [OUT]
%    time_len  : time length(msec)
%         err  : 1 or 2 (1:file does not specified, 2:user input error)

    err = false;
    time_len = [];
    
    meg_info = [];
    switch(file_type)
        case 'meg'
            meg_info = data.meg_info;
        case 'noise'
            meg_info = data.sensor_noise_info;
    end

    if isempty(meg_info)
        err = 1;
        return;
    end        

    try
        sample_freq     = vb_meginfo_get_sampling_frequency(meg_info);
        time_len = vb_numsamp_to_time(data_len,sample_freq);
        %sampling_len = 1000/sample_freq;
        %time_len = sampling_len .* data_len;x
    catch
        err = 2;
    end

function [bayes_parm, err, errormsg] = bayes_parm_check(data, bayes_parm)
% This function check and repair bayes_parm.
% Before saving or execute estimation, call this function.
% [IN]
%          data : application data (@see bayes_parm_editor_init)
%    bayes_parm : structure of bayes parameter.
% [OUT]
%      bayes_parm  : checked structure of bayes parameter.
%      err         : check result(true: ex.parameter dependency error)
%                                (false: This parameter can use for
%                                estimation)

    global vbmeg_inst;
    const = vbmeg_inst.const;

    err = false;
    errormsg = [];
    
    % project root directory
    proj_root = bayes_parm_basic_gui(data.basic_fig, 'get_project_root');
    
    % basis_file_global
    if ~isfield(bayes_parm, 'basisfile_global') | isempty(bayes_parm.basisfile_global)
        bayes_parm.basisfile_global = bayes_parm.basisfile;
    end
    
    % area_key_global
    if ~isfield(bayes_parm, 'area_key_global') | isempty(bayes_parm.area_key_global)
        bayes_parm.area_key_global = bayes_parm.area_key;
    end
    
    % Rfilt_global
    if ~isfield(bayes_parm, 'Rfilt_global') | isempty(bayes_parm.Rfilt_global)
        bayes_parm.Rfilt_global = bayes_parm.Rfilt;
    end
    
    % a0_act
    if data.a0_act_auto == 1
        bayes_parm.a0_act = [];
    end

    % When soft_mode = ON, previous leadfield calculation parameter should be
    %                      basis_parm.Basis_mode = CURRENT_3D;
    %                                .normal_mode = NORMAL_NEIGHBOR_MEAN;
    if bayes_parm.soft_mode == ON
        project_root = bayes_parm_basic_gui(data.basic_fig, 'get_project_root');
        for k=1:length(bayes_parm.basisfile)
            basis_file = [project_root, '/', bayes_parm.basisfile{k}];
            [Norient, normal_mode] = vb_load_basis_info(basis_file);
            if ~((Norient == const.CURRENT_3D ) && (normal_mode == const.NORMAL_NEIGHBOR_MEAN))
                err = true;
                errormsg = sprintf(['When you use soft_mode, specified basis_parm should be as follows.\n',...
                        'basis_parm.normal_mode = NORMAL_NEIGHBOR_MEAN\n',...
                        'basis_parm.Basis_mode = CURRENT_3D\n']);
            end
        end
    end
    
    % Time window(Manual mode) error check
    if length(bayes_parm.Tnext) == 0 || length(bayes_parm.Tperiod) == 0
        err = true;
        errormsg = 'Check Analysis time parameters.';
    end
    
    % File exist check(.brain.mat, .area.mat, .act.mat)
    brain_file = [proj_root, '/', bayes_parm.brainfile];
    area_file  = [proj_root, '/', bayes_parm.areafile];
    act_file   = [proj_root, '/', bayes_parm.actfile];
    
    if exist(brain_file, 'file') ~= 2
        err = true;
        errormsg = sprintf('Cortical model file not specified.');
    end
    if exist(area_file, 'file') ~= 2
        err = true;
        errormsg = sprintf('Cortical area file not specified.');
    end
    if exist(act_file, 'file') ~= 2
        err = true;
        errormsg = sprintf('Cortical activity file not specified.');
    end
    
    % File exist check(.basis.mat)
    Nbasis_file = length(bayes_parm.basisfile);
    if Nbasis_file == 0
        err = true;
        errormsg = sprintf('Leadfield file not specified.');
    end
    for k=1:Nbasis_file
        basis_file = [proj_root, '/', bayes_parm.basisfile{k}];
        if exist(basis_file, 'file') ~= 2
            err = true;
            errormsg = sprintf('Leadfield file not specified.');
            break;
        end
    end
    
    % File exist check(.meg.mat)
    Nmeg_file = length(bayes_parm.megfile);
    if Nmeg_file == 0
        err = true;
        errormsg = sprintf('MEG/EEG file not specified.');
    end
    for k=1:Nmeg_file
        meg_file = [proj_root, '/', bayes_parm.megfile{k}];
        if exist(meg_file, 'file') ~= 2
            err = true;
            errormsg = sprintf('MEG/EEG file not specified.');
            break;
        end
    end
    w = bayes_parm.prior_weight;
    if(w < 0 || w > 1.0)
        err = true;
        errormsg = sprintf('Prior weight takes a value within range of 0-1.0');
    end

    % Output directory/file check
    [p_, f_, e_] = vb_get_file_parts([proj_root, '/', bayes_parm.bayesfile]);
    if exist(p_, 'dir') ~= 7 || isempty(f_) || isempty(e_)
        err = true;
        errormsg = sprintf('Output setting for Current variance file is invalid.');
    end
    
    % Project root directory check
    if exist(proj_root, 'dir') ~= 7
        err = true;
        errormsg = sprintf('Project root directory is invalid.');
    end
    
