function vb_fmri_project_PSmap_cortex(brain_file,act_file,area_file,PSthres,Rthres,SPMimginfo,loadVmni)
% PROJECT VOXEL PERCENT SIGNAL MAP ON THE BRAIN SURFACE MODEL
%
% [Notice]
% con_*.img must be coregistered to T1-structural image used for creating
% a brain file
%
% [Syntax]
%  vb_fmri_project_PSmap_cortex(brain_file,act_file,area_file,PSthres,SPMimginfo)
%
% [Example]
% > brain_file = 'hoge.brain.mat';
% > act_file = 'hoge.act.mat';
% > area_file = 'hoge.area.mat';
% > PSthres    = 1.0; 
% > Rthres   = 3;
% > SPMimginfo = struct('dirname', '/home/hoge/hoge/spm/',...
% > imgfilename,{'con_0001.img','con_0002.img'},imglabel,{'Right index',... 
% > 'Right hand gripping'},'basefilename','beta0010.img','2','1.17');
% > vbdot_fmri_project_PSmap_cortex(brain_file,act_file,areafile,PSthres,Rthres,SPMimginfo)
%
% [Input]
%   brain_file : cortical surface model 
%   act_file   : act file 
%   area_file  : area file
%   PSthres    : PS-value threshold (%)
%   Rthres     : radius threshold (mm)
%   SPMimginfo(ii) : arrays of struct containing information of T-image  
%       .dirname : directory name for image file 
%       .imgfilename :  file name of image to be projected  
%       .imglabel    :  label of image to be projected. These label
%                            are used as actkey and areakey
%       .baseimgfilename :  file name for the baseline image
%       .ncon         :  the number of positive contrast for each CON image
%       .peak         :  regressor peak value
% 
% [Optional]
%  loadVmni : if specfied as 1, loading coordinate of brain file as MNI coordinate  
%
% [Output]
%   Acts are added to "act_file" and Areas are added to "area_file"
%
% Reference : see http://cibsr.stanford.edu/documents/FMRIPercentSignalChange.pdf 
%    
%
% 2016/07/12 O.Yamashita
% * add loadVmni to input arguments to switch coordinate of brain file
% 2016/07/06 O.Yamashita
% * solve the region expansion problem
% 2016/07/01 O.Yamashita
% * first version based on vbdot_fmri_project_Tmap_corext.m
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

fprintf('============== NOTICE ==========================\n')
fprintf('Please confirm con_*.img have been coregistered \n')
fprintf('to the T1-structural image used for creating \n');
fprintf('a brain file. \n');
fprintf('================================================\n')

if nargin < 7
    loadVmni = 0;
end

spmdir = SPMimginfo(1).dirname;

fprintf('Image directory name : %s,...\n', spmdir);
for jj = 1 : length(SPMimginfo),
    fprintf('%s will be registered as %s ...\n', SPMimginfo(jj).imgfilename, SPMimginfo(jj).imglabel);
end

% scale factor for con image

for jj = 1 : length(SPMimginfo),
    loadimgfile = [spmdir SPMimginfo(jj).imgfilename];
    key = SPMimginfo(jj).imglabel;
    
    baseimgfile = [spmdir SPMimginfo(jj).baseimgfilename];
    [B] = vb_load_analyze_to_right(baseimgfile);
    bmean = mean(B(~isnan(B(:))));
    scale = 1./SPMimginfo(jj).ncon/bmean*SPMimginfo(jj).peak*100;

    % surface mesh
    if loadVmni
        V = vb_load_cortex(brain_file, 'MNI');
    else
        V = vb_load_cortex(brain_file, 'subj');
    end
    V = V * 1000;  % mm
      
    
    % vox image and coordinates
    VG=spm_vol(loadimgfile);
    [map,XYZ] = spm_read_vols(VG);
    map = map * scale;

    %
    % thresholding affects resulting surface image !!!
    %
%     map(abs(map) < PSthres) = NaN;
%     ix=find(~isnan(map));
%     XYZ = XYZ(:,ix);
%     map = map(ix)';
    ix = find(~isnan(map));  % brain mask
    XYZ = XYZ(:,ix);  
    map = map(ix);
    map(abs(map) < PSthres) = 0; % 
    
    r2thres = Rthres^2;
    
    
    % vox --> surf 
    Tval = vb_voximg2surimg(XYZ',map,V, r2thres, 1);

        
    
   % add act 
    fprintf('\nAdd PS-map to %s with act-key %s \n', act_file, key)
    act.xxP = Tval;
    act.key = key;
    act.spmdir  = spmdir;
    act.imgname = SPMimginfo(jj).imgfilename;
    vb_add_act(act_file, act, [], OFF);

%       % add area 
%     fprintf('Add PS-map to %s with area-key %s \n', area_file, key)
%     area.Iextract = find(Tval~=0);
%     area.key = key;
%     area.spmdir  = spmdir;
%     area.imgname = SPMimginfo(jj).imgfilename;
%     vb_add_area(area_file, area);
    
    
end