function [Jact ,ix_act, Jbck, ix_bck, Var] ...
          = vbmeg_current_reconstruct(proj_root, bayesfile, ...
			                  ix_area, trial_average, tsubsamp, mode)
% Current reconstruction using Bayesian inverse filter
%
% --- Syntax
% [Jact ,ix_act, Jbck, ix_bck, Var] =...
%   vbmeg_current_reconstruct(proj_root, bayesfile)
% [Jact ,ix_act, Jbck, ix_bck, Var] =...
%   vbmeg_current_reconstruct(proj_root, bayesfile, ix_area)
% [Jact ,ix_act, Jbck, ix_bck, Var] =...
%   vbmeg_current_reconstruct(proj_root, bayesfile, ix_area, trial_average)
% [Jact ,ix_act, Jbck, ix_bck, Var] =...
%   vbmeg_current_reconstruct(proj_root, bayesfile, ix_area, ...
%                                        trial_average, tsubsamp)   
%
% --- Input
% bayesfile : result file obtained by VB estimation
%
% MEG data : B{Nsession}(Nsensors, Tsample, Ntrials)
%
% --- Optional Input
% trial_average = [ON] : average current over all sessions 
%               = OFF  : current for each session
%
% ix_area : Position index to calculate estimated current
%           If 'ix_area' is empty or not given, 
%              currents in the active region are calculated
% tsubsamp   : Specify subsampled time index
%           If 'tsubsamp' is empty or not given, 
%              time subsampling is not done
%
% ---Output
% Jact    : active current
% ix_act  : Vertex index corresponding to active current
%
% Jact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'
% Jact(Nact,Nsample)          for trial_average = ON 
% Jact(Nact,Nsample,Ntrials)  for trial_average = OFF
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrials  : # of trials in all session]
%
% Jbck     : background current
% ix_bck   : Vertex index corresponding to background current
%            same for 'Jact'
% Var      : posterior variance of act_area
%   When active area is empty,
%   posterior variance for background area is calculated
%
% 2004-9-25 M. Sato
% 2004-12-5 M. Sato : Modified
% 2004-12-22 M. Sato : Modified
% 2005-1-5  M. Sato Modified
% 2005-1-15  M. Sato Modified
% 2005-08-19 O. Yamashita
% * Posterior variance is calculated simultaneously
% * Variables are renamed
% * Corresponding to new bayesfile format
% 2005-08-19 O. Yamashita ver.30b
% 2005-09-30 O. Yamashita
% * Minor bug fix when Lact >= 2 & Lact_var = 1
% 2006-08-31 M. Sato
% * Remove focal area from global area
% * VBMEG result & liedfield is loaded by vb_load_result
% * Inverse filter calculation is done in vb_invfilter_calc
% 2006-09-03 M. Sato
% * Non-overlapped concatenation mode is added for spectral analysis
% 2009-04-02 Taku Yoshioka
%  Parameter name changed within this code for readability
%  (just replacing 'resultfile' to bayesfile)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


if ~exist('trial_average','var'), trial_average = ON; end;

if ~exist('ix_area','var'),  ix_area = []; end;
if ~exist('tsubsamp','var'), tsubsamp = []; end;
if ~exist('mode','var'),     mode = 0; end;

error('This function is a canditate to be removed, but invoked.');

%%%%%% load result and MEG data
[B, Model, nGact, nGall, nGGall, Cov, Wact, Wbck, ...
    bsnorm, ix_act, ix_bck, vb_parm] ...
    = vb_load_result(proj_root, bayesfile, ix_area);

% B      : MEG data
% Model  : Estimated model
% nGact  : Normalized leadfield of focal window
% nGall  : Normalized leadfield of global window
% Cov    : Sensor noise covariance matrix
% bsnorm : Normalizing constant
% Wact   : Spatial smoothing matrix of focal window
% Wbck   : Spatial smoothing matrix of global window
% ix_act : Vertex index corresponding to active current
% ix_bck : Vertex index corresponding to background current

% # of vertex
Nvact	   = vb_parm.Nvact;	     % # of active vertex 
Njact_area = vb_parm.Njact_area;
Njall_area = vb_parm.Njall_area;

%% Constant
Nsession = vb_parm.Nsession;	% Number of sessions
Ntrials  = zeros(1,Nsession);
for n = 1:Nsession 
   Ntrials(n)  = size(B{n},3);
end
Ntotal   = sum(Ntrials);	    % Total number of trials in all sessions

Tsample  = vb_parm.Tsample;	    % Number of total time sample
Twindow  = vb_parm.Twindow;

% Temporal subsampling index
if isempty(tsubsamp),
	tsubsamp = 1:Tsample;
end

% Temporal smoothing window weight
[Tweight ,Tindex, Nindex, Nsample] = ...
    vb_overlapped_timewindow_weight(Twindow, Tsample, tsubsamp, mode);

Nwindow = length(Nindex);   	% # of time window

% Nsample : # of subsampled data	

if mode == 1,fprintf('Non-overlapped concatenation mode\n'); end;

%% Initialization
if trial_average == ON
	% Current averaged over trials
	Jact = zeros(Njact_area,Nsample);
	Jbck = zeros(Njall_area,Nsample);
else 
	% Current for each trial
	Jact = zeros(Njact_area,Nsample,Ntotal);
	Jbck = zeros(Njall_area,Nsample,Ntotal);
end

% Posterior variance
if Njact_area > 0
	Var = zeros(Njact_area,Nsample);
elseif Njall_area > 0
	Var = zeros(Njall_area,Nsample);
end

%% Estimated current variance
if isfield(Model,'a')
	a_inv = Model.a;
else
	a_inv = zeros(Nvact,Nwindow);
end
if isfield(Model,'v')
	v_inv = Model.v;
else
	v_inv = zeros(Nwindow,1);
end

% Time window loop
for j=1:Nwindow
	% Subsampling time index
	Tid = Tindex{j};	% subsampled time index
	Nid = Nindex{j};	% index in the total subsampled data
	Nt  = length(Tid);
	
	if Nt == 0, continue; end;
	
	if Njact_area > 0
		Var_sum = zeros(Njact_area,1);
	elseif Njall_area > 0
		Var_sum = zeros(Njall_area,1);
	end
	
	% Session loop 
	Stry = 1;
	
	for n=1:Nsession
		Ntry = Ntrials(n);

		G	 = nGact{n};   % Ga 
		GbW  = nGall{n};   % Gb*Wb' 
		GG   = nGGall{n};  % Gb*Gb'
		Covs = Cov{n};     % Sg
		Nch	 = size(G,1);
		
		%%%% Calculate Inverse filter for current reconstruction
		[KW, KW0, var_tmp] = vb_invfilter_calc(a_inv(:,j), v_inv(j), ...
		                          G, GbW, GG, Covs, Wact, Wbck);
		
		%%%% Time window smoothing
		weight = repmat(Tweight{j}, [Nch 1]);

		if trial_average == ON
		    % MEG trial_average
			Bt = sum( B{n}(:,Tid,:), 3).* weight; 
			% Current reconstruction
		    Jact(:,Nid) = Jact(:,Nid) + (KW  * Bt)/(bsnorm*Ntotal);
		    Jbck(:,Nid) = Jbck(:,Nid) + (KW0 * Bt)/(bsnorm*Ntotal);
		else
		    % MEG for each trial	
		    Bt = B{n}(:,Tid,:);	
		    for m=1:Ntry
			   Bt(:,:,m) = Bt(:,:,m) .* weight;
		    end
		    Bt = reshape( Bt, [Nch, Nt*Ntry]);

		    % Trial index
		    ixTry = [Stry:Stry+Ntry-1];
		    Stry = Stry+Ntry;

			% Current reconstruction
		    Jact(:,Nid,ixTry) = Jact(:,Nid,ixTry) ...
			         + reshape( (KW  * Bt)/bsnorm, [Njact_area, Nt, Ntry]);
		    Jbck(:,Nid,ixTry) = Jbck(:,Nid,ixTry) ...
			         + reshape( (KW0 * Bt)/bsnorm, [Njall_area, Nt, Ntry]);
		end
		
		%%%% Posterior variance calculation
		
		% trial average of inverse variance
		var_tmp = max(var_tmp,eps);
		Var_sum = Var_sum + (Ntry./var_tmp);
		
	end; % Session loop END

%	Var_sum = Var_sum/Ntotal;
	Var_sum = Model.sx(j)./(Var_sum * bsnorm^2); 
	Var(:,Nid) = Var(:,Nid) + Var_sum * Tweight{j};

end % Timewindow loop END

