function [sbasis, L, W, ix_area, ix_area_ex] = vb_prepare_leadfield( ...
    basisfile, brainfile, areafile, area_key, reduce, Rfilt, ...
    expand_spatial_filter, patch_norm, area_remove, spatial_smoother)
% PREPARE LEADFIELD MATRICES  
% 
% Processes handled in this function as follows
% 1. Area (optional)
% 2. Reduce (optional)
% 3. Spatial smoothing matrix (optional)
% 4. Load leadfield for all the sessions
% 5. If 'area_remove' is specified, this area is removed from area
%
% --- Syntax
% [Basis,ix, W, ix_ex,L] = vb_prepare_leadfield(basisfile, brainfile, ...
%    areafile, area_key, reduce, Rfilt, expand_spatial_filter, patch_norm)
% [Basis,ix, W, ix_ex,L] = vb_prepare_leadfield(basisfile, brainfile, ...
%    areafile, area_key, reduce, Rfilt, expand_spatial_filter, ...
%    patch_norm, area_remove, basis_smoother)
%
% --- Input
% See the field names of "bayes_parm"
% smoother   : basis is smoothed using 
%              = 'std'  : standard brain(nextDD, nextIX, xx, xxA)
%              = 'subj' : Individual brain(subj.nextDD, nextIX, xx, xxA)
% --- Output
% Basis{Nsession}: Lead field              (Nsensor x Ndipole)
% W              : Smoothing filter or distributed current filter
% ix_area       : Dipole indices
% ix_area_ex    : Dipole indices for distributed current. If this is
%                  not empty, W is the distributed current filter. 
% L              : Number of components of basis
%
% --- History
% 2005-08-16 
%   Okito Yamashita modified
% 2005-08-22 
%   O. Yamashita ver.30b 
% 2006-03-03 Taku Yoshioka
%   Support patch size normalization
% 2006-04-07 Kawawaki, Dai. 
% 2006-08-25 M. Sato
%   Remove area option is added
% 2006-11-11 M. Sato
%   Rmax of vb_spatial_gauss_filter is specified by 'Rfilt'
% 2008-11-28 Taku Yoshioka
%   Use vb_disp() for displaying message
% 2017-03-15 rhayashi
%   Added spatial_smoother
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

% Leadfield file names for all sessions
if iscell(basisfile),
  Nfile	= length(basisfile);
else
  tmp = basisfile;
  Nfile	= 1;
  basisfile = {tmp};
end;

vb_disp(['Number of basis files (= Number of sessions): ' ...
         num2str(Nfile)],VERBOSE_LEVEL_NOTICE);

% number of dipole vertex
[basis,L]=vb_load_basis(basisfile{1});
Nj = size(basis,1);
Nv = Nj/L; 

%%%% Area for forward model
if ~isempty(areafile) & ~isempty(area_key)
  Area = vb_get_area(areafile, area_key);
  ix_area = Area.Iextract;
  vb_disp(['Area ID: ' area_key],VERBOSE_LEVEL_NOTICE)
  vb_disp(['Number of vertices: ' num2str(length(ix_area))], ...
          VERBOSE_LEVEL_NOTICE);
else
  ix_area = [1:Nv]'; %modified by Kawawaki, Dai at 2006/04/07
  vb_disp(['Area ID: All vertices'],VERBOSE_LEVEL_NOTICE)
  vb_disp(['Number of vertices: ' num2str(length(ix_area))], ...
          VERBOSE_LEVEL_NOTICE);
end
    
%%%% Remove Area
if exist('area_remove','var') & ~isempty(areafile) & ~isempty(area_remove)
  Area = vb_get_area(areafile, area_remove);
  ix_remove = Area.Iextract;
  % Remove 'ix_remove' from 'ix_area'
  ix_area = vb_setdiff2(ix_area, ix_remove);
end

%%%% Reduce vertex
if ~isempty(reduce) & reduce ~=1, 
  ix_area = vb_reduce_cortex(brainfile, ix_area, reduce);
  vb_disp(['Number of reduced vertices: ' ...
           num2str(length(ix_area))], VERBOSE_LEVEL_NOTICE);
end

%%%% Spatial Smoothing
if ~isempty(Rfilt) & Rfilt(1) > 0
  R = Rfilt(1);
    
  if length(Rfilt) == 2,
    Rmax = Rfilt(2);
  else
    Rmax = 2*R;
  end
	
  vb_disp('--- Spatial smoothing filter calculation', ...
          VERBOSE_LEVEL_NOTICE);
  vb_disp(['R = ' num2str(R,'%2.2e') ', Rmax= ' ...
           num2str(Rmax,'%2.2e')],VERBOSE_LEVEL_NOTICE);
    
  [W,ix_area_ex] = vb_spatial_gauss_filter(...
      brainfile, R, Rmax, ix_area, expand_spatial_filter, [], [], spatial_smoother);
else
  W = speye(length(ix_area), length(ix_area));
  ix_area_ex = ix_area; 
end

vb_disp(['Number of vertices = ' num2str(length(ix_area),'%5d')], ...
        VERBOSE_LEVEL_NOTICE);
vb_disp(['Number of vertices in expanded area = ' ...
         num2str(length(ix_area_ex))],VERBOSE_LEVEL_NOTICE);

%%%% Patch size
if patch_norm==ON, 
  if strcmpi(spatial_smoother, 'std')
      [tmp1,tmp2,tmp3,tmp4,xxA] = vb_load_cortex(brainfile);
  else
      [tmp1,tmp2,tmp3,tmp4,xxA] = vb_load_cortex(brainfile, 'subj');
  end
  clear tmp1 tmp2 tmp3 tmp4
end

%%%% Load lead field matrix
sbasis = cell(Nfile,1);

for n = 1:Nfile
  % Load leadfield
  vb_disp(['Basis file for session ' num2str(n) ': ' basisfile{n}], ...
          VERBOSE_LEVEL_NOTICE);
  [basis,L]=vb_load_basis(basisfile{n});
  basis = basis';
  
  % Patch norm normalization constant
  if patch_norm==ON, 
    S = repmat(xxA(ix_area_ex)',[size(basis,1) 1]);
  else
    S = ones(size(basis(:,ix_area_ex)));
  end
  
  % Spatial Smoothing for leadfield
  switch L
   case 1, 
    basis = (basis(:,ix_area_ex).*S)*W;
   case 2, 
    basis = [(basis(:,ix_area_ex).*S)*W (basis(:,Nv+ix_area_ex).*S)*W];
   case 3,
    basis = [(basis(:,ix_area_ex).*S)*W (basis(:,Nv+ix_area_ex).*S)*W ...
	     (basis(:,2*Nv+ix_area_ex).*S)*W];
  end
  
  sbasis{n} = basis; % smoothed basis
end
