function [COV,sx0] = vb_observation_noise_specification(noise_parm)
% Specify observation noise covariance structure and its magnitude
%
% --- Syntax
% function [COV,sx0] = vb_observation_noise_specification(noise_parm)
%
% --- Input
% (Fields of noise_parm)
% .noise_model     : Integer number for specifying noise model
% .megfile_baseline: MEG file for noise calculation
% .twin_noise      : Time window for noise estimation
% .noise_reg       : Regularization parameter for noise covariance
% .temporal_filter (optional): 
% .trial_average   (optional): 
%
%  1.  trial average (optional)
%  2.  temporal filter (optional)
%  3.  calculate BB' for each session
%  4.  calculate sx0  
%
% --- History
% 2005-08-22 
%   ver.30b
% 2005-12-22 
%   bug fix (megfile --> megnoisefile)
% 2008-06-27 Taku Yoshioka 
%   (megnoisefile -> megfile_baseline)
% 2008-11-28 Taku Yoshioka
%   Use vb_disp() for displaying message
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

% field names used in this function
vb_struct2vars(noise_parm,{'noise_model','megfile_baseline','twin_noise',...
                    'noise_reg','temporal_filter','trial_average'}); 

if isempty(temporal_filter), temporal_filter = false; end
if isempty(trial_average), trial_average = false; end

megfile = megfile_baseline;

% MEG data file names given as a cell array
if iscell(megfile),
  Nfile = length(megfile); 
else
  tmp = megfile;
  megfile = {tmp};
  Nfile = 1;
end;

switch noise_model
 case 1, 
  vb_disp('Noise model: spherical',VERBOSE_LEVEL_NOTICE);
 case 2, 
  vb_disp('Noise model: diagonal',VERBOSE_LEVEL_NOTICE);
 case 3, 
  vb_disp('Noise model: full covariance',VERBOSE_LEVEL_NOTICE);
end

COV = cell(Nfile,1); % Noise covariance 
sx0 = 0;
Ndata = 0;

for n = 1:Nfile
  % Load MEG data
  vb_disp(['Load MEG data for noise estimate [' megfile{n} ']'], ...
          VERBOSE_LEVEL_NOTICE);
  bexp = vb_load_meg_data(megfile{n});
  MEGinfo = vb_load_meg_info(megfile{n});
  
  % MEG data averaging
  if trial_average, bexp = mean(bexp,3); end; 
  
  % Variable preparation
  Nch       = size(bexp,1);
  Ntrial  = size(bexp,3);
  Cov = zeros(Nch,Nch);       % Noise covariance matrix
  
  % Time window for noise evaluation
  Tnoise = twin_noise(1):twin_noise(2); 
  Nt = length(Tnoise);
  
  % Noise model
  Tstart = vb_index_to_time(Tnoise(1), MEGinfo);
  Tend = vb_index_to_time(Tnoise(end), MEGinfo);
  vb_disp(['Session ' num2str(n) ': ' num2str(Tstart,'%4.1f') ...
           ' - ' num2str(Tend,'%4.1f') '[ms]'],VERBOSE_LEVEL_NOTICE);     
  b = bexp(:,Tnoise,:);
  
  % temporal filter
  if temporal_filter
    for s = 1:Ntrial
      b(:,:,s) = vb_temporal_smooth(b(:,:,s));
    end
  end
  
  B = reshape(b, [Nch, Nt*Ntrial]);
  BB = B*B';
  
  switch noise_model
   case 1,
    Cov = eye(Nch,Nch); 
    invCov = eye(Nch,Nch);
    
   case 2,
    % covariance calculation
    Cov = diag(diag(BB));
    % normalization
    Cov = Cov/(trace(Cov)/Nch);
    invCov = pinv(Cov);
    
   case 3,
    % covariance calculation (+regularization)
    Cov = BB+(noise_reg*trace(BB)/Nch)*eye(Nch,Nch);        
    % normalization
    Cov = Cov/(trace(Cov)/Nch);
    invCov = pinv(Cov);
  end
  COV{n} = Cov;
  
  Ndata = Ndata + Nch*Ntrial*Nt;
  sx0 = sx0 + sum(sum(invCov.*(BB)));;
end
sx0 = sx0/Ndata;
