function [Zact_ave,Jinfo,bayes_parm,vb_parm,MEGinfo,Jext_ave,Pointlist] ...
    = vb_current_reconstruct_z_tr(proj_root,curr_parm);
% Current reconstruction using Bayesian inverse filter. 
%
% [syntax]
% [Zact,Jinfo,bayes_parm,vb_parm,MEGinfo] ...
%          = vbmeg_current_reconstruct_z_tr(proj_root,curr_parm)
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% curr_parm: <<struct>> Parameters for current estimation.
% --- fields of curr_parm
%  bayesfile    : <<string>> Model parameter file (.bayes.mat). 
%  currfile     : <<string>> Cortical current file (.curr.mat), created
%                 by this function. 
%  jactdir      : <<string>> Directory for saving trial current
%                 files. Relative path from cortical current file. 
%  trial_average: <optional> <<bool>> If true, 
%                 = [ON] : average current over all sessions
%                 = OFF  : current for each session
%  ix_area      : <optional> Vertex indices to calculate estimated 
%                 current. If 'ix_area' is empty or not given, cortical
%                 currents in the active region are calculated. 
%  tsubsmpl     : <optional> <bosolete> Specify subsampled time
%                 index. If 'tsubsmpl' is empty or not given, time
%                 subsampling is not done. 
%  dsampf       : <optional> <<int>> Specify frequency of
%                   downsampling. This value must be smaller than the
%                   original sampling frequency of M/EEG data. 
%  overlap_mode : <optional> <<int>>
%   = 0 : current is averaged over overlapped time window
%   = 1 : current is not averaged for overlapped window
%         current time series of each time windows 
%         are concatenated sequentially for spectral analysis
%  ix_trial     : <optional> Trial indices for which currents are
%                          reconstructed.
%  verbose      : <<bool>> Verbose flag. 
% ---
%
% [note] If following field is given, these values are used instead of
%        bayes_parm field in result file:
% ---
%  curr_parm.basisfile
%  curr_parm.megfile  
%  curr_parm.megfile_baseline
%  curr_parm.twin_meg 
%  curr_parm.Tperiod  
%  curr_parm.Tnext
% ---
%
% [output]
% Zact    : active current
%
% Zact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'
% Zact(Nact,Nsample)          for trial_average = ON 
% Zact(Nact,Nsample,Ntrials)  for trial_average = OFF
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrials  : # of trials in all session]
% Jinfo: <<struct>> Information of cortical current.
% --- fields of Jinfo
%  version   : <<string>> Version of cortical current file.
%  curr_type : <<string>> 'Z-current'. It can be 'J-current' for VBMEG
%              version 0.8 or older. 
%  Wact      : <<float matrix>> Smoothing Gaussian filter, mapping from
%              Z-current to J-current. 
%  ix_act    : <<int vector>>: Vertex indices of Z-current.
%  ix_act_ex : <<int vector>>: Vertex indices of J-current.
%  Lact      : <<int>> Number of current direction at one vertex. 
%  Tsample   : <<int vector>> Time sample indices of the original MEG
%              data. length(Tsample) == size(Zact,2) == size(Jact,2). 
%  Tmsec     : <<float vector>> Time in msec. 
%  SampleFreq: <<float>> Sample frequency of cortical current, not
%              original M/EEG signal [Hz]. 
%  Pretrigger: <<int>> Time points of the length of the pretrigger
%              period of cortical current data. It is neither actual time
%              nor time points of the original M/EEG signal. 
%  Ntrial    : <<int>> Number of trials of estimated current. 
%  patch_norm: <<bool>> Cortical current is patch size normalized
%              (Am/m^2) or not (Am). 
%  Tix       : <<L x 1 cell>> Time sample indices of each time window. 
%              Zact(:,Tix{n},:) is the set of Z-current within the n-th
%              time window.
% ---
%
% [history] 
% 2006-09-03 M. Sato
% * Non-overlapped concatenation mode is added for spectral analysis
% 2008-08-25 Taku Yoshioka
%   Extra dipole support
% 2008-09-30 Taku Yoshioka
%   Minor change for variables in Jinfo
% 2008-10-23 Taku Yoshioka
%  Bug fix for current estimation without extra dipoles
% 2009-04-02 Taku Yoshioka
%  Parameter name changed within this code for readability
%  (just replacing 'resultfile' to bayesfile)
% 2010-03-01 M. Sato
%  Bug fix for Wact index and fieldname(tsubsamp->tsubsmpl)
% 2010-05-26 Taku Yoshioka
%  Message display changed
% 2010-12-06 taku-y
%  [enhancement] curr_parm.dsampf supported. 
%  [minor]       Following fields of Jinfo set in this function: 
%                 SampleFreq
%                 Pretrigger
%                 Tmsec
%  [trivial] Jinfo.version = vb_version. 
% 2010-12-07 taku-y
%  [trivial] Jinfo.version = vbmeg('version');
% 2011-05-11 takiu-y
%  [debug] Jinfo.Tmsec corrected.
% 2011-06-28 taku-y
%  [minor] Jinfo.Tix added.
% 2012-02-15 taku-y
%  [debug] Jinfo.SampleFreq could be empty in a certain case. This bug
%  was fixed. 
% 2017-10-25 Yusuke Takeda
%  Modified for dipoles with multiple orientations
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~isempty(proj_root)
  bayesfile = [proj_root filesep curr_parm.bayesfile];
else
  bayesfile = curr_parm.bayesfile;
end

%
% Verbose level setting
% (note: 'verbose_level' is not related to input variable
% 'curr_parm.verbose'. So two configurations relating to message display
% are coexisting in this function.)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
global vbmeg_inst
verbose_const = vb_define_verbose; 

if isempty(vbmeg_inst) | ~isfield(vbmeg_inst,'verbose_level'), 
  verbose_level = verbose_const.VERBOSE_LEVEL_NOTICE;
else
  verbose_level = vbmeg_inst.verbose_level;
end

%
% load VBMEG estimated result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
load(bayesfile, 'bayes_parm','Model','vb_parm','Model_ext','Pointlist');

%
% check parameter of 'curr_parm'
%
% Values of 'curr_parm' fields dominates over 
%   those of 'bayes_parm' in bayesfile
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% save trial_average setting of bayse.mat for Cov 
bayes_trial_average = bayes_parm.trial_average;
[bayes_parm,ix_area,trial_average,tsubsamp,overlap_mode,verbose,dsampf] ...
    = check_arg(bayes_parm,curr_parm);
			
if ~isempty(proj_root);
  bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);
else
  bayes_parm_abs = bayes_parm;
end

% current file directory
if ~isempty(proj_root)
  currfile = fullfile(proj_root, curr_parm.currfile);
else
  currfile = [curr_parm.currfile];
end 

% jactdir is relative path from current file
jactdir   = curr_parm.jactdir;
curr_root  = fileparts(currfile);
jactdir_ab = fullfile(curr_root, jactdir);

%check directory
if exist(jactdir_ab,'dir') ~= 0
  if verbose_level>=verbose_const.VERBOSE_LEVEL_NOTICE, 
    msg = [ 'jact_dir : ' jactdir_ab ' exists. Do you over write?'];
    btn = questdlg(msg, 'confirm', 'Yes', 'No', 'Yes');
    if strcmp(btn, 'No')
      error('processing aborted.');
      return;
    end
  else
    vb_disp(['jact_dir : ' jactdir_ab ' exists. So files in this ' ...
             'directory are overwritten with new ones.'], ...
            verbose_const.VERBOSE_LEVEL_WARNING);
  end
else
  res = mkdir(curr_root, jactdir);
  if  res~= 1,  error('mkdir failed.'); end
end

% Trial indices
if isfield(bayes_parm_abs,'ix_trial'), 
  bayes_parm_abs.ix_trial = bayes_parm_abs.ix_trial;
elseif isfield(curr_parm,'ix_trial'), 
  bayes_parm_abs.ix_trial = curr_parm.ix_trial;
else
  bayes_parm_abs.ix_trial = [];
end

%
% MEG data preparation
% B      : MEG data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[B,Ntrials,Nch,Tsample,Twindow,Tmsec] ...
    = vb_megdata_preparation(bayes_parm_abs);
MEGinfo = vb_load_meg_info(bayes_parm_abs.megfile{1});

%
% Preparation of lead fields
% Gact   : leadfield of focal window
% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Focal window
vb_disp('Lead field matrix of focal window');

lf_parm.brainfile = bayes_parm_abs.brainfile;
lf_parm.areafile = bayes_parm_abs.areafile;
lf_parm.patch_norm = bayes_parm_abs.patch_norm;
lf_parm.expand_spatial_filter = bayes_parm_abs.expand_spatial_filter;
lf_parm.spatial_smoother = bayes_parm_abs.spatial_smoother;
lf_parm.basisfile = bayes_parm_abs.basisfile;
lf_parm.area_key = bayes_parm_abs.area_key;
lf_parm.reduce = bayes_parm_abs.reduce;
lf_parm.Rfilt = bayes_parm_abs.Rfilt;
lf_parm.remove_area_key = [];

Norient     = vb_parm.Norient;    % Number of current orientation component
Norient_var = vb_parm.Norient_var;% Number of current orientation component
                                  %   for variance estimation
Ratio = Norient/Norient_var;  	  % = (# of orientation)
                                  %   /(# of orientation in variance)

[Gact, ix_act, ix_act_ex, Wact, Lact] = ...
    vb_leadfield_preparation(lf_parm);

% Extra dipole
if isfield(bayes_parm_abs,'extra') & ~isempty(bayes_parm_abs.extra), 
  vb_struct2vars(bayes_parm_abs,{'extra'});
  vb_disp('Lead field matrix of extra dipoles');
  for n=1:length(extra.basisfile)
    tmp = vb_load_basis(extra.basisfile{n});
    Gext{n} = tmp';
  end
else
  Gext = [];
end

%
% Area index in which current is calculated
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(ix_area),
  % Select vertex index 'ix_area' within the active current region
  [jx_area_ex, ix_area_ex] = vb_index2indexsub(ix_area, ix_act_ex);
else
  jx_area_ex = 1:length(ix_act_ex);
end

Wact   = Wact(jx_area_ex,:);
jx_act = find( sum(Wact, 1) > 0);
Wact   = Wact(:,jx_act);

% active index of Z-current
ix_act = ix_act(jx_act);
% active index of J-current
ix_act_ex = ix_act_ex(jx_area_ex);

% # of active vertex
Njact_area = length(jx_act);

% # of extra dipoles
if ~isempty(Gext), Njext = size(Gext{1},2);
else Njext = 0; end

%
% Constant
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Nsession = length(B);	 % Number of sessions
Ntotal   = sum(Ntrials); % Total number of trials in all sessions

%
% Temporal subsampling index
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
sampf = MEGinfo.SampleFreq;

if ~isempty(dsampf), 
  tsubsamp = ceil(1:sampf/dsampf:Tsample);
  Jinfo.SampleFreq = dsampf;
else
  if isempty(tsubsamp), tsubsamp = 1:Tsample; end
  Jinfo.SampleFreq = sampf;
end

Jinfo.Tmsec = Tmsec(tsubsamp);
[tmp,ix] = min(abs(Tmsec(tsubsamp)));
Jinfo.Pretrigger = ix;

%
% Temporal smoothing window weight
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[Tweight,Tindex,Nindex,Nsample] = ...
    vb_overlapped_timewindow_weight(Twindow,Tsample,tsubsamp, ...
                                    overlap_mode);

Nwindow   = length(Nindex); % # of time window
Jinfo.Tix = Nindex;

if overlap_mode == 1,
  vb_disp('Non-overlapped concatenation mode'); 
end

%
% Initialization
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Current averaged over trials
Zact_ave = zeros(Njact_area*Ratio,Nsample);
Jext_ave = zeros(Njext*Ratio,Nsample);

%
% Estimated current variance
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
a_inv = Model.a;
if ~isempty(Model_ext), e_inv = Model_ext.a;
else e_inv = []; end

% Modified by M. Sato 2015-12-25
%Cov = Model.Cov;
% Cov = Normalized noise covariance matrix
bayes_parm_abs.trial_average = bayes_trial_average;
[Cov] = vb_observation_noise_specification(bayes_parm_abs);

%
% Time window loop
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Ntrial_all = 0;

% Current Info
Jinfo.version   = '2.0-0.b.9'; %vbmeg('version');
Jinfo.curr_type = 'Z-current';

Jinfo.tindex    = tsubsamp;
Jinfo.Lact      = Lact;
Jinfo.Wact      = Wact;
Jinfo.ix_act    = ix_act;
Jinfo.ix_act_ex = ix_act_ex;
Jinfo.NJact     = Njact_area;

% MEG time window index
Tstart  = bayes_parm.twin_meg(1);
Tend    = bayes_parm.twin_meg(2);
if isempty(tsubsamp)
  Jinfo.Tsample = Tstart:Tend;
else
  Jinfo.Tsample = tsubsamp + Tstart - 1;
end

% Session loop 
for n=1:Nsession
  Ntry = size(B{n},3);
  Ntrials(n) = Ntry;
	
  % Lead field for each session
  G	 = Gact{n};   % Ga 
  if ~isempty(Gext), Ge = Gext{n}; else Ge = []; end
  Covs = Cov{n};     % Sg
  Nch	 = size(G,1);
	
  %%%% Calculate Inverse filter for all time window
  for j=1:Nwindow
    Nid = Nindex{j};	% index in the total subsampled data
    if isempty(Nid), continue; end

    if ~isempty(e_inv), 
      [KW{j},KW_ext{j}] ...
          = vb_invfilter_z(a_inv(:,j),G,Covs,jx_act,e_inv(:,j),Ge);
    else
      [KW{j},KW_ext{j}] ...
          = vb_invfilter_z(a_inv(:,j),G,Covs,jx_act,[],[]);
    end
  end % Timewindow loop END
	
  %%%% Current reconstruction
  for m=1:Ntry
    Zact = zeros(Njact_area*Ratio,Nsample);
    Jext = zeros(Njext*Ratio,Nsample);
		
    for j=1:Nwindow
      % Subsampling time index
      Tid = Tindex{j};	% subsampled time index
      Nid = Nindex{j};	% index in the total subsampled data
			
      if isempty(Nid), continue; end
			
      %%%% Time window smoothing
      Bt = vb_repmultiply( B{n}(:,Tid,m) , Tweight{j});
			
      Zact(:,Nid) = Zact(:,Nid) + (KW{j} * Bt);
      if ~isempty(KW_ext{j}),
        Jext(:,Nid) = Jext(:,Nid) + (KW_ext{j} * Bt);
      end
    end % Timewindow loop END
		
    % Trial average current
    Zact_ave   = Zact_ave + Zact;
    Jext_ave   = Jext_ave + Jext;
    Ntrial_all = Ntrial_all + 1;
		
    fname = sprintf('data_s%04dt%04d',n,m);
    vb_fsave([jactdir_ab '/' fname], 'Zact','Jinfo','Jext','Pointlist', ...
             'MEGinfo','bayes_parm','vb_parm');
    if verbose==1,
      fprintf('.')
      if rem(m, 20) == 0 % linefeed per 20
        fprintf('\n');
      end
    elseif verbose==2,
      fprintf('progress... session:%04d/%04d, trial:%04d/%04d\n',...
              n,Nsession,m,Ntry);
    end
  end % Trial loop END
end % Session loop END

Zact_ave = Zact_ave/Ntrial_all;
Jext_ave = Jext_ave/Ntrial_all;

Jinfo.Ntrial   = Ntrials;
Jinfo.Nsession = Nsession;
Jinfo.jactdir  = jactdir;

% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
% Jact   = Wact * Zact

% Actual time corresponding to columns of Zact, supporting overlap mode
% and non-overlapped concatenation mode
Tid_all = [];
Nid_all = [];
for j=1:Nwindow
  Tid_all = [Tid_all Tindex{j}];
  Nid_all = [Nid_all Nindex{j}];
end

if overlap_mode==false,
  ix          = unique(Tid_all);
  Jinfo.Tmsec = Tmsec(ix);
else
  Jinfo.Tmsec = Tmsec(Tid_all);
end

%Tstart  = bayes_parm.twin_meg(1);
%Tend    = bayes_parm.twin_meg(2);
%if isempty(tsubsamp)
%  Jinfo.Tsample = Tstart:Tend;
%else
%  Jinfo.Tsample = tsubsamp + Tstart - 1;
%end

if verbose==1,
  fprintf('\n')
end

return;

%%%% ---------------
function [bayes_parm,ix_area,trial_average,tsubsamp,overlap_mode,verbose,dsampf]= ...
    check_arg(bayes_parm,curr_parm)

if isfield(curr_parm,'basisfile'), 
  bayes_parm.basisfile = curr_parm.basisfile;
end;
if isfield(curr_parm,'megfile'), 
  bayes_parm.megfile   = curr_parm.megfile  ;
end;
if isfield(curr_parm,'twin_meg'), 
  bayes_parm.twin_meg  = curr_parm.twin_meg ;
end;
if isfield(curr_parm,'Tperiod'), 
  bayes_parm.Tperiod   = curr_parm.Tperiod  ;
end;
if isfield(curr_parm,'Tnext'), 
  bayes_parm.Tnext     = curr_parm.Tnext    ;
end;
if isfield(curr_parm,'megfile_baseline'), 
  bayes_parm.megfile_baseline = curr_parm.megfile_baseline ;
end;

if ~isfield(curr_parm,'trial_average'), 
  trial_average = ON; 
else
  trial_average = curr_parm.trial_average; 
end;

bayes_parm.trial_average = trial_average;

if ~isfield(curr_parm,'ix_area'),  
  ix_area = []; 
else
  ix_area = curr_parm.ix_area; 
end;
if ~isfield(curr_parm,'tsubsmpl'), 
  tsubsamp = []; 
else
  tsubsamp = curr_parm.tsubsmpl; 
end;
if ~isfield(curr_parm,'overlap_mode'),     
  overlap_mode = 0; 
else
  overlap_mode = curr_parm.overlap_mode; 
end;
if ~isfield(curr_parm,'verbose'),     
  verbose = 1; 
else
  verbose = curr_parm.verbose; 
end;

if isfield(curr_parm,'extra'),
  if isfield(curr_parm.extra,'basisfile'), 
    bayes_parm.extra.basisfile = curr_parm.extra.basisfile;
  end;
end;

if ~isfield(curr_parm,'dsampf'), 
  dsampf = [];
else
  dsampf = curr_parm.dsampf;
end

return;