function [J_out] = vb_convert_current_unit(proj_root, ...
                                           current_file, ...
                                           to_density, new_current_file)
% convert current unit(density/amplitude)
%
% [syntax]
% J_out = vb_convert_current_unit(proj_root, ...
%                                 current_file, to_density...
%                                 [,new_current_file]);
%
% [input]
%       proj_root     : VBMEG result file directory .
%    current_file     : Cortical current file (.curr.mat). 
%                       * relative path from proj_root
%       to_density    : unit after conversion
%                       = true  : convert to density   (Am/m^2).
%                       = false : convert to amplitude (Am).
%    new_current_file : [Optional] save current file.
%                       * relative path from proj_root
%
% [output]
%    J_out : 
%     to_density = true
%        J_out: current density for 'J-current' (Am/m^2). 
%     to_density = false
%        J_out: current amplitude for 'J-current' (Am). 
%
%---  'J-current' is always returned
%-----------------------------------------------------------
%--- Smoothing leadfield
% G   = basis*S*W (S: area of vertex, W: smoothing filter)
%--- MEG - current relation
% MEG = G * Zact; % Nsensor x TimeSample 
%     = basis*S*Wact*Zact
%     = basis*S*Jact_density  (patch_norm: ON)
%     = basis*Jact_amplitude  (patch_norm: OFF)
%--- conversion between J_amplitude and J_density
%       Jact_amplitude = Jact_density.*S
%       Jact_density   = Jact_amplitude./S
%--- S should be applied to Jact
%
% [history]
%    2014-10-2 Masa-aki Sato
%
% [note]
%    It should be noted that in general converted current density may be
%    different from the current density obtained by setting patch_norm = 1
%    using vb_job_vb(current variance estimation) because of inversion 
%    process.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 3
    error('Please check function usage.');
end
if ~exist('new_current_file', 'var')
    new_current_file = '';
end
save_flag = true;
if isempty(new_current_file)
    save_flag = false;
end
%
% --- Main Procedure
%
fprintf('starting unit conversion...');
current_file = fullfile(proj_root, current_file);

[Jinfo,Jact] = vb_load_current(current_file);
%  Jinfo.patch_norm: ON/OFF   (Am/m^2) / (Am). 
%  Jinfo.curr_type : 0: 'J-current' or 1: 'Z-current' 
%  Jinfo.Wact      : Smoothing Gaussian filter
%    Jact = Wact * Zact
%  Jinfo.ix_act    : Vertex indices of Z-current.
%  Jinfo.ix_act_ex : Vertex indices of J-current.


%
% --- Create J_out
%
switch  to_density
case    true
%      J_out: current density for 'J-current' (Am/m^2). 
    if Jinfo.patch_norm==ON
        J_out = Jact;
        fprintf('\nNo need to convert unit because the specified file contains data in density.\n');
        if save_flag
            save_flag = false;
        end
    else
        [Nact,Nsample,Ntrials] = size(Jact);
        % Jact    : current timecourse
        %   Jact(Nact,Nsample,Ntrials)  
        %     Nact     = Lact * Nvact,  Nvact = length(ix_act)
        %     Nsample  : # of time sample, 
        %     Ntrials  : # of trials in all session]

        ix_act_ex = Jinfo.ix_act_ex;
        Lact = Nact/length(ix_act_ex);

        load(current_file,'bayes_parm');

        brainfile  = fullfile(proj_root, bayes_parm.brainfile);

        % Patch norm normalization 
        xxA = vb_load_area_density(brainfile);

        %       Jact_density   = Jact_amplitude./S
        S = repmat(xxA(ix_act_ex),[Lact, Nsample, Ntrials]);
        J_out = Jact./S;
    end
    
    patch_norm = ON;

case    false
%      J_out: current amplitude for 'J-current' (Am). 
    if Jinfo.patch_norm==OFF
        J_out = Jact;
        fprintf('\nNo need to convert unit because the specified file contains data in amplitude.\n');
        if save_flag
            save_flag = false;
        end
    else
        [Nact,Nsample,Ntrials] = size(Jact);
        % Jact    : current timecourse
        %   Jact(Nact,Nsample,Ntrials)  
        %     Nact     = Lact * Nvact,  Nvact = length(ix_act)
        %     Nsample  : # of time sample, 
        %     Ntrials  : # of trials in all session]

        ix_act_ex = Jinfo.ix_act_ex;
        Lact = Nact/length(ix_act_ex);

        load(current_file,'bayes_parm');

        brainfile  = fullfile(proj_root, bayes_parm.brainfile);

        % Patch norm normalization 
        xxA = vb_load_area_density(brainfile);

        %       Jact_amplitude = Jact_density.*S
        S = repmat(xxA(ix_act_ex),[Lact, Nsample, Ntrials]);
        J_out = Jact.*S;

        patch_norm = OFF;
    end
end
disp('done.');

if save_flag
    %
    % --- new current directory file creation
    %
    new_file = fullfile(proj_root, new_current_file);
    inner_new_current_file_dir_create(new_file);
    s = load(current_file);


    % Header modification
    s.Jinfo.curr_type  = 'J-current';
    s.Jinfo.patch_norm = patch_norm;

    ix_act = s.Jinfo.ix_act_ex;

    s.Jinfo            = rmfield(s.Jinfo, 'ix_act_ex');
    s.Jinfo.ix_act     = ix_act;
    s.Jinfo.NJact      = length(ix_act);
    s.Jinfo.Wact       = [];

    % Current timecourse
    if isfield(s, 'Zact')
        s = rmfield(s, 'Zact');
    end
    s.Jact = J_out;

    % save
    disp('Now saving...');
    vb_save_struct(new_file, s);
    disp('done.');
end

if nargout
    if to_density
        unit = 'density';
    else
        unit = 'amplitude';
    end
    disp(['returning value: unit in ' unit, '.']);
end

function inner_new_current_file_dir_create(new_current_file)

[p] = vb_get_file_parts(new_current_file);
if exist(p, 'dir') ~= 7
    vb_mkdir(p);
end
