function [info] = vb_info_add_posfile_info(info, posfile)
% load position data from POS-MAT file and set to the appropriate field of
% MEGinfo or EEGinfo
%
% [usage]
%   [info] = vb_info_add_posfile_info(info, posfile)
%
% [input]
%      info : <optional> <<struct>> MEG or EEG information []
%           :  if this is empty, program creates MEGinfo or EEGinfo.
%   posfile : <optional> <<file>> POS-MAT file ''
%           :   if this is empty, empty fields will be added to info
%
% [output]
%      info : <<struct>> updated info
%
% [note]
%   fields which wll be added are ...
%     MRI_ID
%     Vcenter
%     Vradius
%     device_info.TransInfo
%
% [history]
%   2008-03-12 (Sako) initial version
%   2011-02-08 (Sako) modified to set empty values when posfile is invalid
%   2011-05-27 (Sako) modified according to the new data format
%   2016-12-19 (rhayashi) change behavior when specified posfile not found(warn->error)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('info', 'var'), info = []; end
if ~exist('posfile', 'var'), posfile = []; end
[info, posfile] = inner_check_arguments(info, posfile);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

def_coord = vb_define_coordinate;

if isempty(posfile)
  fprintf('--- (%s)posfile is empty - only set empty fields\n', func_);
  return;
end

% ----- MRI_ID
info.MRI_ID = vb_posfile_load_mrikey(posfile);

% ----- TransInfo
[info] = vb_info_set_transinfo(info, vb_posfile_load_transinfo(posfile));

% ----- Vcenter, Vradius
[center, radius, c_type] = vb_posfile_get_sphericalinfo(posfile);
if ~strcmp(c_type, def_coord.COORDINATE_SPM_RIGHT_M)
  warning('(%s)The type of obtained center coordinate is %s', func_, c_type);
end

info.Vcenter = center;
info.Vradius = radius;

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [info, posfile] = inner_check_arguments(info, posfile)
func_ = mfilename;

if isempty(info)
  % set empty fields
  info.Vcenter = [];
  info.Vradius = [];
  info.MRI_ID = '';
end

if isempty(posfile)
  % set empty fields
  info.Vcenter = [];
  info.Vradius = [];
  info.MRI_ID = '';

elseif exist(posfile, 'file') ~= 2
  error('(%s)cannot find posfile : %s\n', func_, posfile);
  % set empty fields
  info.Vcenter = [];
  info.Vradius = [];
  info.MRI_ID = '';
  posfile = []; 
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
