function	ix = vb_trigger_onset(y,y0,t_length,t_interval)
% Find onset time that signal exceed the threshold value 'y0'
%    ix = vb_trigger_onset(y,y0,t_length)
% --- Input
% y  : analog signal
% y0 : threshold value
% t_length  : period length (sample number)
%   if t_length < 1, t_length = t_length * (max period)
% --- Output
% ix : extracted onset time index
% --- Condition for onset
% 'y' value shoud be larger than 'y0' for period length 't_length'
% If length between two events < t_interval, two events are merged
%
% 2009-6-11 Masa-aki Sato
% 2012-1-8  Masa-aki Sato 
%   Check exceptional cases
% 2012-2-18  Masa-aki Sato 
%   Check t_interval
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

T = length(y);

if nargin < 3 || isempty(t_length), t_length = 0; end;

% find event start (signal exceed the threshold)
ix1 = vb_trigger_start(y,y0);

if isempty(ix1), ix=[]; return; end;
if t_length == 0, return; end;

% find event end   (signal goes under the threshold)
ix2 = vb_trigger_end(y,y0);

% condition : end_time > first start_time
jj  = find(ix2 > ix1(1));
ix2 = ix2(jj);

if isempty(ix2), ix=ix1; return; end;

N1  = length(ix1);
N2  = length(ix2);

% Check start & end pairing
% N1 = N2     : all start & end are paired
% N1 = N2 + 1 : last start do not have paired end
if N2 > N1, error('onset extraction error'); end;
if N1 > N2+1, error('onset extraction error'); end;

ix1 = ix1(:);
ix2 = ix2(:);

if N1 == N2+1
	ix2 = [ix2; T];
end

% event period length
ixd = ix2 - ix1;

if any(ixd < 0), error('onset extraction error'); end;

if t_length < 1, 
	t_length = max(ixd)*t_length;
end

if nargin == 4 && ~isempty(t_interval)
% --- If length between event < t_interval, events are merged
% n-th event period: ix1(n) - ix2(n)
% period between event: ix2(n) - ix1(n+1)

	flg  = zeros(T,1);
	for n = 1:N1
		flg(ix1(n):ix2(n)) = 1;
		if n < N1 && (ix1(n+1) - ix2(n)) < t_length
			flg(ix2(n):ix1(n+1)) = 1;
		end
	end
	
	%  [0 flg] : [ 0 0 1 1  1 0 0 1 1 1  1 0 0 0 1 1 1]
	%  diff      : [ 0 1 0 0 -1 0 1 0 0 0 -1 0 0 1 0 0]
	
	% start of event
	ix1 = find( diff([0; flg]) > 0 );
	% end of event
	ix2 = find( diff([0; flg]) < 0 );
	
	if length(ix1) > length(ix2),
		ix2 = [ix2; T];
	end
end

% reject event whose period length is less than t_length
jj = find( (ix2 - ix1) >= t_length );

ix = ix1(jj);
