function	[zx ,ix_y, ix_z, kx] = vb_get_emg_onset_time(y,yave,y0,z0,parm)
% Find EMG onset time using EMG & smoothed EMG signal
%    [zx] = vb_get_emg_onset_time(y ,yave, y0, z0, parm)
%    [zx, ixx ,jxx, kx] = vb_get_emg_onset_time(y ,yave, y0, z0, parm)
% --- Input
% y    : EMG signal
% yave : smoothed EMG by moving average
% y0   : threshold for abs(y)     
% z0   : smoothed EMG threshold value
%
% parm  : parameter structure
% parm.fsamp    : sampling frequency [Hz]
% --- Optional parameter for EMG onset
% parm.t_event  : minimum distance from previous onset event [150 ms]
% --- Usually following parameters need not be changed
% parm.t_period : minimum period that Smoothed EMG > threshold [50 ms]
% --- Condition for EMG onset
% 1. Smoothed EMG (yave) should exceed 'z0' more than 't_period' length
% 2. Distance from previous onset should be larger than 't_event'
%
% --- Output
% zx   : extracted onset time index
% ixx  : abs(EMG) onset time index
% jxx  : smoothed EMG onset     
% kx   : start & end of slope estimation region at jx
%
% 2009-08-05 Masa-aki Sato
% 2011-12-18 Masa-aki Sato
%   Conditions and parameters for onset are changed
% 2012-2-18  Masa-aki Sato 
%   Use TKE operator EMG filter
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Default value
t_slope   = 25;   % 25 ms
t_period  = 50;   % 50 ms
t_event   = 500;  % 500 ms

if isfield(parm,'t_slope'),  t_slope  = parm.t_slope ;end
if isfield(parm,'t_period'), t_period = parm.t_period ;end
if isfield(parm,'t_event'),  t_event  = parm.t_event ;end

% sample number in [msec]
ds   = parm.fsamp / 1000; 

% msec -> sample number
t_slope  = round(t_slope  * ds);
t_period = round(t_period * ds);
t_event  = round(t_event  * ds);

% EMG threshold
y  = abs(y(:)');
yave = yave(:)';
T  = length(y);

% Extract smoothed EMG onset (time at which yave exceed z0)
% 'yave' value shoud be larger than 'z0' for 't_period' 
ix = vb_trigger_onset(yave,z0,t_period,t_period);

% time length from previous onset should be larger than 't_event'
[ixx, jj] = vb_onset_check(ix,t_event);

% output for matched 'ix' and 'jx'
ix_y = {};
ix_z = {ix; ixx};

% onset candidate
zx  = ixx;
kx  = [];

%return
% --- No zero crossing Estimation ---

if isempty(t_slope), return; end;
if t_slope == 0, return; end;

% --- Estimate zero crossing ---
% by linear fit to smoothed EMG near onset

% number of matched onset
NN = length(ixx);

% --- Positive (non-negative) slope region for yave
T = length(yave);

ix_sp = find( [0 diff(yave)] >= 0 );
slope = zeros(1, T );
slope(ix_sp) = 1;

%  [0 slope] : [ 0 0 1 1  1 0 0 1 1 1  1 0 0 0 1 1 1]
%  diff      : [ 0 1 0 0 -1 0 1 0 0 0 -1 0 0 1 0 0]

% start of positive slope 
k1 = find( diff([0 slope]) > 0 );
% end of positive slope 
k2 = find( diff([0 slope]) < 0 );

N1 = min(length(k1),length(k2));

% start & end of positive slope region
k1 = k1(1:N1);
k2 = k2(1:N1);

% start & end of slope estimation region
kx = zeros(2,NN); 

% Estimate zero crossing 
% by linear fit to smoothed EMG near onset
for n=1:NN
	% onset candidate for smoothed EMG
	j0 = ixx(n);
	
	% find positive slope region that include 'j0'
	kk = find( k1 < j0 & k2 > j0 );
	
	if isempty(kk),
		j1 = j0 - t_slope; 
		j2 = j0 + t_slope;
	else
		j1 = min( k1(kk(1)) , j0 - t_slope);
		j2 = max( k2(kk(1)) , j0 + t_slope);
	end
	
	j1 = max(j1,1);
	j2 = min(j2,T);
	
	% linear fit to positive slope region
	% z - <z> = a * (x - <x>) 
	% z = a * (x - <x>) + <z> = 0
	% x = <x> - <z>/a
	x = j1:j2;
	z = yave(x);
	xm = mean(x);
	zm = mean(z);
	a  = sum((z-zm).*(x-xm))/sum((x-xm).^2);
	zx(n) = xm - zm/a;
	kx(:,n) = [j1 ; j2];
end

% check bad condition for linear fit
ii = find( abs(ixx - zx) > t_slope );

zx(ii) = ixx(ii);

return
