function	z = vb_emg_filter(x,fsamp)
% TKE operator EMG filter
%   z = vb_emg_filter(x,fsamp)
%
% 1. band-pass filtering at 30-300 Hz (6th order Butterworth)
% 2. y(t) = x(t)^2 - x(t-1)x(t+1)     (TKE operator)
% 3. low-pass filtering at 50 Hz      (2nd order Butterworth)
%
% - Solnik S, Rider P, Steinweg K, DeVita P, and Hortobagyi T. 
%   Teager-Kaiser energy operator signal conditioning 
%   improves EMG onset detection
%   Eur J Appl Physiol 110: 489-498, 2010.
%
% Masa-aki Sato 2012-2-18
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% 1. band-pass filtering at 30-300 Hz (6th order Butterworth)
if fsamp <= 250, 
	fcut  = [30 ];
	ftype = 'high';
else
	fcut  = [30 300];
	ftype = 'bandpass';
end

Norder = 6;
[B,A]  = butter(Norder, fcut/(fsamp/2) ,ftype);

y = filtfilt(B,A, x );

y = y(:);
T = length(y);

% 2. y(t) = x(t)^2 - x(t-1)x(t+1) (TKEO)
z = y.^2 - [y(1); y(1:T-1)].*[y(2:T); y(T)];
z = abs(z);

% 3. low-pass filtering at 50 Hz   (2nd order Butterworth)
fcut   = 50;
Norder = 2;
[B,A]  = butter(Norder, fcut/(fsamp/2) ,'low');

z = filtfilt(B,A, z );
