function [pick, Qpick, ref_pick, ref_Qpick, type, meg_info] = ...
  vb_ykgwfile_load_meg_channel_info(ykgwfile, meg_info)
% load and return channel information like as sensor coordinate, setting loaded 
% information into meg_info
%
% [usage]
%   [pick, Qpick, ref_pick, ref_Qpick, type, meg_info] = ...
%     vb_ykgwfile_load_meg_channel_info(ykgwfile, meg_info)
%
% [input]
%     ykgwfile : <required> <<file>> YOKOGAWA data file
%     meg_info : <optional> <<struct>> vase of information []
%              :  if this is not specified or is empty, meg_info will be 
%              :  created newly.
%
% [output]
%       pick : 3D-coordinates of sensor [Nchannel x 3]
%      Qpick : 3D-coordinates of normal vector [Nchannel x 3]
%   meg_info : <<struct>> MEGinfo the updated fields of which are as follows
%            : .MEGch_id      % Active MEG channel index [Nchannel x 1]
%            : .MEGch_name    % Active MEG channel name [Nchannel x 1]
%            : .Nchannel      % number of active axialmeters [x1]
%            : .nch0          % number of all channels [x1]
%            : .sensor_weight % coil weight [Nchannel x Nchannel*2]
%            : .device_info.sensor_weight_ref % sensor weight of ref channels
%   ref_pick : 3D-coordinates of positions of reference channels
%            :  [N_refCh x 3]
%  ref_Qpick : 3D-coordinates of directions of reference channels
%            :  [N_refCh x 3]
%       type : coordinate type like 'SPM_Right_m' etc.
%
% [note]
%   set hardware coordinate at first
%   after that try to transform coordinate system
%
%   --- channel information - now unused
%   ChannelInfo.Active = Channel_active;
%   ChannelInfo.Name   = Channel_name_str;
%   ChannelInfo.Type   = Channel_type;
%   ChannelInfo.ID     = Channel_id;
%   meg_info.ChannelInfo = ChannelInfo;
%
%   See also
%     getYkgwHdrChannel
%     vb_meg_yokogawa_load_ref_sensor
%     vb_yokogawa_get_channel_type
%
% [history]
%   2007-07-10 (Sako) initial version
%   2008-03-28 (Sako) modified according to new specification
%   2008-04-23 (Sako) added ChannelInfo to meg_info (for future)
%   2008-05-12 (M.Sato) modified according to planar gradiometer.
%                       pick, Qpick calculation is separeted by sensor type.
%   2011-07-04 (Sako) modified to use the new YOKOGAWA libarary
%   2011-07-22 (Sako) modified how to deal with sensor_weight_ref
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

global  vbmeg_inst;

% Define constant
define = vbmeg_inst.const;

% MEG Channel_type
MagnetoMeter          = 1;
AxialGradioMeter      = 2;
PlanarGradioMeter     = 3;

% --- CHECK ARGUMENTS --- %
if ~exist('ykgwfile', 'var'), ykgwfile = []; end
if ~exist('meg_info', 'var'), meg_info = []; end
[ykgwfile, meg_info] = inner_check_arguments(ykgwfile, meg_info);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
pick = [];
Qpick = [];
ref_pick = [];
ref_Qpick = [];

channel_info = getYkgwHdrChannel(ykgwfile);
% 
% --- for Refference Magneto Meter
[ref_pick, ref_Qpick, Nref] = vb_meg_yokogawa_load_ref_sensor(channel_info);

% Coil -> Channel matrix
meg_info = vb_meginfo_init_sensor_weight_refmg(meg_info, Nref);

meg_ch_types = vb_yokogawa_get_channel_type(1);

Channel_active = [];
Channel_name = [];
Channel_type = [];
Channel_id = [];

% -----------------------------------------------------------------------------

% --- channel information list [1 x n_channel]
% --- <<struct array>> fields are 'type' and 'data'
ch_info_list = channel_info.channel;
ch_type_list = [ch_info_list(:).type];

% pick1 = zeros(Nch,3);
ch_start = 1;

for i_type = 1:length(meg_ch_types)
  cur_type = meg_ch_types(i_type).id;
  cur_idx = find(ch_type_list(:) == cur_type);
  
  if isempty(cur_idx)
    continue;
  end

  cur_data_list = [ch_info_list(cur_idx).data];

  num_ch = length(cur_idx);

  % active or not (NullChannel or not)
  Channel_active = [Channel_active; ones(num_ch,1)];
  Channel_name   = [Channel_name; (cur_idx-1)];
  Channel_type   = [Channel_type; ch_type_list(cur_idx)'];
  Channel_id     = [Channel_id; cur_idx];

  ch_end = ch_start + num_ch - 1;
  pick1(ch_start:ch_end, 1) = [cur_data_list(:).x]; % X
  pick1(ch_start:ch_end, 2) = [cur_data_list(:).y]; % Y
  pick1(ch_start:ch_end, 3) = [cur_data_list(:).z]; % Z
  ch_start = ch_end + 1;
end

Nch = length(Channel_name);

% ------------------------------------------------------------------------------

Channel_name_str = cell(Nch,1);
for i_ch = 1:Nch
  Channel_name_str(i_ch) = {num2str(Channel_name(i_ch))};
end

% --- field format of channel_info.channel.data ------------------------------ %
%
% ----- fields common to all MEG channels
%
% .x          : x coordinate of inner coil position [meter]
% .y          : y coordinate of inner coil position [meter]
% .z          : z coordinate of inner coil position [meter]
% .size       : Inner coil size [meter]
%
% ---------------------------------------------------------------------------- %

% Select sensor type
% Index for pick1
ix_axial = find(Channel_type == AxialGradioMeter  );
ix_plana = find(Channel_type == PlanarGradioMeter);
ix_magne = find(Channel_type == MagnetoMeter);

% Index for channel_info
id_axial = Channel_id(ix_axial);
id_plana = Channel_id(ix_plana);
id_magne = Channel_id(ix_magne);

% Number of sensors for each type
Nch_axial = length(id_axial);
Nch_plana = length(id_plana);
Nch_magne = length(id_magne);

if Nch ~= (Nch_axial + Nch_plana + Nch_magne),
  error('Channel type mismatch')
end

% --- current (AxialGradioMeter) list
axial_list = [ch_info_list(id_axial).data];
plana_list = [ch_info_list(id_plana).data];
magne_list = [ch_info_list(id_magne).data];

% --- field format of channel_info.channel.data ------------------------------ %
%
% ----- fields for AxialGradioMeter or ReferenceAxialGradioMeter
% .zdir       : Sensor orientation from z-axis [degree]
% .xdir       : Sensor orientation from x-axis [degree]
% .baseline   : Baseline length [meter]
% ---------------------------------------------------------------------------- %
%
% ----- fields for PlanarGradioMeter or ReferencePlanarGradioMeter
% .zdir1      : Sensor orientation from z-axis [degree]
% .xdir1      : Sensor orientation from x-axis [degree]
% .zdir2      : Baseline orientation from z-axis [degree]
% .xdir2      : Baseline orientation from x-axis [degree]
% .baseline   : Baseline length [meter]
% ---------------------------------------------------------------------------- %
%
% ----- fields for MagnetoMeter or ReferenceMagnetoMeter
% .zdir       : Sensor orientation from z-axis [degree]
% .xdir       : Sensor orientation from x-axis [degree]
% ---------------------------------------------------------------------------- %

% --- AxialGradioMeter
if ~isempty(axial_list)
  zdir = ([axial_list(:).zdir] * (pi/180))'; 
  xdir = ([axial_list(:).xdir] * (pi/180))'; 
  baseline = ([axial_list(:).baseline])';

  unit_axial  = [sin(zdir).* cos(xdir), sin(zdir).* sin(xdir), cos(zdir)];
  pick2_axial = pick1(ix_axial, :) + vb_repmultiply( unit_axial, baseline);
else
  unit_axial = [];
  pick2_axial = [];
end


% --- PlanarGradioMeter
if ~isempty(plana_list)
  zdir1 = ([plana_list(:).zdir1] * (pi/180))'; 
  xdir1 = ([plana_list(:).xdir1] * (pi/180))'; 
  zdir2 = ([plana_list(:).zdir2] * (pi/180))'; 
  xdir2 = ([plana_list(:).xdir2] * (pi/180))'; 
  baseline = ([plana_list(:).baseline])';

  unit_plana1 = [sin(zdir1).* cos(xdir1), sin(zdir1).* sin(xdir1), cos(zdir1)];
  unit_plana2 = [sin(zdir2).* cos(xdir2), sin(zdir2).* sin(xdir2), cos(zdir2)];
  pick2_plana = pick1(ix_plana, :) + vb_repmultiply( unit_plana2, baseline);
else
  unit_plana1 = [];
  unit_plana2 = [];
  pick2_plana = [];
end

% --- MagnetoMeter
if ~isempty(magne_list)
  zdir = ([magne_list(:).zdir] * (pi/180))'; % zdir1
  xdir = ([magne_list(:).xdir] * (pi/180))'; % xdir1 

  unit_magne = [sin(zdir) .* cos(xdir), sin(zdir) .* sin(xdir), cos(zdir)];
else
  unit_magne = [];
end

% --- PICK
pick = [pick1; pick2_axial; pick2_plana];

% --- QPICK
% normal vector of pick1, pick2
%
unit_vector = zeros(Nch,3);
if ~isempty(ix_axial)
  unit_vector(ix_axial,:) = unit_axial;
end

if ~isempty(ix_plana)
  unit_vector(ix_plana,:) = unit_plana1;
end

if ~isempty(ix_magne)
  unit_vector(ix_magne,:) = unit_magne;
end

Qpick = [unit_vector; unit_axial; unit_plana1];

Ncoil     = Nch + Nch_axial + Nch_plana;
n_channel = [ix_axial', ix_plana', ix_magne'];
n_axial   = (1:Nch_axial) + Nch;                % pick2_axial -> ix_axial
n_plana   = (1:Nch_plana) + Nch + Nch_axial;    % pick2_plana -> ix_plana
n_coils   = [n_channel , n_axial,   n_plana];
n_channel = [n_channel , ix_axial', ix_plana'];
n_weight  = [ones(1,Nch) , - ones(1,Nch_axial), - ones(1,Nch_plana)];

CoilWeight = sparse( n_channel, n_coils, n_weight, Nch, Ncoil);


% --- store some data to meg_info
meg_info.MEGch_id      = Channel_id;         % Active MEG channel index
meg_info.MEGch_name    = Channel_name_str;   % Active MEG channel name
meg_info.Nchannel      = Nch;                % number of active channel
meg_info.nch0          = length(Channel_id); % number of channel
meg_info.sensor_weight = CoilWeight;         % coil weight

type = define.COORDINATE_YOKOGAWA_RIGHT_M;

% --- channel information - now unused
ChannelInfo.Active = Channel_active;
ChannelInfo.Name   = Channel_name_str;
ChannelInfo.Type   = Channel_type;
ChannelInfo.ID     = Channel_id;
meg_info.ChannelInfo = ChannelInfo;

% --- active channel flags
meg_info = vb_info_set_active_channel(meg_info, ChannelInfo.Active);

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [ykgwfile, meg_info] = ...
  inner_check_arguments(ykgwfile, meg_info)

func_ = mfilename;
if isempty(ykgwfile)
  error('(%s)ykgwfile is a required parameter', func_);
end

if exist(ykgwfile, 'file') ~= 2
  error('(%s)cannot find ykgwfile : %s', func_, ykgwfile);
end

if isempty(meg_info)
  % required no action
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
