function [err] = vb_meg_yokogawa_make_data(proj_root, meg_parm)
% Read MEG data saved on Yokogawa system and convert it to .meg.mat file. 
%
% [usage]
%   [err] = vb_meg_yokogawa_make_data(proj_root, meg_parm)
%
% [input]
%   proj_root : <required> Project root directory
%    meg_parm : <required> <<struct>> structure with following field
%             :     <<fields>>
%             :   .yokogawa_file : <required> <<file>> YOKOGAWA-MEG file
%             :                  :  .con | .ave | .raw
%             :   .meg_file      : <required> Output MEG file.
%             :                  :  This is used in connecting to proj_root.
%             :                  : [proj_root '/' <meg_parm.meg_file>] : 
%             :   .pos_file      : <optional> <<file>> POS-MAT file
%             :                  :   the members of which are as follows
%             :                  :    mri_key   - hash key [x1 string]
%             :                  :    coord_type- coordinate type [x1 string]
%             :                  :    pos       - coordinate [Nch x3 double]
%             :                  :    name      - channel name [Nch x1 string]
%             :                  :    trans_mri - transform matrix [4x4 double]
%             :                  :    header    - header information <<struct>>
%             :                  :      .digit_file_name : <optional>
%             :                  :      .meg_market_id : <optional>
%             :   .ex_ch_gain    : <optional> gain of extra channel[]
%             :   .memory_max    : <optional> const.DEFAULT_MEMORY_MAX(10000000)
%             :   .saveman       : <optional> <<struct>>
%             :                  : .data_dir : ['./(name of .meg_file)_bin']
%             :                  :  - directory where the data will be stored
%             :                  :  - relative path from the path of .meg_file
%             :                  : .precision : ['float64']
%             :                  :  - precision which is used for fwrite
%
% [output]
%         err : error code
%             :      0) success
%             :  not 0) failure
%
% [note]
%   1) If pos member in POS-MAT file is empty, only import MEG data
%
%   2) This function transform coordinates by using trans_mri matrix
%
%   3) data format of output file MEG-MAT is as follows
%      <<MEG-MAT>>
%       pick(n,      1:3) : position of detector coils.
%           (n+1:2n, 1:3) : position of conpensation detector coils.
%       Qpick             : vector of pick [Nch x3]
%       ref_pick          : position of reference channels [N_ref_ch x3]
%       ref_Qpick         : vector of ref_pick [N_ref_ch x3]
%       Measurement       : <<string>> measurement code 'MEG'
%       CoordType         : <<string>> coordinate type which all the 
%                         :  coordinates in MEG-MAT file apply
%       PositionFile      : <<string>> position file (POS-MAT file) name
%       MEGinfo           : <<struct>> with MEG acquisition information
%
%      <<MEGinfo>>
%     .Measurement        : <<string>> measurement code 'MEG'
%     .device             : device name (e.g. 'YOKOGAWA')
%     .sensor_weight      : weight to calculate lead field
%             : sensor_weight = n-th coil weight for m-th sensor channel
%             : basis(channel,dipole) = sensor_weight * basis(coil,dipole)
%     .Nchannel           : number of active gardiometers
%     .Nsample            : number of sampling data
%     .Pretrigger         : number of sampling data before trigger
%     .SampleFreq         : sampling frequency
%     .Nrepeat            : number of trial
%     .Trial              : <<struct>> trial information [Ntrial x 1]
%                         :  .number : trial number (index)
%                         :  .sample : [Nsample x 1] sample list
%     .Vcenter            : [3 x 1] origin of sphere model [m]
%     .MEG_ID             : MEG system ID(meg filename with no extension)
%     .MRI_ID             : MRI key (hash key)
%     .MEGch_name         : Active MEG channel name
%     .MEGch_id           : Active MEG channel index
%     .saveman            : <<struct>> saving specifications
%
%     .device_info        : <<struct>> device dependent information
%       .sensor_weight_ref: 
%       .TransInfo        : <<struct>> transform information
%       .acq_type         : acquisition type
%                         :  (e.g.'Continuous_Raw', 'Evoked_Ave', 'Evoked_Raw')
%       .acq_info         : <<struct>> acquisition information
%
% [history]
%   2007-07-05 (Sako) initial version
%   2008-03-13 (Sako) thoroughly rewrote according to the new specification
%   2009-07-14 (Sako) modified to store data as external file inevitably
%   2010-02-04 (Sako) removed invalid setting of bit_len
%   2011-02-17 (Sako) changed default value of saveman.data_dir
%   2011-02-17 (Sako) got rid of old data fields 'bexp' 'bexp_ext', 'refmg'
%   2011-02-24 (Sako) supported in case ref channels are empty
%   2011-02-24 (Sako) supported in case memory_max is given but empty
%   2011-05-26 (Sako) modified according to the new data format
%   2011-06-22 (Sako) changed the order of load_calibinfo
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
  
% --- CHECK ARGUMENTS --- %
if ~exist('proj_root', 'var'), proj_root = []; end
if ~exist('meg_parm', 'var'), meg_parm = []; end
[proj_root, meg_parm, ykgwfile, posfile, out_file, meg_info] = ...
    inner_check_arguments(proj_root, meg_parm);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
err = 0;  % success

% --- new spec - meg_info has acqfile field
[meg_info] = vb_meginfo_set_acqfile(meg_info, ykgwfile);
% <at this moment> determined fields of meg_info are ...
%   device_info.acq_info.data_file     % acquisition file path


[pick, Qpick, ref_pick, ref_Qpick, CoordType, meg_info] = ...
  vb_ykgwfile_load_meg_channel_info(ykgwfile, meg_info);
% only load hardware coordinate here
% later, apply appropriate coordinate system
% <at this moment> determined fields of meg_info are ...
%   MEGch_id      % Active MEG channel index
%   MEGch_name    % Active MEG channel name
%   Nchannel      % number of active axialmeters
%   nch0          % number of axialmeters
%   sensor_weight % coil weight
%   ActiveChannel % active channel flag


[meg_info] = vb_ykgwfile_load_header(ykgwfile, meg_info);
% <at this moment> newly determined fields of meg_info are ...
%   MEG_ID      % MEG filename (non extention)
%   device      % 'YOKOGAWA' <-- hard-coded
%   coord_type
%   acq_type
%   Nsample     % sampling number of measurement
%   Pretrigger  % number of sample before trigger
%   SampleFreq  % sampling frequency
%   Nrepeat     % repeat number
%   Trial       % struct of trial information


[meg_info] = vb_ykgwfile_load_extra_channel_info(ykgwfile, meg_info);
% <at this moment> newly determined fields of meg_info are ...
%   ExtraChannelInfo  % <<struct>> extra channels' information


[meg_info] = vb_info_add_posfile_info(meg_info, posfile);
% <at this moment> fields of meg_info which may be newly determined are ...
%   Vcenter
%   Vradius
%   MRI_ID
%   TransInfo


% --- load calibrate and separate data
[out_path] = vb_get_file_parts(out_file);
[meg_info] = vb_ykgwfile_load_data(ykgwfile, meg_info, out_path);
% <at this moment> fields of meg_info which may be newly determined are ...
%   MEGch_id


% --- convert information struct ---> MEGinfo
[MEGinfo] = vb_meginfo_convert_raw2vbmeg(meg_info);


% --- affine transform 'pick', 'Qpick', 'ref_pick', 'ref_Qpick'
if ~isempty(ref_pick) && ~isempty(ref_Qpick)
  [ref_pick, ref_Qpick] = ...
    vb_meg_transform_coordinate(ref_pick, ref_Qpick, CoordType, posfile);
end

[pick, Qpick, CoordType] = ...
  vb_meg_transform_coordinate(pick, Qpick, CoordType, posfile);

% store data
% !!! if you use vb_save, it is very slower than using vb_fsave !!!
Measurement = 'MEG';
MEGinfo.Measurement = Measurement;
PositionFile = posfile;
bexp = [];
bexp_ext = [];

vb_fsave(out_file, 'pick', 'Qpick', ...
  'ref_pick', 'ref_Qpick', 'CoordType', ...
  'Measurement', 'PositionFile', ...
  'MEGinfo', 'bexp', 'bexp_ext', 'meg_parm');

fprintf('---\n--- make %s\n---\n', out_file);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [proj_root, meg_parm, ykgwfile, posfile, outfile, meginfo] = ...
   inner_check_arguments(proj_root, meg_parm)
func_ = mfilename;

const = vb_define_yokogawa;

% --- check parameters
if isempty(meg_parm)
  error('(%s)meg_parm is a required parameter', func_);
end

% --- check fields of file
if ~isfield(meg_parm, 'yokogawa_file')
  error('(%s)yokogawa_file is a required field of meg_parm', func_);
end

if ~isfield(meg_parm, 'pos_file')
  meg_parm.pos_file = '';
%   error('(%s)pos_file is a required field of meg_parm', func_);
end

% --- check files
if exist(meg_parm.yokogawa_file, 'file') ~= 2
  error('(%s)cannot find yokogawa_file : %s', func_, meg_parm.yokogawa_file);
end

if ~isempty(meg_parm.pos_file) && exist(meg_parm.pos_file, 'file') ~= 2
  error('(%s)cannot find pos_file : %s', func_, meg_parm.pos_file);
end

ykgwfile = meg_parm.yokogawa_file;
posfile  = meg_parm.pos_file;

% --- MEGinfo
meginfo = [];

% --- EEG GAIN
if isfield(meg_parm, 'ex_ch_gain')
  meginfo = vb_meginfo_set_extra_ch_gain(meginfo, meg_parm.ex_ch_gain);
else
  meginfo = vb_meginfo_set_extra_ch_gain(meginfo, []);
end

% --- OUTPUT FILE
outfile = fullfile(proj_root, meg_parm.meg_file);

% --- SAVING SPECIFICATION
DEFAULT_PRECISION = 'float64';

real_dir = '';

% --- check and fill saveman if necessary
if ~isfield(meg_parm, 'saveman') ...
  || isempty(meg_parm.saveman)

  meg_parm.saveman = struct;
end

if ~isfield(meg_parm.saveman, 'data_dir') ...
  || isempty(meg_parm.saveman.data_dir)

  [DEFAULT_DATA_DIR, real_dir] = vb_device_make_data_dir(outfile);
  meg_parm.saveman.data_dir = DEFAULT_DATA_DIR;
end

if ~isfield(meg_parm.saveman, 'precision') ...
  || isempty(meg_parm.saveman.precision)

  meg_parm.saveman.precision = DEFAULT_PRECISION;
end

if isempty(real_dir)
  base_dir = vb_get_file_parts(outfile);
  real_dir = fullfile(base_dir, meg_parm.saveman.data_dir);
end

% --- check real directory
if exist(real_dir, 'dir') ~= 7
  vb_mkdir(real_dir);
end

meginfo.saveman = meg_parm.saveman;

if isfield(meg_parm, 'memory_max') && ~isempty(meg_parm.memory_max)
  meginfo.memory_max = meg_parm.memory_max;
else
  meginfo.memory_max = const.DEFAULT_MEMORY_MAX;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%--- END OF FILE ---%
