function [labels, result] = ...
  vb_meginfo_get_channel_label_refmg(meginfo, active_swt)
% return label list of refference magnetometer channel
% [usage]
%   [labels, result] = vb_meginfo_get_channel_label_refmg(meginfo, active_swt)
% [input]
%      meginfo : <required> <<struct>> MEGinfo which has 'ExtraChannelInfo'
%   active_swt : <optional> <<boolean>> active filter switch [false]
%              :   true) return active channels
%              :  false) return all the channels
% [output]
%       labels : label list of channels
%       result : <<integer>> error code
%              :  0) success
%              :  1) error - bad meginfo
% [note]
%   refference magnetometer channel stood on its foot from extra channels
%   if an appropriate field does not exist, return empty ([]).
%   for only YOKOGAWA for now
% [history]
%   2007-06-27 (Sako) initial version
%   2008-02-01 (Sako) modified for new specification
%   2008-04-17 (Sako) added active_swt
%   2008-06-06 (Sako) modified by adding return condition
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meginfo', 'var'), meginfo = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
[meginfo, active_swt, result] = inner_check_arguments(meginfo, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
labels = [];
if result ~= 0
  return;
end

all_labels = meginfo.ExtraChannelInfo.Channel_name;
if isempty(all_labels)
  return;
end

% this rule is YOKOGAWA rule
ch_type = vb_meginfo_get_channel_type_extra(meginfo);
active_list = vb_meginfo_get_channel_active_extra(meginfo, active_swt);

if isempty(ch_type) || isempty(active_list)
  return;
end

if length(active_list) ~= length(ch_type)
  fprintf('(%s) different length active list and channel type list\n', ...
    mfilename);
  labels = all_labels;
  return;
end

% ----- channel type over zero means extra channel
% ----- narrow by active filter and channel type
criterion = bitand((active_list == 1),(ch_type > 0));
labels = all_labels(criterion == 1);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meginfo, active_swt, result] = ...
  inner_check_arguments(meginfo, active_swt)

func_ = mfilename;
result = 0;

if isempty(meginfo)
  fprintf('(%s)meginfo is a required parameter\n', func_);
  result = 1;
  return;
end

% meginfo must have EEGinfo field
if ~isfield(meginfo, 'ExtraChannelInfo') || isempty(meginfo.ExtraChannelInfo)
  fprintf('(%s)meginfo must have ExtraChannelInfo field\n', func_);
  result = 1;
  return;
end

if isempty(active_swt)
  active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
