function vb_megfile_update_data(inst_spec)
% Replace data and update MEG-MAT file.
%
% This function is used to create/update MEG-MAT file with arbitrary
% signal preprocessing function not supported by VBMEG. 
%
% [syntax]
%   vb_megfile_update_data(inst_spec)
%
% [input]
%   inst_spec : <<required>> <struct> instruction spec
%       :  fields are as follows:
%       :   .org_file     : <<required>> MEG-MAT file which will be updated
%       :   .new_data     : <<required>> [n_channel x n_sample]
%       :   .channel_type : <<optional>> ['MEG'] | 'AXIAL' | 'PLANAR' | 'EXTRA'
%       :   .new_file     : <<optional>> if you want to make new file.
%       :                 :  ['org_file']
%       :                 :   Empty means overwriting '.org_file'.
%       :   .bin_data_dir : <<optional>> data stored directory.
%       :                 : This parameter is valid when '.new_file' is valid.
%       :                 : [(body of 'new_file')_bin]
%       :                 : e.g. './new_0123.meg.mat' --> './new_0123_bin'
%
% [output]
%   void
%
% [note]
%   The size of inst_spec.new_data must be the same as the original one.
%
% [example]
% 1 File create mode.
% >> bexp     = vb_load_meg_data('meg_file_org.meg.mat');
% >> bexp_new = preprocess_fnc(bexp); % s.t. size(bexp)==size(bexp_new)
% >> inst_spec.org_file     = 'meg_file_org.meg.mat';
% >> inst_spec.new_data     = bexp_new;
% >> inst_spec.new_file = 'meg_file_new.meg.mat';
% >> inst_spec.bin_data_dir = 'meg_file_new'; % relative path
% >> vb_megfile_update_data(inst_spec);
%
% 2 File update mode.
% >> bexp     = vb_load_meg_data('meg_file_org.meg.mat');
% >> bexp_new = preprocess_fnc(bexp); % s.t. size(bexp)==size(bexp_new)
% >> inst_spec.org_file = 'meg_file_org.meg.mat';
% >> inst_spec.new_data = bexp_new;
% >> vb_megfile_update_data(inst_spec);
%
% 3 File update mode with specified ('AXIAL' here) channel type. 
% >> [bexp,channel_info] = vb_load_meg_data('meg_file_org.meg.mat');
% >> ix = find(channel_info.Type==2); % 2 is Type of 'AXIAL' sensor. 
%                                     % This is valid for Yokogawa MEG.
% >> bexp_axial     = bexp(ix,:); % get 'AXIAL' sensor data
% >> bexp_axial_new = preprocess_fnc(bexp_axial);
% >> inst_spec.org_file = 'meg_file_org.meg.mat';
% >> inst_spec.new_data = bexp_axial_new; % Only 'AXIAL' data updated
% >> inst_spec.channel_type = 'AXIAL';
% >> vb_megfile_update_data(inst_spec)
% 
% [history]
%   2010-11-22 (Sako) trial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('inst_spec', 'var'), inst_spec = []; end
[inst_spec] = inner_check_arguments(inst_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
meg_file = inst_spec.org_file;
[ch_info] = vb_load_channel_info(meg_file, inst_spec.channel_type);

% --- check data is storing inside or outside
[state_bexp, const] = vb_util_check_variable_in_matfile(meg_file, 'bexp');

if state_bexp ~= const.VALID
  inner_update_external_data(inst_spec, ch_info);
else
  inner_update_internal_data(inst_spec, ch_info);
end

%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [inst_spec] = inner_check_arguments(inst_spec)
if isempty(inst_spec)
  error('(%s) inst_spec is a required parameter', mfilename);
end

if ~isfield(inst_spec, 'org_file') || isempty(inst_spec.org_file)
  error('(%s) inst_spec.org_file is a required field', mfilename);
end

if exist(inst_spec.org_file, 'file') ~= 2
  error('(%s) cannot find inst_spec.org_file : %s', ...
    mfilename, inst_spec.org_file);
end

if ~isfield(inst_spec, 'new_data') || isempty(inst_spec.new_data)
  error('(%s) inst_spec.new_data is a required field', mfilename);
end

if ~isfield(inst_spec, 'channel_type') || isempty(inst_spec.channel_type)
  inst_spec.channel_type = 'MEG';
end

if ~isfield(inst_spec, 'new_file') || isempty(inst_spec.new_file)
  inst_spec.new_file = '';
  
elseif ~isfield(inst_spec, 'bin_data_dir') || isempty(inst_spec.bin_data_dir)
  [fpath, fname] = vb_get_file_parts(inst_spec.new_file);
  idx = strfind(fname, '.')-1;
  inst_spec.bin_data_dir = [fname(1:idx) '_bin'];
end
%
% --- end of inner_check_arguments()

% --- inner_update_external_data()
%
function inner_update_external_data(inst_spec, ch_info)
func_ = mfilename;

meg_file = inst_spec.org_file;
meg_info = vb_load_measurement_info(meg_file);

if ~isempty(inst_spec.new_file)
  % --- copy meg_file - by using vb_load_meg_data()
  load_spec.ChannelType = 'ALL';
  load_spec.saveman = meg_info.saveman;
  load_spec.saveman.data_dir = inst_spec.bin_data_dir;
  vb_load_meg_data(meg_file, load_spec, inst_spec.new_file);

  tmp_dir = inst_spec.bin_data_dir;
  meg_root = vb_get_file_parts(inst_spec.new_file);

else
  % --- overwrite original file
  tmp_dir = meg_info.saveman.data_dir;
  meg_root = vb_get_file_parts(meg_file);
  
  inst_spec.new_file = meg_file;
end

% ----- bin directory is defined by relative path from megfile directory
data_dir = [meg_root '/' tmp_dir];

PRECISION = meg_info.saveman.precision;

const = vb_define_yokogawa;

n_channel = size(ch_info.Name, 1);
for i_ch = 1:n_channel
  ch_name = ch_info.Name{i_ch};
  cur_ch_file = sprintf('%s/%s.%s', ...
    data_dir, ch_name, const.EXT_CHANNEL_BIN);

  cur_fid = fopen(cur_ch_file, 'wb');
  if cur_fid == -1
    error('(%s)cannot open file (%s)', func_, cur_ch_file);
  end

  fwrite(cur_fid, inst_spec.new_data(i_ch, :), PRECISION);
  fclose(cur_fid);
%   fprintf('--- UPDATED %s\n', cur_ch_file);
end

% --- no change .meg.mat file of their own ...

return;
%
% --- end of inner_update_external_data()

% --- inner_update_internal_data()
%
function inner_update_internal_data(inst_spec, ch_info)
meg_file = inst_spec.meg_file;
meg = load(meg_file);
meg.bexp(ch_info.ID, :) = inst_spec.new_data(:,:);

if ~isempty(inst_spec.new_meg_file)
  vb_save_struct(inst_spec.new_meg_file, meg);
else
  vb_save_struct(meg_file, meg);
end
return;
%
% --- end of inner_update_internal_data()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
