function vb_megfile_inherit_gene(orgfile, newfile, ...
  meg, ext, ref, saveman, meg_labels, ext_labels, ref_labels)
% make new megfile which inherit features from orgfile
% [usage]
%   vb_megfile_inherit_gene(orgfile, newfile, ...
%     meg, ext, ref, saveman, meg_labels, ext_labels, ref_labels)
% [input]
%       orgfile : <required> <<file>> MEG-MAT file before loading
%       newfile : <required> <<file>> new MEG-MAT file name
%           meg : <required> MEG data [Nch_meg x Nsample x Ntrial] []
%           ext : <required> EXTRA data [Nch_ext x Nsample x Ntrial] []
%           ref : <required> REFERENCE data [Nch_ref x Nsample x Ntrial]
%       saveman : <optional> <<struct>> 
%               :  If this is not given or empty, data (meg, ext, ref) will be
%               :  stored internally as (bext/bexp_ext/refmg).
%    meg_labels : <optional> {Nchannel_meg x 1} [orgfile one]
%               :  label list of new meg channels
%               :  the number of which must be the same as that of meg
%    ext_labels : <optional> {Nchannel_ext x 1} [orgfile one]
%               :  label list of new extra channels
%               :  the number of which must be the same as that of ext
%    ref_labels : <optional> {Nchannel_ref x 1} [orgfile one]
%               :  label list of new refmg channels
%               :  the number of which must be the same as that of ref
% [output]
%   none
% [note]
%   !NOTICE! Trial information in header will not be changed.
% [history]
%   2010-06-28 (Sako) initial version
%   2011-03-02 (Sako) does not save empty data fields(bexp,bexp_ext,refmg)
%   2011-07-21 (Sako) modified making new MEGinfo
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('orgfile', 'var'), orgfile = []; end
if ~exist('newfile', 'var'), newfile = []; end
if ~exist('meg', 'var'), meg = []; end
if ~exist('ext', 'var'), ext = []; end
if ~exist('ref', 'var'), ref = []; end

if ~exist('saveman', 'var'), saveman = []; end

if ~exist('meg_labels', 'var'), meg_labels = []; end
if ~exist('ext_labels', 'var'), ext_labels = []; end
if ~exist('ref_labels', 'var'), ref_labels = []; end
[orgfile, newfile, meg, ext, ref, ...
  saveman, meg_labels, ext_labels, ref_labels] = ...
  inner_check_arguments(orgfile, newfile, meg, ext, ref, ...
    saveman, meg_labels, ext_labels, ref_labels);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

org = load(orgfile);

% --- new MEGinfo
MEGinfo = org.MEGinfo;

Nchannel_meg = 0;
Nchannel_ext = 0;
Nchannel_ref = 0;

Nsample_meg  = 0;
Nsample_ext  = 0;
Nsample_ref  = 0;

Ntrial_meg   = 0;
Ntrial_ext   = 0;
Ntrial_ref   = 0;

if ~isempty(meg)
  Nchannel_meg = size(meg, 1);
  Nsample_meg  = size(meg, 2);
  Ntrial_meg   = size(meg, 3);
end

if ~isempty(ext)
  Nchannel_ext = size(ext, 1);
  Nsample_ext  = size(ext, 2);
  Ntrial_ext   = size(ext, 3);
end

if ~isempty(ref)
  Nchannel_ref = size(ref, 1);
  Nsample_ref  = size(ref, 2);
  Ntrial_ref   = size(ref, 3);
end

% --- check consistency

% --- channel
if ~iscell(meg_labels), meg_labels = {meg_labels}; end
if ~iscell(ext_labels), ext_labels = {ext_labels}; end
if ~iscell(ref_labels), ref_labels = {ref_labels}; end

if Nchannel_meg ~= length(meg_labels)
  error('(%s) Nchannel_meg is incorrect (%d, %d)', ...
    func_, Nchannel_meg, length(meg_labels));
end

if Nchannel_ext ~= length(ext_labels)
  error('(%s) Nchannel_ext is incorrect (%d, %d)', ...
    func_, Nchannel_ext, length(ext_labels));
end

if Nchannel_ref ~= length(ref_labels)
  error('(%s) Nchannel_ref is incorrect (%d, %d)', ...
    func_, Nchannel_ref, length(ref_labels));
end

% --- sample
if ~(Nsample_meg == Nsample_ext && Nsample_meg == Nsample_ref)
  error('(%s) the number of sample must be the same (%d, %d, %d)', ...
    func_, Nsample_meg, Nsample_ext, Nsample_ref);
end

% --- trial
if ~(Ntrial_meg == Ntrial_ext && Ntrial_meg == Ntrial_ref)
  error('(%s) the number of trial must be the same (%d, %d, %d)', ...
    func_, Ntrial_meg, Ntrial_ext, Ntrial_ref);
end

% --- representative number
Nsample_new = Nsample_meg;
Ntrial_new  = Ntrial_meg;

% --- update Nchannel
MEGinfo = vb_meginfo_set_channel_number(MEGinfo, Nchannel_meg);

% --- update Nsample
MEGinfo = vb_meginfo_set_sample_number(MEGinfo, Nsample_new);

% ----- Nrepeat
MEGinfo = vb_info_set_Nrepeat(MEGinfo, Ntrial_new);

% MEGinfo = vb_info_adjust_trial(MEGinfo);

% ----- MEGch_id
[ch_idx] = ...
  vb_meginfo_get_channel_index_meg(MEGinfo, meg_labels);
MEGinfo.MEGch_id = MEGinfo.MEGch_id(ch_idx);

% ----- MEGch_name
MEGinfo.MEGch_name = MEGinfo.MEGch_name(ch_idx);
  
% ----- ActiveChannel
active_ch = vb_info_get_active_channel(MEGinfo);
if isempty(active_ch)
  Nch = length(MEGinfo.MEGch_id);
  MEGinfo = vb_info_set_active_channel(MEGinfo, ones(Nch,1));
elseif length(active_ch) ~= length(MEGinfo.MEGch_id)
  MEGinfo = vb_info_set_active_channel(MEGinfo, active_ch(ch_idx));
end

% ----- ChannelInfo
%   new_channel_info = vb_megfile_make_channel_info(orgfile, ...
%     meg_labels, ext_labels, ref_labels);
% --- MEGinfo.ChannelInfo saves only MEG channels
new_channel_info = vb_megfile_make_channel_info(orgfile, meg_labels, [], []);
MEGinfo = vb_info_set_channel_info(MEGinfo, new_channel_info);

% --- Extra channel

% ----- Sensor (pick,Qpick,sensor_weight)
if ~isempty(org.pick) && ~isempty(org.Qpick)
  [org.pick, org.Qpick, MEGinfo] = ...
    vb_meg_adjust_sensor(org.pick, org.Qpick, ch_idx, MEGinfo);
end

% ----- ExtraChannelInfo
all_ex_label = [ext_labels; ref_labels];
[MEGinfo, org.ref_pick, org.ref_Qpick] = ...
  vb_meginfo_solve_extra_channel( MEGinfo, ...
    org.ref_pick, org.ref_Qpick, all_ex_label);

do_save_ext = vb_saveman_get_switch(saveman);
new_meg = org;
new_meg.MEGinfo = MEGinfo;

if ~do_save_ext
  new_meg.bexp = meg;
  new_meg.bexp_ext = ext;
  new_meg.refmg = ref;
end

% --- make meg.mat file
vb_save_struct(newfile, new_meg);

if do_save_ext
  const = vb_define_yokogawa;

  f_path = vb_get_file_parts(newfile);
  b_path = vb_saveman_get_dir(saveman);
  PRECISION = vb_saveman_get_precision(saveman);
  
  if exist([f_path '/' b_path], 'dir') ~= 7
    mkdir([f_path '/' b_path]);
  end

  % --- MEG channel
  for i_ch = 1:Nchannel_meg
    inner_make_external_data_file( ...
      f_path, b_path, meg_labels{i_ch}, const.EXT_CHANNEL_BIN, ...
      PRECISION, meg(i_ch,:,:));
  end

  % --- EXTRA channel
  for i_ch = 1:Nchannel_ext
    inner_make_external_data_file( ...
      f_path, b_path, ext_labels{i_ch}, const.EXT_CHANNEL_BIN, ...
      PRECISION, ext(i_ch,:,:));
  end

  % --- REFERENCE channel
  for i_ch = 1:Nchannel_ref
    inner_make_external_data_file( ...
      f_path, b_path, ref_labels{i_ch}, const.EXT_CHANNEL_BIN, ...
      PRECISION, ref(i_ch,:,:));
  end
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [orgfile, newfile, meg, ext, ref, ...
  saveman, meg_labels, ext_labels, ref_labels] = ...
  inner_check_arguments(orgfile, newfile, meg, ext, ref, ...
    saveman, meg_labels, ext_labels, ref_labels)
func_ = mfilename;
if isempty(orgfile)
  error('(%s)orgfile is a required parameter', func_);
end

if exist(orgfile, 'file') ~= 2
  error('(%s)cannot find orgfile : %s', func_, orgfile);
end

if isempty(newfile)
  error('(%s)newfile is a required parameter', func_);
end

if isempty(meg)
  error('(%s)meg is a required parameter', func_);
end

if isempty(ext)
  error('(%s)ext is a required parameter', func_);
end

if isempty(ref)
  error('(%s)ref is a required parameter', func_);
end

if isempty(saveman)
  % require no action
end

if isempty(meg_labels)
  meg_labels = vb_megfile_get_channel_label_meg(orgfile);
end
if isempty(ext_labels)
  ext_labels = vb_megfile_get_channel_label_extra(orgfile);
end
if isempty(ref_labels)
  ref_labels = vb_megfile_get_channel_label_refmg(orgfile);
end
return;
%
% --- end of inner_check_arguments()

% --- inner_make_external_data_file()
%
function inner_make_external_data_file(fpath, d_dir, fname, f_ext, prec, data)
func_ = mfilename;

ch_file = sprintf('%s/%s/%s.%s', fpath, d_dir, fname, f_ext);
fid = fopen(ch_file, 'wb');
if fid == -1
  error('(%s)cannot open file (%s)', func_, ch_file);
end

fwrite(fid, data(:), prec);
fclose(fid);
return;
%
% --- end of inner_make_external_data_file()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
