function [pick, Qpick, CoilWeight, Vcenter, channel_info] = ...
    vb_load_neuromag_sensor(meg_file, active_swt)
% Load NEUROMAG MEG sensor information
% [usage]
%   [pick, Qpick, CoilWeight, Vcenter] = ...
%     vb_load_neuromag_sensor(meg_file, ref_swt, active_swt)
%
% [input]
%     meg_file : <required> meg-mat file
%   active_swt : <optional> <<boolean>> flag whether to filter active channels
%              :    true) : Yes. This function filters active channels.
%              : [false]) : No. This function does not filter active channels.
%
% [output]
%    pick              : position of detector coils.
%    Qpick             : direction of coils.
%                      : if ref is true, these are directions of refference
%    CoilWeight        : (m,n) n-th coil weight for m-th sensor channel
%                  basis(channel,dipole) = sensor_weight * basis(coil,dipole)
%    Vcenter           : % origin of sphere model [m]
%    channel_info      : <<struct>> channel information of loaded data
%                        :  .Active [Nchannel x 1]
%                        :  .Name   [Nchannel x 1]
%                        :  .Type   [Nchannel x 1]
%                        :  .ID     [Nchannel x 1]
%                       not supported reference channel info.
% [history]
%   2018-08-22 (Takeda) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meg_file', 'var'), meg_file = ''; end
if ~exist('active_swt', 'var'), active_swt = []; end
[meg_file, active_swt] = ...
    inner_check_arguments(meg_file, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
channel_info = [];

% --- meg channels
load(meg_file, 'MEGinfo', 'pick', 'Qpick');

CoilWeight = vb_meginfo_get_sensor_weight_meg(MEGinfo);

if isfield(MEGinfo, 'Vcenter')
    Vcenter = MEGinfo.Vcenter;
else
    Vcenter = [];
end

% ----- active filter
ch_info_meg    = vb_load_channel_info(meg_file, 'MEG');
active_channel = vb_info_get_active_channel(MEGinfo, 1);
if isempty(active_channel)
    active_swt = false;
end

if ~active_swt
    channel_info = ch_info_meg;
    return;
else
    active_idx = find(active_channel == 1);
    active_coil_idx = find(sum(abs(CoilWeight(active_idx, :)), 1) > 0);
    
    if ~isempty(pick)
        pick = pick(active_coil_idx, :);
    end
    
    if ~isempty(Qpick)
        Qpick = Qpick(active_coil_idx, :);
    end
    
    if ~isempty(CoilWeight)
        CoilWeight = CoilWeight(active_idx, active_coil_idx);
    end
end
channel_info.Active = ch_info_meg.Active(active_idx);
channel_info.Name   = ch_info_meg.Name(active_idx);
channel_info.Type   = ch_info_meg.Type(active_idx);
channel_info.ID     = ch_info_meg.ID(active_idx);

%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meg_file, active_swt] = ...
    inner_check_arguments(meg_file, active_swt)
func_ = mfilename;

if isempty(meg_file)
    error('(%s) meg_file is a required parameter', func_);
end

if exist(meg_file, 'file') ~= 2
    error('(%s) cannot find meg_file (%s)', func_, meg_file);
end

if isempty(active_swt)
    active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %

