function	[data] = vb_filter_raw_data(data,parm)
% filtering and down sampling
%  [data] = vb_filter_raw_data(data,parm)
% --- Input
%  data : input signal [Nchannel x Tsample] or [Ntrial x Tsample]
% --- preprocess parameter
% parm.freq        : Sampling frequency [Hz]
% parm
%  .bias_flg    : Bias correction flag  (= 0/1/2 : OFF/Bias/Linear)
%               = N > 5: Bias correction by first N sample
%  .common_flg  : Common reference flag (= 1/0 : ON/OFF)
%  .highpass    : Highpass filter cutoff frequency [Hz]
%  .lowpass     : Lowpass  filter cutoff frequency [Hz]
%  .notch       : notch filter stop band frequency = [flow fhigh] [Hz]
%  .fsamp       : Down sampling frequency [Hz]
% --- If these fields are empty,
%        corresponding process are not applied
% --- Type of filter specification
%  .highpass_online : filter order of IIR highpass filter
%      = 0: FIR highpass filter (eegfilt)
%      = 1: online highpass filter (exponential)
%      > 1: Butterworth highpass filter (filtfilt)
%      < 0: Butterworth highpass filter (online)
%
%  .lowpass_online : filter order of IIR lowpass filter
%      = 0: FIR lowpass filter (eegfilt)
%      = 1: online lowpass filter (exponential)
%      > 1: Butterworth lowpass filter (filtfilt)
%      < 0: Butterworth lowpass filter (online)
%
%  2008-5-26 Masa-aki Sato
%  2009-2-16 Masa-aki Sato
%  2009-8-10 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if (size(data,1) > 1) && (size(data,2)==1),
	data = data';
end

[Nch ,T] = size(data);

fsamp = parm.freq;

% Bias correction
if  isfield(parm,'bias_flg') && ~isempty(parm.bias_flg) 
	if parm.bias_flg > 5,
		% Bias correction by first N sample
		%fprintf('Bias correction by first %d sample\n',parm.bias_flg)
		bias = mean(data(:,1:parm.bias_flg) , 2);
		data = vb_repadd(data, - bias);
	elseif parm.bias_flg > 0
		% Bias correction by all data
		%fprintf('Bias correction mode = %d\n',parm.bias_flg)
		data = vb_channel_bias_remove(data,parm.bias_flg);
	end
end

% Online highpass filter
if isfield(parm,'highpass') && ~isempty(parm.highpass),
	if isfield(parm,'highpass_online') 
		% Highpass cutoff frequency
		fcut = parm.highpass; 
		Norder = parm.highpass_online;
		
		if Norder == 1
			% Exponential online highpass filter
			%fprintf('Online highpass filter %6.1f [Hz]\n',fcut)
			data = vb_online_highpass_cut(data, fsamp, fcut);
		elseif Norder > 1
			%fprintf('butter filtfilt highpass filter %6.1f [Hz]\n',fcut)
			[B,A] = butter(Norder, fcut/(fsamp/2) ,'high');
			for n=1:Nch
				data(n,:) = filtfilt(B,A, data(n,:) );
			end
		elseif Norder < 0
			%fprintf('Online butter highpass filter %6.1f [Hz]\n',fcut)
			[A, B, C, D, Z] = ...
				vb_online_butter_init(fcut,fsamp,abs(Norder),Nch,'high');
			if Nch > 10
				[data ,Z] = vb_online_filter_loop(A, B, C, D, data, Z);
			else
				[data ,Z] = vb_online_filter_loop_mex(A, B, C, D, data, Z);
			end
		end
	end
end

% Lowpass filter
if isfield(parm,'lowpass') && ~isempty(parm.lowpass)
	fcut = parm.lowpass;
	
	if isfield(parm,'lowpass_online') 
		% fprintf('Online Lowpass filter %6.1f [Hz]\n', fcut)
		% Online Butterworth filter
		Norder = parm.lowpass_online; % filter order
		if Norder > 0
			%fprintf('butter filtfilt lowpass filter %6.1f [Hz]\n',fcut)
			[B,A] = butter(Norder, fcut/(fsamp/2) ,'low');
			for n=1:Nch
				data(n,:) = filtfilt(B,A, data(n,:) );
			end
		elseif Norder < 0
			%fprintf('Online butter lowpass filter %6.1f [Hz]\n',fcut)
			[A, B, C, D, Z] = ...
				vb_online_butter_init(fcut,fsamp,abs(Norder),Nch,'low');
			if Nch > 10
				[data ,Z] = vb_online_filter_loop(A, B, C, D, data, Z);
			else
				[data ,Z] = vb_online_filter_loop_mex(A, B, C, D, data, Z);
			end
		end
	end
	if ~isfield(parm,'lowpass_online') || parm.lowpass_online==0
		% FIR Lowpass filter
		% fprintf('FIR Lowpass filter %6.1f [Hz]\n', fcut)
		data = eegfilt(data, fsamp, 0, fcut );
	end
end

% IIR notch filter
if isfield(parm,'notch') && ~isempty(parm.notch),
	Norder = 5;
	[B,A] = butter(Norder, fcut/(fsamp/2) ,'stop');
	
	for m=1:Ntr
		for n=1:Nch
			data(n,:,m) = filtfilt(B,A, data(n,:,m) );
		end
	end
end

% Down sampling
if isfield(parm,'fsamp') && ~isempty(parm.fsamp) && fsamp ~=parm.fsamp
	fsnew = parm.fsamp;
	%fprintf('Down sampling to %6.1f [Hz]\n',fsnew)
	
	data  = vb_convert_freq(data, fsamp, fsnew);
	fsamp = fsnew;
end

% FIR Highpass filter
if isfield(parm,'highpass') && ~isempty(parm.highpass)
	if ~isfield(parm,'highpass_online') || parm.highpass_online==0
		%fprintf('FIR Highpass filter %6.1f [Hz]\n',parm.highpass)
		data = eegfilt(data, fsamp, parm.highpass, 0);
	end
end

% Common reference
if isfield(parm,'common_flg') && ~isempty(parm.common_flg) && parm.common_flg==1
%	fprintf('Common reference\n')
	data = vb_repadd(data, - mean(data,1));
end
