function [eeg_data, channel_info] = ...
  vb_eegfile_load_data(eegfile, loadspec, newfile, verbose_swt)
% load eeg data file and return eeg data (Nchannel x Nsample x Ntrial)
%
% [usage]
%   [eeg_data, channel_info] = ...
%     vb_eegfile_load_data(eegfile, loadspec, newfile, verbose_swt)
%
% [input]
%    eegfile : <required> <<file>> EEG-MAT file
%
%   loadspec : <optional> <<struct>> loading instruction
%    .ChannelName : <optional> channel name list [Nch x 1] []
%                 :  name(string) list of  target channel
%                 :  The way to be used this depends on the
%                 :  "ChannelSwitch" field
%                 :  e.g.
%                 :   {'1';'2';'3';'4'}
%                 :  If this is empty, all the channels are
%                 :  specified
%  .ChannelSwitch : <optiolal> <<boolean>> [true] | false
%                 :   Which is "ChannelName", to read, or not?
%                 :   [true] : to read
%                 :    false : to omit
%    .ChannelType : <optional> <<string>> channel type ['EEG']
%                 :  'EEG'   : EEG main channel data
%                 :  'EXTRA' : extra channel data
%                 :  'ALL'   : 'EEG' + 'EXTRA'
%                 :  - case insensitive
%        .Trigger : <optional> [1] sample list of trigger[Ntrigger x1 double]
%     .Pretrigger : <optional> [0] number of sample before trigger
%    .Posttrigger : <optional> [Nsample - 1] number of sample after trigger
%    .TrialNumber : <optional> trial number list [] [Ntrial x 1]
%                 :   "1" start
%                 :    e.g. [1;2;3]
%                 :    if this is empty, all the trials are specified
%  .ActiveChannel : <optional> <<boolean>> [false]
%                 :  active channel filter switch
%                 :    true) active channels
%                 :   false) all the channels
%    .ActiveTrial : <optional> <<boolean>> [false]
%                 :  active trial filter switch
%                 :  valid only when data type is Evoked_Raw
%                 :    true) active trials
%                 :   false) all the trials
%   .bin_data_dir : <optional> directory to newly store data
%                 :  This field is valid only when #3 argument newfile is 
%                 :  specified.
%                 :  ['name of newfile'_bin]
%
%   newfile : <optional> new MEG-MAT file name which is stored picked up
%           :  data and changed information []
%           :  if this is empty, new file will not be created
%   verbose_swt : <optional> <<boolean>> verbose message switch
%               :      true : output
%               :   [false] : do not output
%
% [output]
%      eeg_data : eeg channel data (Nchannel x Nsample x Ntrial)
%  channel_info : <<struct>> channel information of loaded data
%               :  .Active [Nchannel x 1]
%               :  .Name   [Nchannel x 1]
%               :  .Type   [Nchannel x 1]
%               :  .ID     [Nchannel x 1]
% [note]
%   it has been prepared for only BIOSEMI EEG-MAT file
%
% [history]
%   2008-03-31 (Sako) initial version
%   2008-04-16 (Sako) added ActiveSwitch to LoadSpec
%   2008-06-06 (Sako) you can specify ChannelType and get channel_info
%   2009-07-16 (Sako) modified making channel_info (EEGinfo.ChannelInfo)
%   2009-07-29 (Sako) refactored
%   2009-08-10 (Sako) supported the condition that ex_info.Type is empty
%   2009-12-25 (Sako) modified not to calibrate when loading data
%   2011-02-08 (Sako) added .bin_data_dir to loadspec to store data externally
%   2011-09-28 (Sako) supported the case without position infomation
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('eegfile', 'var'), eegfile = []; end
if ~exist('loadspec', 'var'), loadspec = []; end
if ~exist('newfile', 'var'), newfile = ''; end
if ~exist('verbose_swt', 'var'), verbose_swt = []; end
[eegfile, loadspec, newfile, ch_list, target_sample, tr_list, ...
  eegfile_dir, VERBOSE] = ...
    inner_check_arguments(eegfile, loadspec, newfile, verbose_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% ----- constants
func_ = mfilename;
vb_define_device;

old_eeginfo = vb_eegfile_load_eeginfo(eegfile);

% ----- read channel label
eeg_labels = vb_eeginfo_get_channel_label(old_eeginfo, false);
ext_labels = vb_eeginfo_get_channel_label_extra(old_eeginfo, false);

% --- loadspec has been arranged in inner_check_arguments()
[eeg_ch_list, eeg_idx] = vb_util_get_labels_to_read(eeg_labels, ...
            loadspec.ChannelName, loadspec.ChannelSwitch);
[ext_ch_list, ext_idx] = vb_util_get_labels_to_read(ext_labels, ...
            loadspec.ChannelName, loadspec.ChannelSwitch);
          
% ----- eeg and ext are managed devided
N_channel = vb_info_get_channel_number(old_eeginfo);
ext_idx_tmp = ext_idx + N_channel;

%
% --- !!! decide target channels here !!!
%
act_ch_list = [eeg_ch_list;ext_ch_list];
act_idx = [eeg_idx;ext_idx_tmp];

n_channel_new = length(act_ch_list);
if n_channel_new == 0
  error('(%s)cannot make channel list', func_);
end

n_sample_new = target_sample(1,2)-target_sample(1,1) + 1;

n_trial = vb_info_get_Nrepeat(old_eeginfo);

if n_trial == 1
  n_trial_new = size(target_sample, 1);

else
  n_trial_new = length(tr_list);
end

% ----- allocate buffer
eeg_data = zeros(n_channel_new, n_sample_new, n_trial_new);

% --- information to store data
data_type = vb_eeginfo_get_datatype(old_eeginfo);
act_data_type = data_type(act_idx);

file_ext = FILE_EXT_BIN_CH_EEG;

% ----- is base data external or internal?
% -----  - storing internally is old fashioned
internal_data = vb_eegfile_load_internal_data(eegfile);

% ----- "trial" will be set as the new EEGinfo.Trial
if isempty(internal_data)
  % --- EXTERNAL
  
  n_sample = vb_info_get_sample_number(old_eeginfo);

  % ----- binary data directory
  src_data_dir = [eegfile_dir filesep vb_eeginfo_get_datadir(old_eeginfo)];

  % ----- header
  EDF = vb_eeginfo_get_header(old_eeginfo);
  
%   % ----- constant because base data is continuous data
%   trial_number = 1;
  
  % ---------------------------- %
  % ----- LOAD BINARY DATA ----- %
  % ---------------------------- %
  for i_ch = 1:n_channel_new

    src_datafile = sprintf('%s%s%s.%s', ...
      src_data_dir, filesep, act_ch_list{i_ch}, file_ext);

    fid = fopen(src_datafile, 'r');
    if fid < 0
      error('(%s)cannot open : %s', func_, src_datafile); 
    end

    % ----- status channel check
    if strcmp(act_ch_list{i_ch}, STATUS_CH_LABEL)
      status_ch = true;
      data_size = STATUS_BIN_DATA_SIZE;
    else
      status_ch = false;
      data_size = STANDARD_BIN_DATA_SIZE;
    end

    if VERBOSE, fprintf('  --- read channel: ''%s''\n', act_ch_list{i_ch}); end

    % ----- read data according to n_trial
    ch_data = fread(fid, inf, act_data_type{i_ch});
    fclose(fid);

    if n_trial == 1
      % --- continuous data

      for i_trial = 1:n_trial_new
        sample_from = target_sample(i_trial, 1);
        sample_to   = target_sample(i_trial, 2);
        sample_idx  = sample_from:sample_to;

        this_eeg = ch_data(sample_idx);

        % converting to phisical value is done when importing
        eeg_data(i_ch,:,i_trial) = this_eeg;
      end

    else
      % --- chopped trial data
      org_ch_data = reshape(ch_data, n_sample, n_trial);
      
      % --- target_sample expected to be [1 x 2]
      sample_from = target_sample(1,1);
      sample_to   = target_sample(1,2);
      sample_idx  = sample_from:sample_to;
      eeg_data(i_ch,:,:) = org_ch_data(sample_idx, tr_list);
    end
  
    if VERBOSE
      fprintf(' ------------------------------------------------------\n');
    end
  end % channel loop

  
else  % internal data is not empty

  % --- INTERNAL
  n_trial = vb_info_get_Nrepeat(old_eeginfo);
  if n_trial == 1
    % continuous data
    n_trial_new = size(target_sample, 1);

    % --- chop to make trial data
    for i_trial = 1:n_trial_new
      sample_idx = target_sample(i_trial,1):target_sample(i_trial,2);
      eeg_data(:,:,i_trial) = internal_data(act_idx, sample_idx);
    end

  else
    % chopped data
    sample_idx = target_sample(1,1):target_sample(1,2);
    eeg_data = internal_data(act_idx, sample_idx, tr_list);
  end
end % data is external or internal

%
% --- update EEGinfo
%
EEGinfo = old_eeginfo;

EEGinfo.ChannelID = old_eeginfo.ChannelID(eeg_idx);
EEGinfo.ChannelName = eeg_ch_list;
EEGinfo.Nchannel = length(eeg_idx);
  
% ----- sensor position
old_sensor_pos = vb_eeginfo_get_sensor_position(EEGinfo);
if ~isempty(old_sensor_pos)
  EEGinfo = vb_eeginfo_set_sensor_position(EEGinfo, old_sensor_pos(eeg_idx,:));
end
  
% ----- data type of each sensor
EEGinfo.DataType = EEGinfo.DataType(act_idx);
    
% ----- sample information
EEGinfo.Nsample = n_sample_new;
EEGinfo.RecordTime = EEGinfo.Nsample / EEGinfo.SampleFrequency;

% ----- ExtraChannelInfo
old_exinfo = vb_info_get_channel_info(old_eeginfo, 1);
new_exinfo.Channel_active = old_exinfo.Active(ext_idx);
new_exinfo.Channel_name   = old_exinfo.Name(ext_idx);
new_exinfo.Channel_id     = old_exinfo.ID(ext_idx);

if isempty(old_exinfo.Type)
  new_exinfo.Channel_type   = ones(length(ext_idx), 1);
else
  new_exinfo.Channel_type   = old_exinfo.Type(ext_idx);
end

EEGinfo.ExtraChannelInfo = new_exinfo;

% --- EEGinfo.Trial
if ~isempty(loadspec.TrialNumber)
  % --- chopped trial --> narrowed down chopped trial
  old_trial = vb_info_get_trial_data(EEGinfo, loadspec.TrialNumber);

  sample_idx = target_sample(1,1):target_sample(1,2);

  for i_trial = 1:length(old_trial)
    new_trial(i_trial) = old_trial(i_trial);
    
    % --- update only sample
    sample_old = old_trial(i_trial).sample;
    new_trial(i_trial).sample = sample_old(sample_idx);
  end

else
  old_trial = vb_info_get_trial_data(EEGinfo);
  n_trial_old = length(old_trial);

  if n_trial_old == 1
    % --- continuous --> chopped trial or continuous
    n_trial_new = size(target_sample, 1);
    for i_trial = 1:n_trial_new
      sample_idx = target_sample(i_trial,1):target_sample(i_trial,2);
      new_trial(i_trial).sample = sample_idx;
      new_trial(i_trial).number = i_trial;
      new_trial(i_trial).Active = true;
    end
  else
    % --- chopped trial --> narrowed down chopped trial
    sample_idx = target_sample(1,1):target_sample(1,2);

    for i_trial = length(old_trial)
      new_trial(i_trial) = old_trial(i_trial);
    
      % --- update only sample
      sample_old = old_trial(i_trial).sample;
      new_trial(i_trial).sample = sample_old(sample_idx);
    end
  end
end
% [N x 1]
EEGinfo.Trial = vb_util_arrange_list(new_trial, 0);

EEGinfo = vb_info_adjust_trial(EEGinfo);

% ----- Pretrigger
if ~isempty(loadspec.Pretrigger)
  EEGinfo = vb_info_set_pre_trigger(EEGinfo, loadspec.Pretrigger);
end

% ----- new ActiveChannel
old_active_channel = vb_info_get_active_channel(old_eeginfo);
new_ActiveChannel = old_active_channel(eeg_idx);
EEGinfo = vb_info_set_active_channel(EEGinfo, new_ActiveChannel);
  
% ----- EEGinfo.ChannelInfo
%   .Active [Nchannel x 1]
%   .Name   [Nchannel x 1]
%   .Type   [Nchannel x 1]
%   .ID     [Nchannel x 1]
eeg_channel_info = ...
  vb_eeginfo_make_channel_info(EEGinfo, loadspec.ChannelName, 1);
EEGinfo.ChannelInfo = eeg_channel_info;

% --- channel_info as a return value
channel_info = vb_eeginfo_make_channel_info(EEGinfo, loadspec.ChannelName, 0);

if ~isempty(newfile)
  % --- make new file

  ParentInfo = inner_make_parent_info(eegfile);
  % ----- create new EEG-MAT file
  Measurement = 'EEG';
  if VERBOSE
    fprintf(' --- now on saving to new EEG-MAT file : %s ...', newfile);
  end

  % --- file data
  if ~isfield(loadspec, 'new_data_dir') || isempty(loadspec.new_data_dir)
    data_in = true;
  else
    data_in = false;
  end
  
  EEGinfo.File.BaseFile  = eegfile;
  EEGinfo.File.OutputDir = loadspec.new_path;
  EEGinfo.File.EEGFile   = loadspec.new_name;
  EEGinfo.File.DataDir   = loadspec.bin_data_dir;

  loadspec = inner_remove_temporary_fields(loadspec);
  LoadSpec = loadspec;

  if data_in
    % --- The data is stored inside of EEG-MAT file.
    vb_fsave(newfile, ...
      'eeg_data', 'EEGinfo', 'Measurement', 'ParentInfo', 'LoadSpec');
  else
    % --- The data is stored outside of EEG-MAT file.
    vb_fsave(newfile, ...
      'EEGinfo', 'Measurement', 'ParentInfo', 'LoadSpec');

    dst_data_dir = loadspec.new_data_dir;
    for i_ch = 1:n_channel_new
      inner_store_data_externally( dst_data_dir, ...
        act_ch_list{i_ch}, eeg_data(i_ch,:), act_data_type{i_ch}, file_ext)
    end
  end
  
  if VERBOSE
    fprintf(' done!\n');
  end
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [eegfile, loadspec, newfile, ...
  read_ch_list, target_samples, trial_list, ...
  eegfile_dir, verbose_swt] = ...
    inner_check_arguments(eegfile, loadspec, newfile, verbose_swt)

func_ = mfilename;

if isempty(eegfile)
  error('(%s)eegfile is a required parameter', func_);
end

if exist(eegfile, 'file') ~= 2
  error('(%s)cannot find eegfile: %s', func_, eegfile);
end

eegfile_dir = vb_get_file_parts(eegfile);

% eeg_data = [];

% ----- set default values for loadspec
loadspec = vb_loadspec_check(loadspec, eegfile);

% ----- narrow channel by user-specified list and active filter
eeginfo = vb_eegfile_load_eeginfo(eegfile);
[read_ch_list] = vb_eeginfo_get_read_channels(eeginfo, ...
  loadspec.ChannelName, loadspec.ChannelSwitch, loadspec.ActiveChannel);

if isempty(read_ch_list)
  error('(%s) read channels are empty', func_);
end
% ------- reset channel spec in loadspec
loadspec.ChannelName = read_ch_list;
loadspec.ChannelSwitch = true;

target_samples = vb_loadspec_make_trial_sample(loadspec);
if isempty(target_samples)
  n_sample = vb_info_get_sample_number(eeginfo);
  target_samples = [1 n_sample];
end

% ----- check TrialNumber
N_trial = vb_info_get_Nrepeat(eeginfo);
if N_trial == 1
  trial_list = [];
else
  % --- trigger info
  if size(target_samples, 1) > 1
    fprintf('(%s) loadspec.Trigger may be wrong\n', mfilename);
    target_samples = target_samples(1,:);
  end
  
  if isempty(loadspec.TrialNumber)
    cur_trial = vb_info_get_trial_data(eeginfo);
  else
    cur_trial = vb_info_get_trial_data(eeginfo, loadspec.TrialNumber);
  end

  trial_list = [cur_trial.number];

  if loadspec.ActiveTrial
    % if both information exist, the priority of the information is
    %   eeginfo.ActiveTrial > eeginfo.trial.Active
    if vb_info_active_trial_is_valid(eeginfo)
        trial_list = vb_info_get_active_trial_index(eeginfo);
    else
        active_list = [cur_trial.Active];
        trial_list = trial_list(active_list==true);
    end
  end
end
loadspec.TrialNumber = trial_list;

% % ------- valid trial
% if vb_info_active_trial_is_valid(eeginfo) && loadspec.ActiveTrial
%   active_idx = vb_info_get_active_trial_index(eeginfo);
%   trial_list = vb_info_get_trial_list(eeginfo);
%   if ~isfield(loadspec, 'TrialNumber') || isempty(loadspec.TrialNumber)
%     loadspec.TrialNumber = trial_list(active_idx);
%   else
%     idx = vb_util_get_index(loadspec.TrialNumber, active_idx);
%     loadspec.TrialNumber = loadspec.TrialNumber(idx);
%   end
% else
%   % It may not have to do anything
% end

if ~isempty(newfile)
  [new_path, new_name] = vb_get_file_parts(newfile);

  if ~isempty(new_path) && exist(new_path, 'dir') ~= 7
    vb_mkdir(new_path);
  end

  loadspec.new_path = new_path;
  loadspec.new_name = new_name;
  
  if ~isfield(loadspec, 'bin_data_dir') || isempty(loadspec.bin_data_dir)
    % --- if loadspec.bin_data_dir is invalid, store data inside.
    loadspec.bin_data_dir = '';
    new_data_dir = '';
  else
    new_data_dir = [new_path '/' loadspec.bin_data_dir];
  end
  
  if ~isempty(new_data_dir) && exist(new_data_dir, 'dir') ~= 7
    vb_mkdir(new_data_dir);
  end
  loadspec.new_data_dir = new_data_dir;
end

if isempty(verbose_swt)
%   verbose_swt = true;
  verbose_swt = false;
end
return;
%
% --- end of inner_check_arguments()

% --- inner_make_parent_info()
%
function parent_info = inner_make_parent_info(eegfile)
tmp_eeg = load(eegfile);
parent_info.EEGinfo = tmp_eeg.EEGinfo;
if isfield(tmp_eeg, 'ParentInfo')
  parent_info.ParentInfo = tmp_eeg.ParentInfo;
end
if isfield(tmp_eeg, 'LoadSpec')
  parent_info.LoadSpec = tmp_eeg.LoadSpec;
end
return;
%
% --- end of inner_make_parent_info()


% --- inner_store_data_externally()
%
function inner_store_data_externally(data_dir, ch_name, cur_data, d_type, f_ext)

datafile = sprintf('%s%s%s.%s', data_dir, filesep, ch_name, f_ext);
cur_fid = fopen(datafile, 'wb');
if cur_fid == -1
  error('(%s)cannot open file (%s)', mfilename, datafile);
end

fwrite(cur_fid, cur_data, d_type);
fclose(cur_fid);
% fprintf('--- %s\n', datafile);
return;
%
% --- end of inner_store_data_externally

% --- inner_remove_temporary_fields()
%
function loadspec = inner_remove_temporary_fields(loadspec)
target_field{1} = 'new_path';
target_field{2} = 'new_name';
target_field{3} = 'new_data_dir';

for i_field = 1:length(target_field)
  if isfield(loadspec, target_field{i_field})
    loadspec = rmfield(loadspec, target_field{i_field});
  end
end

return;
%
% --- end of inner_remove_temporary_fields()

%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
