function errcode = ...
  vb_eegfile_add_position(eegfile_old, posfile, eegfile_new)
% overwrite position data from POS-MAT file for EEG-MAT file
% [usage]
%   errcode = eegfile_overwrite_position(eegfile_old, posfile, eegfile_new)
% [input]
%   eegfile_old : <required> <<file>> EEG-MAT file which will be updated
%       posfile : <optional> <<file>> POS-MAT file ['']
%               :  if this is not specified or empty, do nothing
%   eegfile_new : <optional> <<string>> new EEG-MAT file if it is necessary
%               :  [eegfile_old]
%               :  if this is not specified, eegfile_old will be overwrite
% [output]
%       errcode : error code
%               :  1 ) success
%               :  0 ) do nothing
%               :  -1) invalid eegfile_old
% [note]
%   overwritten fields are as follows:
%       .Vcenter
%       .Vradius
%       .Coord
%       .CoordType
%       .device_info.TransInfo
%
% [history]
%   2008-03-19 (Sako) initial version
%   2011-05-27 (Sako) modified, according to the new data format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('eegfile_old', 'var'), eegfile_old = ''; end
if ~exist('posfile', 'var'), posfile = ''; end
if ~exist('eegfile_new', 'var'), eegfile_new = ''; end
[errcode, eegfile_old, posfile, eegfile_new] = ...
  inner_check_arguments(eegfile_old, posfile, eegfile_new);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
if errcode ~= 1
  % do nothing
  return;
end

EEGinfo = vb_eegfile_load_eeginfo(eegfile_old);
pos = load(posfile);

% ----- overwritten fields of EEGinfo
%       .Vcenter
%       .Vradius
%       .Coord
%       .CoordType
%       .device_info.TransInfo
% -----
EEGinfo = vb_eeginfo_set_sensor_position(EEGinfo, pos.pos);
EEGinfo.Vcenter = pos.spherical_info.Vcenter;
EEGinfo.Vradius = pos.spherical_info.Vradius;
EEGinfo.CoordType = pos.coord_type;
EEGinfo = vb_info_set_transinfo(pos.trans_info);

if ~isequal(eegfile_old, eegfile_new)
  copyfile(eegfile_old, eegfile_new);
end
vb_save(eegfile_new, 'EEGinfo');
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [errcode, eegfile_old, posfile, eegfile_new] = ...
  inner_check_arguments(eegfile_old, posfile, eegfile_new)
func_ = mfilename;
errcode = 1; % success

if isempty(eegfile_old)
  fprintf('(%s)eegfile_old is a required parameter\n', func_);
  errcode = -1;
  return;
else
  if exist(eegfile_old, 'file') ~= 2
    fprintf('(%s)cannot find eegfile: %s\n', func_, eegfile_old);
    errcode = -1;
  end
end

if isempty(posfile)
  fprintf('(%s)As posfile is not specified, do nothing\n', func_);
  errcode = 0;
  return;
end

if isempty(eegfile_new)
  eegfile_new = eegfile_old;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
