function [EEGfile, EEGinfo] = vb_read_bdf_save_ch_bin(bdf_file, ...
  figure_info, eeg_file, output_dir, device, eeginfo_version, verbose_swt)
% read BDF file and make initial EEGinfo data and file
%
% [usage]
%   [EEGfile, EEGinfo] = vb_read_bdf_save_ch_bin(bdf_file, figure_info, ...
%     eeg_file, output_dir, device, eeginfo_version, verbose_swt)
%
% [input]
%          bdf_file : <required> <<file>> BDF file
%       figure_info : <required> <<struct>> information to complete BDF file
%                   :  .SensorPosition : [N x 3]
%                   :  .CoordType      : coordinate type e.g. 'SPM_Right_m' etc.
%                   :  .ChannelLabel   : cell array of string {N x 1}
%                   :  .Vcenter        : center coordinate [1 x 3]
%                   :  .Vradius        : spherical radius  [x1]
%          eeg_file : <required> <<file>> output EEG-MAT file with directory
%        output_dir : <optional> output directory of binary data ['./']
%            device : <optional> device name ['BIOSEMI']
%   eeginfo_version : <optional> verson of EEGinfo struct [vb_latest_version()]
%                   :  - usually unused
%       verbose_swt : <optional> <<boolean>> [true]
%                   :  switch for output verbose message
%
% [output]
%   EEGfile : EEG-MAT file with its relative path
%   EEGinfo : <struct> eeg data structure
%     : .Measurement           : measurement type 'EEG'
%     : .File                  : <<struct>> file information in/out
%     :   .DataFile            : usually input BDF file
%     :   .OutputDir           : output directory for binary files
%     :   .EEG_File            : EEG-MAT file which will be made
%     : .ChannelName : channel name list {Nch x 1 cell}
%     : .ChannelID   : channel index except extra channels
%     : .Nchannel              : number of channel
%     : .SampleFrequency       : sample rate of original data [Hz]
%     : .Nsample               : total number of sample
%     :                        :  --> SampleFrequency * RecordTime
%     : .DataType              : list of type of binary data (e.g. 'float32')
%     :                        :  {Nch x 1 cell}
%     : .Coord                 : position coordinates of sensor [NallCh x 3]
%     : .CoodType              : coordinate type e.g. 'SPM_Right_m'
%     : .Device                : device name e.g. 'BIOSEMI'
%     : .Vcenter               : center of spherical head [1 x 3]
%     : .Vradius               : radius of spherical head [x1]
%     : .Trial                 : <<struct>> trial information
%     :   .number              : trial index [x1]
%     :   .sample              : sample list [Nsample x 1]
%     --- device dependent data
%     : .device_info
%         .Reference           : how to refer
%         .RecordTime          : record time of original data [sec]
%         .Header              : <<struct>> Biosemi header
%         .Version             : version of EEGinfo - unused
%         .History             : history of EEGinfo
%
% [note]
%   See also
%     vb_latest_version, vb_mkdir, vb_eeginfo_add_history
%     vb_openbdf, vb_readbdf
%
% [history]
%   2006-12-21 (Sako) initial version
%   2007-04-13 (Sako) got precise information of BIOSEMI's channel label
%   2007-08-10 (Sako) added sensor index to EEGinfo
%   2007-08-22 (Sako) modified not to be necessary digit and face file
%   2008-02-14 (Sako) thoroughly revised
%   2011-05-27 (Sako) modified according to the new data format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
tic;

vb_define_device;

% --- CHECK ARGUMENT --- %
if ~exist('bdf_file', 'var'), bdf_file = []; end
if ~exist('figure_info', 'var'), figure_info = []; end
if ~exist('eeg_file', 'var'), eeg_file = []; end
if ~exist('output_dir', 'var'), output_dir = []; end
if ~exist('device', 'var'), device = []; end
if ~exist('eeginfo_version', 'var'), eeginfo_version = []; end
if ~exist('verbose_swt', 'var'), verbose_swt = []; end

[bdf_file, figure_info, eeg_file, output_dir, device, eeginfo_version, ...
  VERBOSE] = inner_check_arguments(bdf_file, figure_info, ...
    eeg_file, output_dir, device, eeginfo_version, verbose_swt);

% --- PREPARATION --- %
SensorPosition         = figure_info.SensorPosition;
CoordType              = figure_info.CoordType;
ChannelName_coordinate = figure_info.ChannelLabel;

[DAT, header, RecordTime, Nch, SampleRate, data_type, ChannelName_hardware] ...
  = inner_get_local_var_bdf_file(bdf_file);

[channel_label, match_sensor_idx] = ...
  inner_rearrange_channel_label(ChannelName_coordinate, ChannelName_hardware);
  
% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

% get file id of each channel data
[sid, data_type_list] = inner_make_ch_file_id( ...
  channel_label, output_dir, data_type, VERBOSE);

% 
% --- make binary data files
%
inner_write_channel_data(sid, DAT, RecordTime, Nch, data_type_list, VERBOSE);

if VERBOSE
  fprintf('\n--- record time: %d [sec], sample rate:%d[Hz]\n', ...
    RecordTime, SampleRate);
end

% make return values
EEGinfo.Measurement      = 'EEG';
EEGinfo.Device           = device;
EEGinfo.ChannelID        = match_sensor_idx;
EEGinfo.ChannelName      = channel_label;
EEGinfo.Nchannel         = Nch;
EEGinfo.SampleFrequency  = SampleRate;
EEGinfo.Nsample          = RecordTime * SampleRate;
EEGinfo.DataType         = data_type_list;
EEGinfo = vb_eeginfo_set_sensor_position(EEGinfo, SensorPosition);
EEGinfo.CoordType        = CoordType;
EEGinfo.Vcenter          = figure_info.Vcenter;
EEGinfo.Vradius          = figure_info.Vradius;

  % files
EEGinfo.File.DataFile    = bdf_file;
EEGinfo.File.OutputDir   = output_dir;
EEGinfo.File.EEG_File    = eeg_file;

  % trial
Trial.number             = 1;
Trial.sample             = 1:EEGinfo.Nsample;
EEGinfo.Trial            = Trial;

% --- device dependent data
EEGinfo = vb_eeginfo_set_reference(EEGinfo, EEGINFO_REFERENCE_COMMON);
EEGinfo = vb_eeginfo_set_rectime(EEGinfo, RecordTime);
EEGinfo = vb_eeginfo_set_version(EEGinfo, eeginfo_version);
EEGinfo = vb_eeginfo_set_header(EEGinfo, header);

% ----- record history
history = ...
  sprintf('%s(''%s'', ''%s'', ''%s'')', func_, bdf_file, device, eeg_file);
EEGinfo = vb_eeginfo_add_history(EEGinfo, history);
EEGfile = eeg_file;

% measurement information
if VERBOSE, fprintf('--- now making EEG-MAT file(%s) ...', EEGfile); end

Measurement = 'EEG';
vb_fsave(EEGfile, 'EEGinfo', 'Measurement');

if VERBOSE, fprintf(' done!\n'); end
if VERBOSE, fprintf('=== READ BDF (%f[sec])=== \n', toc); end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [bdf_file, figure_info, eeg_file, output_dir, ...
    device, eeginfo_version, verbose_swt] = ...
  inner_check_arguments(bdf_file, figure_info, eeg_file, output_dir, ...
    device, eeginfo_version, verbose_swt)
func_ = mfilename;

vb_define_device;

% --- BDF FILE
if isempty(bdf_file)
  error('(%s)bdf_file is a required parameter', func_);
end

if exist(bdf_file, 'file') ~= 2
  error('(%s)cannot read bdf_file : %s', func_, bdf_file);
end

% --- FIGURE INFO
if isempty(figure_info)
  error('(%s)figure_info is a required parameter', func_);
end

if ~isfield(figure_info, 'SensorPosition')
  error('(%s)figure_info must have SensorPosition field', func_);
end

if ~isfield(figure_info, 'CoordType')
  error('(%s)figure_info must have CoordType field', func_);
end

if ~isfield(figure_info, 'ChannelLabel')
  error('(%s)figure_info must have ChannelLabel field', func_);
end

if ~isfield(figure_info, 'Vcenter')
  error('(%s)figure_info must have Vcenter field', func_);
end

% --- OUTPUT FILE / OUTPUT DIRECTORY
%     -- if directories do not exist, create them here
if isempty(eeg_file)
  error('(%s)eeg_file is a required parameter', func_);
end

[eeg_file_dir] = vb_get_file_parts(eeg_file);
if ~isempty(eeg_file_dir) && exist(eeg_file_dir, 'dir') ~= 7
  vb_mkdir(eeg_file_dir);
end

if isempty(output_dir)
  output_dir = './';
end

if exist(output_dir, 'dir') ~= 7
  vb_mkdir(output_dir);
end

if isempty(device)
  device = 'BIOSEMI';
end

if isempty(eeginfo_version)
  eeginfo_version = vb_latest_version('EEGinfo');
end

if isempty(verbose_swt)
  verbose_swt = true;
end
return;
%
% --- end of inner_check_arguments()

% --- inner_get_local_var_bdf_file()
%
function [DAT, header, RecordTime, Nch, SampleRate, data_type, ...
  ChannelLabel_hardware] = inner_get_local_var_bdf_file(bdf_file)

vb_define_device;

[DAT,H1] = vb_openbdf(bdf_file);

% procedure parameters
header     = DAT.Head;
RecordTime = DAT.Head.NRec;
Nch        = DAT.Head.NS;
SampleRate = max(DAT.Head.SampleRate);
data_type  = STANDARD_BIN_DATA_TYPE;
% data_size  = STANDARD_BIN_DATA_SIZE;

ChannelLabel_hardware = cell(1,Nch);

% get channel label for hardware
for ch = 1:Nch
%   EEGinfo.ChannelLabel  = DAT.Head.Label;
  ChannelLabel_hardware{ch} = deblank(DAT.Head.Label(ch,:));
end
%
% --- end of inner_get_local_var_bdf_file()

% --- inner_rearrange_channel_label()
%
function [channel_label, match_sensor_idx] = ...
  inner_rearrange_channel_label(ChannelLabel_coordinate, ChannelLabel_hardware)
func_ = mfilename;

vb_define_device;

% reset label size to [n_channel x 1]
ChannelLabel_coordinate = vb_util_arrange_list(ChannelLabel_coordinate);
ChannelLabel_hardware = vb_util_arrange_list(ChannelLabel_hardware);

% unmatch check
[idx_list] = ...
  vb_util_get_cross_checked_list(ChannelLabel_hardware, ChannelLabel_coordinate);

match_sensor_idx = find(idx_list(:,2) ~= -1);
NChannelCoord = length(ChannelLabel_coordinate);
NMatchChannel = length(match_sensor_idx);
if size(NChannelCoord > NMatchChannel)
%   fprintf('--- (%s)there are some unmatch sensor channels\n', func_);
  fprintf('--- (%s)number of channel : coordinate:%d, match:%d\n', ...
    func_, NChannelCoord, NMatchChannel);
end

channel_label = ChannelLabel_hardware;
return;
%
% --- end of inner_rearrange_channel_label()

% --- inner_make_ch_file_id()
%
function [sid, data_type_list] = ...
  inner_make_ch_file_id(channel_label, output_dir, data_type, VERBOSE)

func_ = mfilename;

vb_define_device;

Nch = size(channel_label,1);

DataFile = cell(Nch,1);
sid = zeros(Nch,1);
data_type_list = cell(Nch,1);

% support 'Status' channel
common_data_type = data_type;
status_data_type = STATUS_CH_DATA_TYPE;
status_channel_label = STATUS_CH_LABEL;

for ch = 1:Nch
  ch_name = channel_label{ch};
  
  DataFile{ch} = sprintf('%s%s%s%s', ...
    output_dir, filesep, ch_name, FILE_EXT_EEG_BIN_CH);
  sid(ch) = fopen(DataFile{ch},'w');
  
  if sid(ch) < 0
    error('(%s)cannot open : %s', func_, DataFile{ch});
  end

  % data type
  if strcmp(ch_name, status_channel_label) == 1
    data_type_list{ch} = status_data_type;
  else
    data_type_list{ch} = common_data_type;
  end
  
  if VERBOSE, fprintf(' >>> open %s\n', DataFile{ch}); end
end
return;
%
% --- end of inner_make_ch_file_id()

% --- inner_write_channel_data()
%
function inner_write_channel_data(sid, DAT, RecordTime, Nch, data_type, VERBOSE)

if VERBOSE
  PLOT_EVERY = 30;
  fprintf('--- now reading and storing BDF data (plot every %d/%d [sec])', ...
    PLOT_EVERY, RecordTime);
end

ReadMode   = 0;

for r = 1:RecordTime
  [DAT,DataBuf] = vb_readbdf(DAT, r, ReadMode);

  for ch = 1:Nch
    fwrite(sid(ch), DataBuf(:,ch), data_type{ch}); 
  end
  if VERBOSE && rem(r,PLOT_EVERY) == 0, fprintf('.'); end
end
if VERBOSE, fprintf('\n'); end

for ch=1:Nch
  fclose(sid(ch));
end
return;
%
% --- end of inner_write_channel_data()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
