function vb_do_mark_trigger(plot_spec, draw_spec)
% mark trigger point by hand
% [usage]
%   vb_do_mark_trigger(plot_spec, draw_spec)
% [input]
%   plot_spec : <required> <<struct>> mark_trigger_spec type
%    :                data : <required> (Nchannel x Nsample) 
%    :            plot_idx : <required> list of index of 'data' to be plotted
%    : trigger_channel_idx : <optional> [[]]
%    :                     : trigger channel index [Nch x 1] or [1 x Nch]
%    :                     : re-arrange to [Nch x 1] in this function
%    :  trigger_sample_idx : <optional> [[]]
%    :                     : list of trigger points as sample index
%    :         output_file : <optional> output file which has trigger_sample_list
%    :                     :   ['.\trigger_(date).mat']
%   draw_spec : <optional> <<struct>> specification for drawing
%    :          data_color : color of normal sensor data
%    :  trigger_data_color : color of trigger data
%    :  trigger_line_color : color of trigger line
% [output]
%   none
% [note]
%   Nchannel must be equal to sum of sizes of plot_idx and trigger_idx.
%   @prior_condition : 'trigger_sample_idx' is a list of sample number
%   @see : vb_cb_clip_trigger
%   @see : vb_define_process_trigger
% [history]
%   2007-01-25 (Sako) prototype #2 added zoom function
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('plot_spec', 'var'), plot_spec = []; end;
if ~exist('draw_spec', 'var'), draw_spec = []; end;
[plot_spec, draw_spec] = inner_check_arguments(plot_spec, draw_spec);


% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% common definition
vb_define_process_trigger;

my_data = plot_spec.data;

plot_idx = plot_spec.plot_idx;
trigger_idx = plot_spec.trigger_channel_idx; % unused for now
% plot_idx and trigger_idx are (N x 1)

data_color = draw_spec.data_color;
trigger_color = draw_spec.trigger_data_color;

% figure('Position', DPT_WINDOW_POSITION);

if isempty(plot_spec.trigger_sample_idx)
  figure('Position', DPT_WINDOW_POSITION);
  % --- line trigger <<by MANUAL>>
  %
  for ch = plot_idx'
    dh = plot(my_data(ch,:), 'Color', data_color); hold on;
    set(dh, 'Tag', DPT_DATALINE_TAG);
  end

  inner_set_controls(plot_spec.output_file);
  %
  % --- end of <<by MANUAL>>
  
else
  % --- line trigger <<by FULL or SEMI-AUTO>>
  %
  trigger_list = plot_spec.trigger_sample_idx;
% trigger_list is (N x 1)
% trigger_list is a array of <struct>
%   - all_idx : all
%   - beg_idx : beginning indices
%   - end_idx : ending indices

  trigger_num = size(trigger_list,1);

  for ntr = 1:trigger_num
    figure;

    for ch = plot_idx'
      dh = plot(my_data(ch,:), 'Color', data_color); hold on;
      set(dh, 'Tag', DPT_DATALINE_TAG);
    end

    % line spec
    trigger_line_gap = 10;
    trigger_line_clr = draw_spec.trigger_line_color;

    y_limit = get(gca, 'YLim');
    y_min = y_limit(1);
    y_max = y_limit(2);

    % draw trigger line
    y_data = [y_min:trigger_line_gap:y_max];

%   plot_index = trigger_list(ntr).all_idx;
    plot_index = trigger_list(ntr).beg_idx;
%   plot_index = trigger_list(ntr).all_idx;

    for this_x = plot_index
      % trigger point label
      vb_util_line_trigger(this_x, y_data, trigger_line_clr);
    end

    % output file name
    if trigger_num > 1
      output_file = sprintf('%s_%d', plot_spec.output_file, ntr);
    else
      output_file = plot_spec.output_file;
    end

    inner_set_controls(output_file);
  end   % for ntr
  %
  % --- end of <<by FULL or SEMI-AUTO>>
end
%
% --- END OF MAIN PROCEDURE ------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_argument()
% check argument
% @prior conditions : plot_spec and draw_spec exist
%   plot_spec : <<struct>> mark_trigger_spec type
%     :         data : <required> (Nchannel x Nsample) 
%     :     plot_idx : <required> list of index of 'data' to be plotted
%     :  trigger_idx : <optional> trigger index of 'data'
%     : trigger_list : <optional> list of trigger points as sample number
%     :  output_file : <optional> output file which has trigger_sample_list
%     :              :  ['.\trigger_(date).mat']
%   draw_spec : <optional> <<struct>> specification for drawing
%     :         data_color : color of normal sensor data
%     : trigger_data_color : color of trigger data
%     : trigger_line_color : color of trigger line
function [plot_spec, draw_spec] = inner_check_arguments(plot_spec, draw_spec)

vb_define_process_trigger;

if isempty(plot_spec)
  error('plot_spec is a required parameter');
end

if ~isfield(plot_spec, 'data')
  error('data is a required field of plot_spec');
end

if ~isfield(plot_spec, 'plot_idx')
  error('plot_idx is a required field of plot_spec');
elseif ~isempty( find(plot_spec.plot_idx > size(plot_spec.data,1) ...
    | plot_spec.plot_idx < 1))
  error('invalid index : plot_spec.plot_idx');
else
  plot_spec.plot_idx = vb_util_arrange_list(plot_spec.plot_idx);
end

%
% trigger channel index
%
if ~isfield(plot_spec, 'trigger_channel_idx') ...
    || isempty(plot_spec.trigger_channel_idx)
  
  plot_spec.trigger_channel_idx = [];
  
elseif ~isempty(find(plot_spec.trigger_channel_idx > size(plot_spec.data,1) ...
    | plot_spec.trigger_channel_idx < 1))
  
  error('invalid index : plot_spec.trigger_channel_idx');
  
else
  plot_spec.trigger_channel_idx = ...
    vb_util_arrange_list(plot_spec.trigger_channel_idx);
end

%
% trigger sample index
%
if ~isfield(plot_spec, 'trigger_sample_idx')
  plot_spec.trigger_sample_idx = [];
end

%
% output file
%
if ~isfield(plot_spec, 'output_file') || isempty(plot_spec.output_file)
  plot_spec.output_file = sprintf('.%strigger_%s.mat', filesep, date);
end

%
% draw_spec
%
if isempty(draw_spec)
  draw_spec.data_color = DPT_DEFAULT_ND_COLOR;
  draw_spec.trigger_data_color = DPT_DEFAULT_TD_COLOR;
  draw_spec.trigger_line_color = DPT_DEFAULT_TL_COLOR;
else
  if ~isfield(draw_spec, 'data_color') ...
      || isempty(draw_spec.data_color)
    draw_spec.data_color = DPT_DEFAULT_ND_COLOR;
  end
  if ~isfield(draw_spec, 'trigger_data_color') ...
      || isempty(draw_spec.trigger_data_color)
    draw_spec.trigger_data_color = DPT_DEFAULT_TD_COLOR;
  end
  if ~isfield(draw_spec, 'trigger_line_color') ...
      || isempty(draw_spec.trigger_line_color)
    draw_spec.trigger_line_color = DPT_DEFAULT_TL_COLOR;
  end
end
%
% --- end of inner_check_argument()


% --- inner_set_controls()
%
function inner_set_controls(output_file)
vb_define_process_trigger;

% check defined parameters
if ~exist('MAX_ZOOM_RANGE', 'var'), MAX_ZOOM_RANGE = 100; end;

% --- set figure
set(gcf,'WindowButtonDownFcn','vb_cb_clip_trigger down')
set(gcf,'WindowButtonMotionFcn','vb_cb_clip_trigger move')
set(gcf,'WindowButtonUpFcn','vb_cb_clip_trigger up')

% WINDOW_POSITION = [100 100 640 480];
% set(gcf, 'Position', DPT_WINDOW_POSITION);


%
% --- set initial figure's properties
%

% default output file
ud.output_file = output_file;

% LINE/RANGE switch
% ud.drag_switch = false;
ud.drag_switch = true;

% original x-range / y-range
xlim = get(gca, 'XLim');
ylim = get(gca, 'YLim');

axis_ends = [];
axis_ends = vb_axisends_set(axis_ends, 'ORIGINAL', 'X', xlim(1), xlim(2));
axis_ends = vb_axisends_set(axis_ends, 'PREVIOUS', 'X', xlim(1), xlim(2));
axis_ends = vb_axisends_set(axis_ends, 'SLIDER',   'X', xlim(1), xlim(2));

axis_ends = vb_axisends_set(axis_ends, 'ORIGINAL', 'Y', ylim(1), ylim(2));
axis_ends = vb_axisends_set(axis_ends, 'PREVIOUS', 'Y', ylim(1), ylim(2));
axis_ends = vb_axisends_set(axis_ends, 'SLIDER',   'Y', ylim(1), ylim(2));

ud.axis_ends = axis_ends;

set(gcf, 'UserData', ud);
set(gcf, 'Tag', DPT_FIGURE_TAG);
  
%
% --- set initial controllers' properties
%
if ud.drag_switch == true
  mode_str = DPT_LBL_MODE_RANGE;
else
  mode_str = DPT_LBL_MODE_LINE;
end

% --- ZOOM SLIDER
%
SLD_L = 50;
SLD_B = 10;
SLD_W = 100;
SLD_H = 15;

% --- zoom X
LBL_W = 50;
LBL_H = SLD_H;
LEFT = SLD_L;
uicontrol('Tag', 'zoom_x_label', ...
  'Style','Text', ...
  'String',DPT_LBL_X_ZOOM, ...
  'Position',[LEFT SLD_B LBL_W LBL_H]);

LEFT = LEFT + LBL_W;

sld_x = uicontrol('Tag', DPT_CTRL_TAG_ZOOM_X, ...
  'Style','Slider', ...
  'Min',1, 'Max',MAX_ZOOM_RANGE, ...
  'SliderStep', [1/MAX_ZOOM_RANGE 10/MAX_ZOOM_RANGE], ...
  'Value', 1, ...
	'Callback',['vb_cb_clip_trigger(''zoom_slide_x'')'], ...
  'Position',[LEFT SLD_B SLD_W SLD_H]);

% --- zoom Y
SLD_Y_L = 30;
SLD_Y_B = 50;
SLD_Y_W = SLD_H;
SLD_Y_H = SLD_W;
sld_y = uicontrol('Tag', DPT_CTRL_TAG_ZOOM_Y, ...
  'Style','Slider', ...
  'Min',1, 'Max',MAX_ZOOM_RANGE, ...
  'SliderStep', [1/MAX_ZOOM_RANGE 10/MAX_ZOOM_RANGE], ...
  'Value', 1, ...
	'Callback',['vb_cb_clip_trigger(''zoom_slide_y'')'], ...
  'Position',[SLD_Y_L SLD_Y_B SLD_Y_W SLD_Y_H]);

LBL_Y_L = 5;
LBL_Y_B = SLD_Y_B + SLD_Y_H;
uicontrol('Tag', DPT_LBL_Y_ZOOM, ...
  'Style', 'Text', ...
  'String', DPT_LBL_Y_ZOOM, ...
  'Position',[LBL_Y_L LBL_Y_B LBL_W LBL_H]);

% --- move Y
SLD_Y_L = SLD_Y_L;
SLD_Y_B = LBL_Y_B + LBL_H + 10;
SLD_Y_W = SLD_H;
SLD_Y_H = SLD_W;
sld_y = uicontrol('Tag', DPT_CTRL_TAG_ZOOM_Y, ...
  'Style','Slider', ...
  'Min',-100, 'Max',MAX_ZOOM_RANGE, ...
  'SliderStep', [1/MAX_ZOOM_RANGE 10/MAX_ZOOM_RANGE], ...
  'Value', 0, ...
	'Callback',['vb_cb_clip_trigger(''move_slide_y'')'], ...
  'Position',[SLD_Y_L SLD_Y_B SLD_Y_W SLD_Y_H]);

% LBL_Y_L = 5;
LBL_Y_B = SLD_Y_B + SLD_Y_H;
uicontrol('Tag', DPT_LBL_Y_MOVE, ...
  'Style', 'Text', ...
  'String', DPT_LBL_Y_MOVE, ...
  'Position',[LBL_Y_L LBL_Y_B LBL_W LBL_H]);

%
% --- base coordinats of control buttons
%
BTN_L = LEFT + SLD_W + 20;  % left
BTN_B = SLD_B;              % buttom
BTN_W = 80;                 % width
BTN_H = 20;                 % height
BTN_I = BTN_W + 5;          % interval between left positions of buttons

% --- MODE SWITCH LABEL
%
LEFT = BTN_L;

% --- MODE SWITCH
%
% LEFT = LEFT + BTN_I;
control_btn_1 = uicontrol(gcf, ...
  'Tag',   DPT_CTRL_TAG_MODE_SWITCH, ...
  'Style', 'Pushbutton', ...
  'Position', [LEFT BTN_B BTN_W BTN_H], ...
  'String', mode_str, ...
  'Callback', ['vb_cb_clip_trigger(''mode'')'] ...
  );

% --- ZOOMIN
%
LEFT = LEFT + BTN_I;
control_btn_2 = uicontrol(gcf, ...
  'Style', 'Pushbutton', ...
  'Position', [LEFT BTN_B BTN_W BTN_H], ...
  'String', DPT_LBL_ZOOMIN, ...
  'Callback', ['vb_cb_clip_trigger(''zoomin'')'] ...
  );

% --- PREVIOUS
%
LEFT = LEFT + BTN_I;
control_btn_3 = uicontrol(gcf, ...
  'Style', 'Pushbutton', ...
  'Position', [LEFT BTN_B BTN_W BTN_H], ...
  'String', DPT_LBL_PREVIOUS, ...
  'Callback', ['vb_cb_clip_trigger(''prev'')'] ...
  );

% --- RESTORE
%
% RESTORE_LABEL = 'Restore';
LEFT = LEFT + BTN_I;
control_btn_4 = uicontrol(gcf, ...
  'Style', 'Pushbutton', ...
  'Position', [LEFT BTN_B BTN_W BTN_H], ...
  'String', DPT_LBL_INITIALIZE, ...
  'Callback', ['vb_cb_clip_trigger(''restore'')'] ...
  );

% --- SAVE
%
LEFT = LEFT + BTN_I;
control_btn_5 = uicontrol(gcf, ...
  'Style', 'Pushbutton', ...
  'Position', [LEFT BTN_B BTN_W BTN_H], ...
  'String', DPT_LBL_SAVE, ...
  'Callback', ['vb_cb_clip_trigger(''save'')'] ...
  );

% --- AUX
%
% LEFT = LEFT + BTN_I;
% control_btn_6 = uicontrol(gcf, ...
%   'Style', 'Pushbutton', ...
%   'Position', [LEFT BTN_B BTN_W BTN_H], ...
%   'String', 'aux_1', ...
%   'Callback', ['vb_cb_clip_trigger(''aux_1'')'] ...
%   );

select_xrange('init', gca);
%
% --- end of inner_set_controls()

%%% END OF FILE %%%
