function	fileinfo = vb_get_active_cond(fileinfo,cond_list,session_no)
%  Get active trial index with given conditions and sessions
%   fileinfo = vb_get_active_cond(fileinfo,cond_list,session_no)
% --- Input
% cond_list  : list of condition to select
%              if empty, all conditions are selected
% session_no : session number to select
%              if empty, all sessions are selected
% fileinfo : structure with file information for multiple files
% fileinfo.filename{n}  : n-th session data file names [n=1:Nsession]
% fileinfo.Nchannel     : # of total channels
% fileinfo.channel_id   : active channel id [1 x (# of acticeChannel)]
% fileinfo.Nsample      : # of samples in one trial
% fileinfo.Ntotal       : # of all trials 
% fileinfo.Ntrial       : # of trials for each session [1 x Nsession]
% fileinfo.session_id   : session index for each trial [1 x Ntotal]
% fileinfo.cond_id      : condition number for each trials [1 x Ntotal]
%  - The above fields have info. for all trials in multiple epoch files
%  - following fields are active trial info.
% fileinfo.act_trial    : active trial index among all trials [1 x Nactive]
% --- Output (fields changed from input)
% Select active trials with given conditions and sessions
% fileinfo.act_trial    : active trial index  [1 x Nactive]
% fileinfo.cond_list  : selected condition numbers
% fileinfo.session_no : selected session numbers
%
% Masa-aki Sato 2008-8-10
% Masa-aki Sato 2009-1-31 ver.2
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if nargin < 2, cond_list  = [];end
if nargin < 3, session_no = [];end

% If no condition is given, return
if isempty(cond_list) && isempty(session_no), return; end;

% Number of session
Nsession = length(fileinfo.filename);

if isempty(cond_list)
	cond_list = 1:max(fileinfo.cond_id);
end
if isempty(session_no)
	session_no = 1:Nsession;
end

Ntotal = fileinfo.Ntotal;
Ncond  = length(cond_list);

% --- Set selected condition flag
cond_flg = zeros(1,Ntotal);

for n=1:Ncond
	ix = find( fileinfo.cond_id == cond_list(n));
	cond_flg(ix) = 1;
end

% --- Set selected session flag
session_flg = zeros(1,Ntotal);

for n = session_no
	ix = find( fileinfo.session_id == n );
	session_flg(ix) = 1;
end

% --- Set active trial flag
act_flg = zeros(1,Ntotal);
if isfield(fileinfo, 'act_trial')
    act_flg(fileinfo.act_trial) = 1;
else
    act_flg = fileinfo.ActiveTrial;
end

% --- Take AND of condition & session & active trial
act_flg = (cond_flg .* session_flg .* act_flg);

if isfield(fileinfo, 'act_trial')
    act_trial = find( act_flg > 0 );
    fileinfo.act_trial = act_trial;
else
    fileinfo.ActiveTrial = act_flg;
end
fileinfo.cond_list = cond_list;
fileinfo.session_no = session_no;
