function str = vb_struct2executable_str(s, struct_name)
% Convert struct to string.
% The string can reproduce the structure using eval function.
% [USAGE]
%    str = vb_struct2excutable_str(s [,struct_name]);
% [IN]
%              s : <<struct>> structure.
%    struct_name : (opt) <<string>> struct name. 
%                    The program put the struct name into the string.
%                    (i.e. 's.foo = ...')
%                    'inputname' function is used for detecting the name of 
%                    the struct. But in case of struct in array,
%                    inputname cannot detect the struct name.
%                    (error: vb_struct2_executable_str(structarray{1});)
%                    Specify the second argument in such a case.
%                    vb_struct2_executable_str(structarray{1}, 'structarray{1}');
% [OUT]
%            str : converted string. eval(str) reproduce the original struct.
%
% [Restriction]
%   - Over three-dimensional cell is not supported.
%   - Over three-dimensional matrix is not supported.
%
% [example]
%  s = struct;
%  s.foo = 'test';
%  s.bar = {'cell1', 'cell2'};
%  s.val = [1,2,3; 4,5,6];
%
%  str = vb_struct2executable_str(s)
%  str =
%  s.foo = 'test';
%  s.bar{1,1} = 'cell1';
%  s.bar{1,2} = 'cell2';
%  s.val = [1  2  3; 4  5  6; ];
%
%  eval(str); % reproduce s
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('s', 'var')
    error('Structure is not specified.');
end
str = '';

%
% --- Main Procedure
%
if ~exist('struct_name', 'var')
    struct_name = inputname(1);
end

if iscell(s)
    elements = size(s);
    if length(elements) >= 3
        error('Over three dimension cell is not supported.');
    end
    for m=1:elements(1)
        for n=1:elements(2)
            element = s{m,n};
            if iscell(element) || isstruct(element)
                str = [str, vb_struct2executable_str(element, [struct_name, '{', num2str(m), ',', num2str(n), '}'])];
            else
                str = [str, Put(element, [struct_name, '{', num2str(m), ',', num2str(n), '}'])];
            end
        end
    end
elseif isstruct(s)

    Nstruct = length(s);
    struct_name_m = struct_name;
    for m=1:Nstruct
        single_struct = s(m);
        fields = fieldnames(single_struct);
        Nfield = length(fields);
        if Nstruct>1
            struct_name_m = [struct_name, '(', num2str(m), ')'];
        end
        for k=1:Nfield
            field_name  = fields{k};
            field_value = single_struct.(field_name);
            if isstruct(field_value) || iscell(field_value)
                str = [str, vb_struct2executable_str(field_value, [struct_name_m, '.', field_name])];
            else
                str = [str, Put(field_value, [struct_name_m, '.', field_name])];
            end
        end
    end
else
    str = [str, Put(s, struct_name)];
end

function str = Put(target, str)

if ischar(target)
    % Put string(char)
    tmp = [' = ''' target ''';'];
    str = [str, tmp, sprintf('\n')];
else
    % Put matrix
    tmp = ' = [';
    [M, N] = size(target);
    for m=1:M
        tmp = [tmp, num2str(target(m, :))];
        tmp = [tmp, '; '];
    end
    if ~isempty(target)
        tmp(end-1:end) = []; % remove tail '; '.
    end
    tmp = [tmp, '];'];
    if M==1 && N==1
        tmp = strrep(tmp, '[', '');
        tmp = strrep(tmp, ']', '');
    end
    str = [str, tmp, sprintf('\n')];
end
