function [data_dir, filenames] = vb_saver_cmn_make_binary_file(proc_spec)
% (utility to save data) make binary file of channel data
% [usage]
%   [data_dir, filenames] = vb_saver_cmn_make_binary_file(proc_spec)
%
% [input]
%   proc_spec : <required> <<struct>> condition to save
%             :  following fields are all required
%             :  .data      : channel data  [n_channel x n_sample]
%             :  .ch_label  : channel label {1 x n_channel}
%             :  .dir       : directory where channel data will be saved
%             :  .ext       : extension of channel data file
%             :             : e.g. (MEG) ch.meg.dat, (EEG) ch.eeg.dat
%             :  .precision : precision to be saved
%             :             : e.g. 'float64'
%
% [output]
%    data_dir : data directory
%   filenames : file names of channel data to be stored
%
% [note]
%   the number of channel of proc_spec.data and proc_spec.ch_label
%   must be the same.
%
%   See also:
%   vb_define_device.m
%
% [history]
%   2013-02-07 (Sako) initial version

% --- CHECK ARGUMETNS --- %
if ~exist('proc_spec', 'var'), proc_spec = []; end
[proc_spec] = inner_check_arguments(proc_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
n_ch = size(proc_spec.data, 1);
data_dir = proc_spec.dir;
filenames = cell(1, n_ch);

for i_ch = 1:n_ch
  
  filename = sprintf('%s%s', proc_spec.ch_label{i_ch}, proc_spec.ext);
  new_file = sprintf('%s/%s', data_dir, filename);
  fid = fopen(new_file, 'wb');
  if fid == -1
    warining('(%s) cannot open file : %s\n', mfilename, new_file);
    continue;
  end
  fwrite(fid, proc_spec.data(i_ch, :), proc_spec.precision);
  fclose(fid);
  
  filenames{i_ch} = filename;
end

return

end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [proc_spec] = inner_check_arguments(proc_spec)
func_ = mfilename;

% --- proc_spec
if isempty(proc_spec)
  error('(%s) proc_spec is a required parameter', func_);
end

% ----- proc_spec.data
if ~isfield(proc_spec, 'data') || isempty(proc_spec.data)
  error('(%s) proc_spec.data is a required field', func_);
end

% ----- proc_spec.ch_label
if ~isfield(proc_spec, 'ch_label') || isempty(proc_spec.ch_label)
  error('(%s) proc_spec.ch_label is a required field', func_);
end

if ~iscell(proc_spec.ch_label)
  proc_spec.ch_label = {proc_spec.ch_label};
end

n_ch_data = size(proc_spec.data, 1);
n_ch_label = length(proc_spec.ch_label);

if n_ch_data ~= n_ch_label
  error('(%s) the number of channel must be the same data and ch_label', ...
    func_);
end

% ----- proc_spec.dir
if ~isfield(proc_spec, 'dir') || isempty(proc_spec.dir)
  error('(%s) proc_spec.dir is a required field', func_);
end

% ----- proc_spec.precision
if ~isfield(proc_spec, 'precision') || isempty(proc_spec.precision)
  error('(%s) proc_spec.precision is a required parameter', func_);
end

% ----- proc_spec.ext
if ~isfield(proc_spec, 'ext') || isempty(proc_spec.ext)
  error('(%s) proc_spec.ext is a required parameter', func_);
end

if proc_spec.ext(1) ~= '.'
  proc_spec.ext = ['.' proc_spec.ext];
end

if exist(proc_spec.dir, 'dir') ~= 7
  fprintf('(%s) make directory : %s\n', func_, proc_spec.dir);
  mkdir(proc_spec.dir);
end
return
end
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %


% --- END OF FILE --- %
