function [IsOK] = vb_dlg_compare_list(name_list1, title1, name_list2, title2, figure_title)
% dialog for compare list.
% [USAGE]
%    [IsOK] = vb_dlg_compare_list(<name_list1>, <title1>,...
%                              <name_list2>, <title2> [,figure_title]);
% [IN]
%      name_list1 : name list1                       {Nx1}
%          title1 : explanation string of name_list1 [string]
%      name_list2 : name list2                       {Mx1}
%          title2 : explanation string of name_list2 [string]
%    figure_title : figure title bar string          [string]
% [OUT]
%    IsOK : = true  : user pushed OK
%           = false : user pushed Cancel
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('name_list1', 'var')
    error('name_list1 is a required parameter.');
end
if ~exist('title1', 'var')
    error('title1 is a required parameter.');
end
if ~exist('name_list2', 'var')
    error('name_list2 is a required parameter.');
end
if ~exist('title2', 'var')
    error('title2 is a required parameter.');
end
if ~exist('figure_title', 'var'), figure_title = ''; end

%
% --- Main Procedure
%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% figure_title
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%<--->
% LEFT_MARGIN
%
%        title1        title2
%         WIDTH         WIDTH
%    H|------------||------------| 
%    E| name_list1 || name_list2 | 
%    I|            ||            | 
%    G|            ||            | 
%    H|            ||            | 
%    T|------------||------------| 
% =
% | BOTTOM_MARGIN
% =
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

BOTTOM_MARGIN = 50;
LEFT_MARGIN   = 10;
WIDTH         = 200;
HEIGHT        = 300;

% Create figure and two Listboxes
h = figure('Visible', 'off', 'WindowStyle', 'modal');
set(h, 'Name', figure_title);
set(h, 'NumberTitle', 'off');
listbox1 = ...
    uicontrol(...
        'Parent', h, 'style', 'listbox', ...
        'Position', [LEFT_MARGIN, BOTTOM_MARGIN, WIDTH, HEIGHT], 'String', name_list1);
listbox2 = ...
    uicontrol(...
        'Parent', h, 'style', 'listbox', ...
        'Position', [LEFT_MARGIN + WIDTH, BOTTOM_MARGIN, WIDTH, HEIGHT], 'String', name_list2);

% Create title1, title2 text
LABEL_HEIGHT = 20;
LABEL_WIDTH  = WIDTH;
uicontrol('Parent', h, 'style', 'Text', ...
          'String', title1,...
          'HorizontalAlignment', 'Left', ...
          'Position', [LEFT_MARGIN, BOTTOM_MARGIN + HEIGHT, LABEL_WIDTH, LABEL_HEIGHT]);
uicontrol('Parent', h, 'style', 'Text', ...
          'String', title2, ...
          'HorizontalAlignment', 'Left', ...
          'Position', [LEFT_MARGIN+WIDTH, BOTTOM_MARGIN + HEIGHT, LABEL_WIDTH, LABEL_HEIGHT]);

BUTTON_WIDTH  = WIDTH/2;
BUTTON_HEIGHT = BOTTOM_MARGIN/2;

% Create OK, CANCEL button
ok_btn = uicontrol(...
    'style','pushbutton',...
    'string', 'OK',...
    'Position',[LEFT_MARGIN + WIDTH, BOTTOM_MARGIN-BUTTON_HEIGHT, BUTTON_WIDTH, BUTTON_HEIGHT],...
    'callback',{@doOK});

cancel_btn = uicontrol(...
    'style','pushbutton',...
    'string','Cancel',...
    'Position',[LEFT_MARGIN + WIDTH+BUTTON_WIDTH, BOTTOM_MARGIN-BUTTON_HEIGHT, BUTTON_WIDTH, BUTTON_HEIGHT],...
    'callback', {@doCancel});

% Sort button
CHECKBOX_WIDTH = WIDTH;
CHECKBOX_HEIGHT = BOTTOM_MARGIN/2;

uicontrol(...
    'style', 'checkbox', ...
    'string', 'Sort', ...
    'Position', [LEFT_MARGIN, BOTTOM_MARGIN - CHECKBOX_HEIGHT, CHECKBOX_WIDTH, CHECKBOX_HEIGHT],...
    'callback', {@doSort, listbox1, listbox2});

% Set figure size
fig_pos = get(h, 'Position');
fig_pos(3) = WIDTH * 2 + LEFT_MARGIN * 2;
fig_pos(4) = BOTTOM_MARGIN + HEIGHT + LABEL_HEIGHT + 10;
set(h, 'Position', fig_pos, ...
    'MenuBar', 'none', 'Visible', 'on', 'Resize', 'off',...
    'closerequestfcn', {@doCancel});


% make initial data
data = struct;
data.IsOK = false;
data.name_list1 = name_list1;
data.name_list2 = name_list2;
guidata(h, data);

waitfor(h, 'Visible', 'off');

% get result
data = guidata(h);
IsOK = data.IsOK;

% delete figure
delete(h);

return;
%-------------------------------------------------end of function

%
% --- Callback functions(Flag ON/OFF)
%
function doOK(ok_btn, evd)
    result = struct;
    result.IsOK = true;
    guidata(gcbf, result);
    set(gcbf, 'Visible', 'off');

function doCancel(cancel_btn, evd)
    result = struct;
    result.IsOK = false;
    guidata(gcbf, result);
    set(gcbf, 'Visible', 'off');

function doSort(checkbox, evd, listbox1, listbox2)
    % load original data
    data = guidata(gcbf);
    
    if get(checkbox, 'Value')
        name_list1 = sort(data.name_list1);
        name_list2 = sort(data.name_list2);
        set(listbox1, 'String', name_list1);
        set(listbox2, 'String', name_list2);
    else
        set(listbox1, 'String', data.name_list1);
        set(listbox2, 'String', data.name_list2);
    end
        
