function [bexp, bexp_ext, refmg] = ...
  vb_combine_meg_files(in_files, out_file, ext_spec)
% Combine .meg.mat files (for VBMEG version 0.8)
%
% --- Syntax
% [bexp, bexp_ext, refmg] = vb_combine_meg_files(in_files, out_file, ext_spec)
%
% --- Input
% in_files (cell array): .meg.mat files to be combined
% out_file (String)    : Combined .meg.mat file is saved to this file. 
% ext_spec (struct) : extra specification
%          .saveman : <optional> <<struct>>
%                   :   .data_dir : directory (relative path) for external file
%                   :   .precision : usually 'float64'
%
% --- Output
%   concatenated data as folows:
%   bexp     [Nchannel x Nsample x Ntrial]
%   bexp_ext [Nchannel x Nsample x Ntrial]
%   refmeg   [Nchannel x Nsample x Ntrial]
%
%   !!!NOTICE!!! If data is stored externally, these parameters will be empty
%
% --- Note
% [preconditions]
%  * following data must correspond with each other
%  * Measurement
%  * CoordType
%  * MEGinfo
%     .Nchannel
%     .Nsample
%     .device
%     .SampleFreq
%     .Pretrigger
%    (.sensor_weight)
%    (.MEGch_id)
%    (.MEGch_name)
%
% [postconditions]
% 
%  * data format after combining is as follows:
%    (Sensor position will be averaged across .meg.mat files.
%     Accordingly, it is assumed that MEG data in combined files 
%     were recorded without head movement.)
%    pick
%    Qpick
%    ref_pick
%    ref_Qpick
% 
%    (apply the values of the first input file)
%    CoordType
%    Measurement
% 
%    (be concatenated in "trial" dimension)
%    bexp
%    bexp_ext
%    refmg
% 
%    (the fields of MEGinfo if it exist)
%    MEGinfo (standard format)
%      .Measurement   (apply the value of the first file)
%      .device        (apply the value of the first file)
%      .Nchannel      (apply the value of the first file)
%      .Nsample       (apply the value of the first file)
%      .Nrepeat       (summation of all input data)
%      .SampleFreq    (apply the value of the first file)
%      .Pretrigger    (apply the value of the first file)
%      .sensor_weight (apply the value of the first file)
%      .MEGch_id      (apply the value of the first file)
%      .MEGch_name    (apply the value of the first file)
%      .ActiveChannel (apply the value of the first file)
%      .ActiveTrial   (apply the value of the first file)
%      .Vcenter       (average)
%      .Vradius       (average)
%      .MEG_ID        (set empty)
%      .MRI_ID        (apply the value of the first file)
%      .Trial
%      .ChannelInfo
%      .ExtraChannelInfo
%      .saveman
%
% --- History
% 2007-01-12 (Taku Yoshioka) initial version for VBMEG 0.5
% 2009-09-02 (Sako) modified according to the latest format and added return
% 2010-08-31 (Sako) modified combined file could be used as a input file of
%                   vb_load_meg_data file with loadspec.
% 2011-02-24 (Sako) when making new file, sampling data is stored outer
% 2011-06-01 (Sako) modified according to the new standard format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('in_files', 'var'), in_files = []; end
if ~exist('out_file', 'var'), out_file = []; end
if ~exist('ext_spec', 'var'), ext_spec = []; end
[in_files, out_file, info, measurement, coord_type, saveman, d_path] = ...
  inner_check_arguments(in_files, out_file, ext_spec);

% [pick,Qpick] = vb_load_sensor(in_files{1});

% --- initialize base variables
pick_sum      = [];
Qpick_sum     = [];
ref_pick_sum  = [];
ref_Qpick_sum = [];

vcenter_sum   = [];
vradius_sum   = 0;

n_pick      = 0;
n_Qpick     = 0;
n_ref_pick  = 0;
n_ref_Qpick = 0;

n_vcenter = 0;
n_vradius = 0;

n_trial   = 0;

n_file = length(in_files);

% --- when you want to store data outside
if ~isempty(saveman)
  % ----- common information to store externally
  const = vb_define_yokogawa;
  data_ext = const.EXT_CHANNEL_BIN;

  prec = vb_saveman_get_precision(saveman);
end

bexp = [];
bexp_ext = [];
refmg = [];

% --- combine file by file
for i_file = 1:n_file
  load(in_files{i_file}, 'pick', 'Qpick', 'ref_pick', 'ref_Qpick');
  
  % --- sensor coordinates
  if exist('pick', 'var')
    if isempty(pick_sum)
      pick_sum = zeros(size(pick,1), size(pick,2));
    end
    pick_sum = pick_sum + pick;
    n_pick = n_pick + 1;
  end

  if exist('Qpick', 'var')
    if isempty(Qpick_sum)
      Qpick_sum = zeros(size(Qpick,1), size(Qpick,2));
    end
    Qpick_sum = Qpick_sum + Qpick;
    n_Qpick = n_Qpick + 1;
  end

  if exist('ref_pick', 'var')
    if isempty(ref_pick_sum)
      ref_pick_sum = zeros(size(ref_pick,1), size(ref_pick,2));
    end
    ref_pick_sum = ref_pick_sum + ref_pick;
    n_ref_pick = n_ref_pick + 1;
  end

  if exist('ref_Qpick', 'var')
    if isempty(ref_Qpick_sum)
      ref_Qpick_sum = zeros(size(ref_Qpick,1), size(ref_Qpick,2));
    end
    ref_Qpick_sum = ref_Qpick_sum + ref_Qpick;
    n_ref_Qpick = n_ref_Qpick + 1;
  end
  
  if isfield(info{i_file}, 'Vcenter')
    if isempty(vcenter_sum)
      vcenter_sum = ...
        zeros(size(info{i_file}.Vcenter,1), size(info{i_file}.Vcenter,2));
    end
    vcenter_sum = vcenter_sum + info{i_file}.Vcenter;
    n_vcenter = n_vcenter + 1;
  end
  
  if isfield(info{i_file}, 'Vradius')
    vradius_sum = vradius_sum + info{i_file}.Vradius;
    n_vradius = n_vradius + 1;
  end
  
  % --- data
  loadspec.ActiveChannel = 0;
  loadspec.ChannelType = 'MEG';
  cur_bexp = vb_load_meg_data(in_files{i_file}, loadspec);
  
  loadspec.ChannelType = 'EXTRA';
  cur_bexp_ext = vb_load_meg_data(in_files{i_file}, loadspec);

  loadspec.ChannelType = 'REFERENCE';
  cur_refmg = vb_load_meg_data(in_files{i_file}, loadspec);
  
  % --- get trial info
  if ~isempty(cur_bexp)
    trial_begin = n_trial + 1;
    trial_end   = trial_begin + size(cur_bexp, 3) - 1;
    n_trial = trial_end;
    
  elseif ~isempty(cur_bexp_ext)
    trial_begin = n_trial + 1;
    trial_end   = trial_begin + size(cur_bexp_ext, 3) - 1;
    n_trial = trial_end;

  elseif ~isempty(cur_refmg)
    trial_begin = n_trial + 1;
    trial_end   = trial_begin + size(cur_refmg, 3) - 1;
    n_trial = trial_end;
  end

  if ~isempty(saveman)
    meginfo = info{1};
  end
  
  if ~isempty(cur_bexp)
    if ~isempty(saveman)
      meg_labels = vb_meginfo_get_channel_label_meg(meginfo);

      n_ch = length(meg_labels);
      for i_ch = 1:n_ch
        vb_util_make_external_data_file(d_path, meg_labels{i_ch}, ...
          data_ext, prec, cur_bexp(i_ch,:,:), 'ab' );
      end

    else
      bexp(:,:, (trial_begin:trial_end)) = cur_bexp;
    end
  end

  if ~isempty(cur_bexp_ext)
    if ~isempty(saveman)
      ext_labels = vb_meginfo_get_channel_label_extra(meginfo);

      n_ch = length(ext_labels);
      for i_ch = 1:n_ch
        vb_util_make_external_data_file(d_path, ext_labels{i_ch}, ...
          data_ext, prec, cur_bexp_ext(i_ch,:,:), 'ab' );
      end

    else
      bexp_ext(:,:, (trial_begin:trial_end)) = cur_bexp_ext;
    end
  end

  if ~isempty(cur_refmg)
    if ~isempty(saveman)
      ref_labels = vb_meginfo_get_channel_label_refmg(meginfo);
      
      n_ch = length(ref_labels);
      for i_ch = 1:n_ch
        vb_util_make_external_data_file(d_path, ref_labels{i_ch}, ...
          data_ext, prec, cur_refmg(i_ch,:,:), 'ab' );
      end

    else
      refmg(:,:, (trial_begin:trial_end)) = cur_refmg;
    end
  end
end

% --- sensor cordinates are by averaging
if n_pick > 0
  pick = pick_sum / n_pick;
end

if n_Qpick > 0
  Qpick = Qpick_sum / n_Qpick;
end

if n_ref_pick > 0
  ref_pick = ref_pick_sum / n_ref_pick;
end

if n_ref_Qpick > 0
  ref_Qpick = ref_Qpick_sum / n_ref_Qpick;
end

% --- new MEGinfo
if isfield(info{1}, 'Measurement')
  MEGinfo.Measurement = info{1}.Measurement;
else
  MEGinfo.Measurement = measurement{1};
end
MEGinfo.device     = info{1}.device;
MEGinfo.Nchannel   = info{1}.Nchannel;
MEGinfo.Nsample    = info{1}.Nsample;
MEGinfo.Nrepeat    = n_trial;
MEGinfo.SampleFreq = info{1}.SampleFreq;
MEGinfo.Pretrigger = info{1}.Pretrigger;
MEGinfo.MEG_ID     = '';
MEGinfo.MRI_ID     = info{1}.MRI_ID;

if ~isempty(saveman)
  MEGinfo.saveman = saveman;
end

if isfield(info{1}, 'sensor_weight')
  MEGinfo.sensor_weight = info{1}.sensor_weight;
end

if isfield(info{1}, 'MEGch_id')
  MEGinfo.MEGch_id = info{1}.MEGch_id;
end

if isfield(info{1}, 'MEGch_name')
  MEGinfo.MEGch_name = info{1}.MEGch_name;
end

if n_vcenter > 0
  MEGinfo.Vcenter = vcenter_sum / n_vcenter;
end

if n_vradius > 0
  MEGinfo.Vradius = vradius_sum / n_vradius;
end

% --- for not minimum version
MEGinfo.Trial = [];
MEGinfo.ActiveTrial = [];
prev_n_trial = 0;
prev_n_sample = 0;

for i_file = 1:n_file
  cur_n_trial = length(info{i_file}.Trial);
  cur_n_sample = info{i_file}.Nsample;
  
  for i_tr = 1:cur_n_trial
    info{i_file}.Trial(i_tr).sample = ...
      info{i_file}.Trial(i_tr).sample + prev_n_sample;
    info{i_file}.Trial(i_tr).number = ...
      info{i_file}.Trial(i_tr).number + prev_n_trial;
  end
  MEGinfo.Trial = [MEGinfo.Trial; info{i_file}.Trial];
  MEGinfo.ActiveTrial = [MEGinfo.ActiveTrial; info{i_file}.ActiveTrial];
  prev_n_trial = prev_n_trial + cur_n_trial;
  prev_n_sample = prev_n_sample + cur_n_sample;
end

if isfield(info{1}, 'ChannelInfo')
  MEGinfo.ChannelInfo = info{1}.ChannelInfo;
end

if isfield(info{1}, 'ActiveChannel')
  MEGinfo.ActiveChannel = info{1}.ActiveChannel;
end

if isfield(info{1}, 'ExtraChannelInfo')
  MEGinfo.ExtraChannelInfo = info{1}.ExtraChannelInfo;
end

% --- top header
Measurement = measurement{1};
CoordType   = coord_type{1};

% ----- PositionFile
target_key = 'PositionFile';
PositionFile = '';

for  i_file = 1:n_file
  [state, guide_def] = ...
    vb_util_check_variable_in_matfile(in_files{i_file}, target_key);
  if state == guide_def.VALID
    load(in_files{i_file}, target_key);
    break;
  end
end

if ~isempty(out_file)
  if ~isempty(saveman)
    if (n_ref_pick > 0) && (n_ref_Qpick > 0)
      vb_fsave(out_file, 'Measurement', ...
        'pick','Qpick', 'ref_pick', 'ref_Qpick', 'CoordType', 'MEGinfo', ...
        'PositionFile');
    else
      vb_fsave(out_file, 'Measurement', ...
        'pick','Qpick', 'CoordType', 'MEGinfo', ...
        'PositionFile');
    end
  else
    if (n_ref_pick > 0) && (n_ref_Qpick > 0)
      vb_fsave(out_file, 'Measurement', 'bexp', 'bexp_ext', 'refmg', ...
        'pick','Qpick', 'ref_pick', 'ref_Qpick', 'CoordType', 'MEGinfo', ...
        'PositionFile');
    else
      vb_fsave(out_file, 'Measurement', 'bexp', 'bexp_ext', 'refmg', ...
        'pick','Qpick', 'CoordType', 'MEGinfo', ...
        'PositionFile');
    end
  end
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [in_files, out_file, info, measurement, coord_type, ...
  saveman, real_dir] = ...
  inner_check_arguments(in_files, out_file, ext_spec)
func_ = mfilename;

if isempty(in_files)
  error('(%s) in_files is a required parameter', func_);
end

if isempty(out_file)
  % require no actions
end

if ~iscell(in_files)
  in_files = {in_files};
end

% --- check in_files
n_files = length(in_files);

for i_file = 1:n_files
  if exist(in_files{i_file}, 'file') ~= 2
    error('(%s) cannot find in_file{%d} (%s)', ...
      func_, i_file, in_files{i_file});
  end
  
  % --- collect information to check preconditions
  info{i_file} = vb_load_measurement_info(in_files{i_file});

  load(in_files{i_file}, 'Measurement', 'CoordType');
  measurement{i_file} = Measurement;
  coord_type{i_file} = CoordType;
end

% --- preconditions to check
%  * following data must correspond with each other
%  * Measurement
%  * CoordType
%  * MEGinfo
%     .Nchannel
%     .Nsample
%     .device
%     .SampleFreq
%     .Pretrigger
%    (.sensor_weight)
%    (.MEGch_id)
%    (.MEGch_name)

% minimum set of MEGinfo
n_channel_base   = info{1}.Nchannel;
n_sample_base    = info{1}.Nsample;
sample_freq_base = info{1}.SampleFreq;
pretrigger_base  = info{1}.Pretrigger;
device_base      = info{1}.device;

% extra info of MEGinfo
if isfield(info{1}, 'sensor_weight')
  sensor_weight_base = info{1}.sensor_weight;
else
  sensor_weight_base = [];
end

if isfield(info{1}, 'MEGch_id')
  MEGch_id_base = info{1}.MEGch_id;
else
  MEGch_id_base = [];
end

if isfield(info{1}, 'MEGch_name')
  MEGch_name_base = info{1}.MEGch_name;
else
  MEGch_name_base = {};
end

if isfield(info{1}, 'ActiveChannel')
  ActiveChannel_base = info{1}.ActiveChannel;
else
  ActiveChannel_base = {};
end

measurement_base = measurement{1};
coord_type_base  = coord_type{1};

for i_file = 2:n_files

  % --- basic fields
  % ----- Nchannel
  if info{i_file}.Nchannel ~= n_channel_base
    error('(%s - %s) all numbers of channel must be the same', ...
      func_, in_files{i_file});
  end

  % ----- Nsample
  if info{i_file}.Nsample ~= n_sample_base
    error('(%s - %s) all numbers of sample must be the same', ...
      func_, in_files{i_file});
  end

  % ----- SampleFreq
  if info{i_file}.SampleFreq ~= sample_freq_base
    error('(%s - %s) all sampling frequencies must be the same', ...
      func_, in_files{i_file});
  end

  % ----- Pretrigger
  if info{i_file}.Pretrigger ~= pretrigger_base
    error('(%s - %s) all pretriggers must be the same', ...
      func_, in_files{i_file});
  end

  % ----- device
  if ~strcmp(info{i_file}.device, device_base)
    error('(%s - %s) each device must be the same', ...
      func_, in_files{i_file});
  end

  % ----- Measurement
  if ~strcmp(measurement{i_file}, measurement_base)
    error('(%s - %s) each Measurement must be the same', ...
      func_, in_files{i_file});
  end
  
  % --- extra fields
  % ----- sensor_weight
  if isfield(info{i_file}, 'sensor_weight')
    if ~isequal(info{i_file}.sensor_weight, sensor_weight_base)
      error('(%s - %s) each sensor_weight must be the same', ...
        func_, in_files{i_file});
    end
  end

  % ----- MEGch_id
  if isfield(info{i_file}, 'MEGch_id')
    if ~isequal(info{i_file}.MEGch_id, MEGch_id_base)
      error('(%s - %s) each MEGch_id must be the same', ...
        func_, in_files{i_file});
    end
  end

  % ----- MEGch_name
  if isfield(info{i_file}, 'MEGch_name')
    if ~isequal(info{i_file}.MEGch_name, MEGch_name_base)
      error('(%s - %s) each MEGch_name must be the same', ...
        func_, in_files{i_file});
    end
  end

  % ----- CoordType
  if ~strcmp(coord_type{i_file}, coord_type_base)
    error('(%s - %s) each CoordType must be the same', ...
      func_, in_files{i_file});
  end
  
  % ----- ActiveChannel
  if isfield(info{i_file}, 'ActiveChannel')
    if ~isequal(info{i_file}.ActiveChannel, ActiveChannel_base)
      error('(%s - %s) each ActiveChannel must be the same', ...
        func_, in_files{i_file});
    end
  end

end

saveman = [];
real_dir = [];

if ~isempty(out_file)
  if isempty(ext_spec) ...
      || ~isfield(ext_spec, 'saveman') ...
      || ~vb_saveman_get_switch(ext_spec.saveman)
    [ext_spec.saveman.data_dir, real_dir] = vb_device_make_data_dir(out_file);
    ext_spec.saveman.precision = 'float64';
  else
    f_path = vb_get_file_parts(out_file);
    real_dir = [f_path '/' ext_spec.saveman.data_dir];
  end
  saveman = ext_spec.saveman;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
