function    Dinv = vb_sparse_inv_distance(X, Y, Rmax, Nblk);
% Compute sparse inverse distance matrix with (distance) < Rmax
%
% Dinv = vb_sparse_distance(X, Y, Rmax, Nblk);
% Dinv = vb_sparse_distance(X, Y, Rmax);
%
% X : (M x D) Data matrix
% Y : (N x D) Data matrix
% X(m,:) : D-dim vector for m-th point 
% Y(n,:) : D-dim vector for n-th point 
%
% M, N : total number of points for X, Y
% D : data dimension
%
% Rmax : Max distance
% Nblk : block length for block-wise distance matrix calculation
%
% Sparse inverse distance matrix with (distance) < Rmax
%   Dinv = sparse(I, J, 1./dd, M, N);
%   
% dd(n) = ||X(I(n),:) - Y(J(n),:)||  < Rmax
%
%   2017-3-6 Masa-aki Sato 

if nargin < 4 || isempty(Nblk), Nblk = 100; end;
if nargin < 3,  help vb_sparse_inv_distance; return; end
if (size(X,2) ~= size(Y,2)), error('size(X,2) ~= size(Y,2)'); end

% Rmax : Max distance
Rmax = Rmax^2;

% total number of points
M = size(X,1);
N = size(Y,1);

% Index for Sparse distance matrix
II = zeros(M,1);
JJ = zeros(M,1);
DD = zeros(M,1);

ns = 0;

if M < N
    % Loop for X
    for i1=1:Nblk:M
        i2 = i1+Nblk-1;
        if (i2> M), i2=M; end;
        
        % block-wise distance matrix
        XX = X(i1:i2,:);  
        dd = vb_sq_distance(XX,Y,1);
        
        % find neighbor points with distance < Rmax
        sw = (dd < Rmax);
        [I,J] = find( sw );
        D = dd(sw);
        
        n = length(I);
        ix = (1:n) + ns;
        
        II(ix) = I(:) + i1 - 1;
        JJ(ix) = J(:);
        DD(ix) = D(:);
        
        ns = ns + n;
        
%       II = I + i1 - 1;
%       ds = ds + sparse(II(:), J(:), DD(:), M, N);
    end;
else
    % Loop for Y
    for i1=1:Nblk:N
        i2 = i1+Nblk-1;
        if (i2> N), i2=N; end;
        
        % block-wise distance matrix
        YY = Y(i1:i2,:);  
        dd = vb_sq_distance(X,YY,1);
        
        % find neighbor points with distance < Rmax
        sw = (dd < Rmax);
        [I,J] = find( sw );
        D = dd(sw);
        
        n = length(I);
        ix = (1:n) + ns;
        
        II(ix) = I(:);
        JJ(ix) = J(:) + i1 - 1;
        DD(ix) = D(:);
        
        ns = ns + n;
        
%       JJ = J + i1 - 1;
%       ds = ds + sparse(I(:), JJ(:), DD(:), M, N);
    end;
end;


% matrix index with (distance) < Rmax 
II = II(1:ns);
JJ = JJ(1:ns);

% distance with (distance) < Rmax 
DD = DD(1:ns);

%fprintf('rate of nonzero/(MxN) = %g in sparce inverse distance\n', ...
%   100*ns/(M*N))

% set min distance for inverse distance
MinVal = Rmax * 1e-10;
DD = max(DD, MinVal);

% inverse distance 
DD = sqrt(1./DD);

% Sparse inverse distance matrix with (distance) < Rmax
Dinv = sparse(II, JJ, DD, M, N);

