function [V,F,xx,BV_index,Vinfo,Vext,Fext] = ...
                vb_make_brain_mni(brain_parm,reg_mode)
% Make cortex data from MNI standard brain for group analysis
% 1. Map MNI standard brain to subject brain
% 2. Find nearest point from subject cortical surface 
%    extracted by FreeSurfer
% 
%   [V,F,xx,ORG_index,Vinfo] = vb_make_brain_mni(brain_parm,reg_mode);
%
%--- Input
%
% brain_parm   : structure with following field
% --- fields of brain_parm
% .FS_left_file         = [fs_dir '/bem/lh.smoothwm.asc'];
% .FS_right_file        = [fs_dir '/bem/rh.smoothwm.asc'];
% 
% .FS_left_label_file  = [fs_dir '/label/lh.cortex.label'];
% .FS_right_label_file = [fs_dir '/label/rh.cortex.label'];
%
% if reg_mode=='SPM'
%  .spm_normalization_file: SPM normalization file
%                          (_sn.mat)
% if reg_mode=='FS'
%  .FS_left_sphere_file
%  .FS_right_sphere_file
%  .FS_sphere_key 
%
% 
%--- Output
% 
% V    : Cortical vertex point cordinate (SPM_Right_m) [Nvertex, 3]
%          ĺ  ĺο, 3
% xx   : Normal vector to cortical surface   [Nvertex, 3]
%           ĺˡȰ֡ ĺο, 3
% F      : Patch index structure
%           (3ѷ)룳Ĥĺֹ ( ̤ο, 3)  
%  .F3R  : Right cortex
%  .F3L  : Left  cortex
%  .F3   : Left + Right
% BV_index  : original vertex index in BV corresponding to brain
%    .Left     : Left  brain
%    .Right    : Right brain
% Vinfo        : Vertex dimension structure
%   .Ndipole   : # of vertex
%   .NdipoleL  : # of vertex in Left cortex
%   .Npatch    : # of patch
%   .Coord     = 'SPM_Right_m';
%
% [history]
% Ver.2.0 New version for group analysis
% 2015-12-27 M. Sato 
% 2017-01-30 Y. Takeda vb_find_nearest_point->vb_find_nearest_point_no_overlap
% 2017-03-06 M. Sato create Vext, Fext.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

global vbmeg_inst;
const = vb_define_verbose;

% Max distance for nearest point search
Rmax = 30.0; % 30 mm

EPS  = 1e-10;

Nvertex = brain_parm.Nvertex;

vb_disp('Load original subject brain');

%---------------------------------------------------
% Load Subject original cortical surface (SPM_Right mm cordinate)
%---------------------------------------------------
[V0L,F0L,n0L,V0R,F0R,n0R] = vb_load_orig_brain_surf(brain_parm);
% V0L/R  vertex coordinate (SPM_Right mm)
% n0L/R  unit vector 
% FOL/R  triangle patch

% Extract cortex area index of original surface
vb_disp('Extract cortex area index');

[Vindex] = vb_extract_cortex(V0L,F0L,V0R,F0R,brain_parm);
% Cortex area index of original FreeSurfer surface
% Vindex.cortexL;
% Vindex.cortexR + NdipoleL0;

%---------------------------------------------------
% Load standard brain cortical surface
%---------------------------------------------------
[std_parm, std_brain_dir] = vb_set_icbm152(Nvertex);

std_brain = [std_brain_dir filesep std_parm.brain_file];

[Vstd, Fmni] = vb_load_cortex(std_brain);
% Fmni :  triangle patch for MNI cordinate

load(std_brain,'Vinfo');

Ndipole   = Vinfo.Ndipole;
NdipoleL  = Vinfo.NdipoleL;
NdipoleR  = Ndipole - NdipoleL;

%---------------------------------------------------
% Map MNI-coordinate to individual's cortical surface
%---------------------------------------------------

switch  reg_mode
case    'SPM'
    vb_disp('SPM registration Mode');
    
    % reference coordinate: subject original SPM-R mm coordinate
    VLref = V0L;
    VRref = V0R;
    
    snfile    = brain_parm.spm_normalization_file;
    
    % target coordinate: MNI cordinate in standard brain
    Vmni = vb_load_cortex(std_brain,'MNI');
    
    % Back transformation to subject brain
    % Use mm-coordinates
    sn = load((snfile)); 
    [Vsubj] = vb_unsn(Vmni'*1000,sn);  

    % target coordinate: MNI cordinate
    % transformed to SPM-R mm-coordinates in subject brain
    VL = Vsubj(:,1:NdipoleL)';
    VR = Vsubj(:,(NdipoleL+1):Ndipole)';
    
case    'FS'
    vb_disp('FS spherical coordinate registration Mode');
    
    % reference coordinate: subject original spherical coordinate in FreeSurfer
    VLref = vb_fs_load_surface(brain_parm.FS_left_sphere_file );
    VRref = vb_fs_load_surface(brain_parm.FS_right_sphere_file);
    
    % target coordinate: spherical coordinate of MNI standard brain
    Vsph = vb_load_cortex(std_brain,'sphere.reg');
    
    VL = Vsph(1:NdipoleL,:);
    VR = Vsph((NdipoleL+1):Ndipole,:);

otherwise
    error('Registration mode error');
end

vb_disp('--- Mapping MNI-coordinate onto a subject-brain');
%---------------------------------------------------
% Find nearest point from 'V' to 'VRref'
%---------------------------------------------------

% vertex number of reference original FreeSurfer surface
NLref  = size(VLref,1);
NRref  = size(VRref,1);

% Cortex area index of original FreeSurfer surface
ref_corL = Vindex.cortexL;
ref_corR = Vindex.cortexR;
ref_nonL = vb_setdiff2([1:NLref], ref_corL);
ref_nonR = vb_setdiff2([1:NRref], ref_corR);

% Cortex area index of MNI brain model
corL = Vinfo.cortexL;
corR = Vinfo.cortexR - NdipoleL;
nonL = vb_setdiff2([1:NdipoleL], corL);
nonR = vb_setdiff2([1:NdipoleR], corR);

% Cortex area vertex of original FreeSurfer surface
VLref_cor = VLref(ref_corL,:);
VRref_cor = VRref(ref_corR,:);
VLref_non = VLref(ref_nonL,:);
VRref_non = VRref(ref_nonR,:);

% Cortex area vertex of brain model
VL_cor = VL(corL,:);
VR_cor = VR(corR,:);
VL_non = VL(nonL,:);
VR_non = VR(nonR,:);

%--> Find nearest point from 'V' to 'Vref'
[IL_cor, ddL_cor] = vb_find_nearest_point_no_overlap(VLref_cor, VL_cor,Rmax);
[IR_cor, ddR_cor] = vb_find_nearest_point_no_overlap(VRref_cor, VR_cor,Rmax);
[IL_non, ddL_non] = vb_find_nearest_point_no_overlap(VLref_non, VL_non,Rmax);
[IR_non, ddR_non] = vb_find_nearest_point_no_overlap(VRref_non, VR_non,Rmax);

% index of brain model mapped from MNI brain
IndxL = zeros(NdipoleL,1);
IndxR = zeros(NdipoleR,1);

IndxL(corL) = ref_corL(IL_cor);
IndxR(corR) = ref_corR(IR_cor);
IndxL(nonL) = ref_nonL(IL_non);
IndxR(nonR) = ref_nonR(IR_non);

ddL = zeros(NdipoleL,1);
ddR = zeros(NdipoleR,1);

ddL(corL) = ddL_cor;
ddR(corR) = ddR_cor;
ddL(nonL) = ddL_non;
ddR(nonR) = ddR_non;

dd = [ddL; ddR];

% % Cortex area index of original FreeSurfer surface
% NLref  = size(VLref,1);
% NRref  = size(VRref,1);
fprintf('# of subj_orig total vertex     = %d (L), %d (R)\n', ...
    NLref,NRref)
fprintf('# of subj_orig cortical vertex  = %d (L), %d (R)\n',...
    length(ref_corL),length(ref_corR))
fprintf('# of target cortical vertex     = %d (L), %d (R)\n',...
    length(corL),length(corR))
fprintf('# of target non-cortical vertex = %d (L), %d (R)\n\n',...
    length(nonL),length(nonR))

fprintf('Max (Mean) distance in cortex     = %4.1f (%4.1f) mm \n', ...
    max([ddL_cor(:); ddR_cor(:)]),mean([ddL_cor(:); ddR_cor(:)]))
fprintf('Max (Mean) distance in non-cortex = %4.1f (%4.1f) mm \n', ...
    max([ddL_non(:); ddR_non(:)]),mean([ddL_non(:); ddR_non(:)]))

%---------------------------------------------------
% --- SPM cordinate in subject brain [m] 
%---------------------------------------------------
VL = V0L(IndxL,:);
VR = V0R(IndxR,:);
V   = [VL; VR]/1000;

%---------------------------------------------------
% ---  Normal vector obtained from original FS-model
%---------------------------------------------------
n3L = n0L(IndxL,:);
n3R = n0R(IndxR,:);
xx  = [n3L ; n3R]; 
[xx] = check_normal_vector(xx);

%---------------------------------------------------
% --- Get patch information for V
%---------------------------------------------------
%
% --- Reference coordinate: 
%     Reduced Subject SPM-R mm cordinate
%
[FLs,VLs] = vb_reducepatch( F0L, V0L, Nvertex ); 
[FRs,VRs] = vb_reducepatch( F0R, V0R, Nvertex ); 

%
% ---- Check closed surface by solid_angle/(2*pi) = 1
%
[FLs, VLs, xxLs] = check_surface(FLs,VLs);
[FRs, VRs, xxRs] = check_surface(FRs,VRs);

%%---------------------------------------------------
% --- Find nearest point of VL from Reduced subject cordinate VLs
%---------------------------------------------------

%--> Find nearest point from 'VL','VR' to 'VLs','VRs' 
[IL, ddL,VLs,FLs] = vb_find_no_overlap_divide(VLs, VL, FLs,Rmax);
[IR, ddR,VRs,FRs] = vb_find_no_overlap_divide(VRs, VR, FRs,Rmax);

[Vext,FL,FR] = vb_calc_subj_patch(IL,IR,VLs,FLs,VRs,FRs);

% Add extra vertex to define complete reduced patch
Vext = [V; Vext];

% reduced patch for extended vertex (Vext)
Fext.F3  = [FL ; FR];
Fext.F3R = FR;
Fext.F3L = FL;

% reduced patch for model vertex (V)
% select patch whose vertices consist of 'V'
%FL = inner_patch_select([1:NdipoleL], FL);
%FR = inner_patch_select([(1:NdipoleR) + NdipoleL], FR);
%F.F3  = [FL ; FR];
%F.F3R = FR;
%F.F3L = FL;
%
%F.NdipoleL = NdipoleL;
%F.Fmni = Fmni;

F = [];

%---------------------------------------------------
% ---  Vinfo : Dimensional info
%---------------------------------------------------

NdipoleL0 = size(VLref,1);
Ndipole0  = size(VRref,1) + NdipoleL0;

%---------------------------------------------------
% Original vertex index corresponding to extended vertex (Vext)
%---------------------------------------------------
%--> Find nearest point from 'Vref' for 'V'
%[II, dd] = vb_find_nearest_point_no_overlap([V0L;V0R], Vext);
%BV_index.Vall  = II; 

BV_index.Left  = IndxL; 
BV_index.Right = IndxR + NdipoleL0;

% Cortex area index of original FreeSurfer surface
BV_index.cortexL = Vindex.cortexL;
BV_index.cortexR = Vindex.cortexR + NdipoleL0;

% Dimensional info
% number of all vertex in 'VLs'
NLall = size(VLs, 1);
NRall = size(VRs, 1);

Vinfo.Ndipole   = Ndipole;
Vinfo.NdipoleL  = NdipoleL;
Vinfo.Ndipole_extL  = Ndipole;
Vinfo.Ndipole_extR  = Ndipole + NLall;
Vinfo.Ndipole0  = Ndipole0;
Vinfo.NdipoleL0 = NdipoleL0;
Vinfo.Coord     = 'SPM_Right_m';
Vinfo.dd_match  = dd;

return

function Fnew = inner_patch_select(Vix,F)
% select patch whose vertices consist given vertex points

Nmax = max( max(Vix(:)) , max(F(:)) );

Itrans = zeros(Nmax,1);
Itrans(Vix) = 1:length(Vix);

Fnew = Itrans(F);

ix = find( Fnew(:,1).*Fnew(:,2).*Fnew(:,3) > 0);

Fnew = Fnew(ix,:);

return

function    [FL, VL, xxL] = check_surface(FL,VL)

EPS  = 1e-10;

[FL, VL, xxL] = vb_out_normal_surf(FL,VL);

%
% ---- Check closed surface by solid_angle/(2*pi) = 1
%
omega  = vb_solid_angle_check(VL,FL);
vb_disp(sprintf('solid_angle/(2*pi) = %5.3f',omega));
if abs(omega - 1) > EPS,
    vb_disp('Surface is not closed ');
end

return

function    [xx] = check_normal_vector(xx)
% Normalize normal vectors
xxn = zeros(size(xx));
nn = sum(xx.^2 ,2); 
ix = find(nn > 0);
xxn(ix,:) = xx(ix,:)./repmat(sqrt(nn(ix)) ,[1 3]);
xx = xxn;

% Check zero normal vector
if length(ix) == size(xx,1),
    vb_disp('--- All normal vector norms are 1');
else
    warning('There are zero normal vector');
end

return
